module CommandLineParser

open System

open FsOmegaLib.JSON

open Util
open SolverConfiguration

let VERSION = "1.1"

type Mode = 
    | Planning
    | ParityGame

type InputFormat = 
    | Explicit
    | NuSMV

type CommandLineArguments = 
    {
        InputFiles : list<string>
        Mode : Mode
        InputFormat : InputFormat

        DebugOutputs : bool
        ComputeBisimulation : bool
        AutomatonSemantics : ConstructPlanningInstance.AutomatonSemantics
    }

    static member Default = 
        {
            InputFiles = []
            Mode = Planning 
            InputFormat = NuSMV
            DebugOutputs = false
            ComputeBisimulation = true
            AutomatonSemantics = ConstructPlanningInstance.AutomatonSemantics.SAFE
        }

let rec private splitByPredicate (f : 'T -> bool) (xs : list<'T>) = 
    match xs with 
        | [] -> [], []
        | x::xs -> 
            if f x then 
                [], x::xs 
            else 
                let r1, r2 = splitByPredicate f xs 
                x::r1, r2

let parseCommandLineArguments (args : list<String>) =
    let rec parseArgumentsRec (args : list<String>) (opt : CommandLineArguments) = 

        match args with 
            | [] -> Result.Ok opt
            | x::xs -> 
                match x with 
                | "--exp" -> 
                    parseArgumentsRec xs {opt with InputFormat = Explicit}
                | "--nusmv" -> 
                    parseArgumentsRec xs {opt with InputFormat = NuSMV}
                | "--debug" -> 
                    parseArgumentsRec xs { opt with DebugOutputs = true}
                | "--no-bisim" -> 
                    parseArgumentsRec xs { opt with ComputeBisimulation = false}
                | "--safe" -> 
                    parseArgumentsRec xs { opt with AutomatonSemantics = ConstructPlanningInstance.AutomatonSemantics.SAFE}
                | "--reach" -> 
                    parseArgumentsRec xs { opt with AutomatonSemantics = ConstructPlanningInstance.AutomatonSemantics.REACH}
                | "--pg" -> 
                    parseArgumentsRec xs { opt with Mode = ParityGame}
                | "--planning" -> 
                    parseArgumentsRec xs { opt with Mode = Planning}
                | "--version" -> 
                    printfn "HyPlan (version 1.0)"
                    exit 0
                | s when s.StartsWith("-") -> 
                    Result.Error ("Option " + x + " is not supported" )
                | _ ->
                    let args, ys = splitByPredicate (fun (x : String) -> x.[0] = '-') args

                    if List.length args < 2 then 
                        Result.Error "Must give at least two input files"
                    else 
                        parseArgumentsRec ys {opt with InputFiles = args}
                    
        
    parseArgumentsRec args CommandLineArguments.Default
                                