module PlanningInstance 

type Type = string 
type Parameter = string 
type Predicate = string 
type Object = string
type Action = string

type PredicateDefinition = 
    {
        Name : Predicate
        Parameters : list<Parameter * Type>
    }


type BooleanFormula = 
    | Atom of Predicate * list<string> // Arguments are either parameters or objects
    | And of list<BooleanFormula>
    | Or of list<BooleanFormula>
    | Not of BooleanFormula
    | Imply of BooleanFormula * BooleanFormula
    | When of BooleanFormula * BooleanFormula
    | Oneof of list<BooleanFormula>

module BooleanFormula =
    let rec containsOneOf f = 
        match f with 
        | Atom _ -> false
        | And l | Or l -> List.exists containsOneOf l
        | Not f -> containsOneOf f
        | Imply (f1, f2) | When (f1, f2) -> containsOneOf f1 || containsOneOf f2
        | Oneof _ -> true

type ActionDefinition = 
    {
        Name : Action 
        Parameters : list<Parameter * Type>
        Precondition : BooleanFormula
        Effect: BooleanFormula
    }


type PlanningDomain = 
    {
        Name : string 
        Types : list<string>
        Constants : list<Object * Type>
        Predicates : list<PredicateDefinition>
        Actions : list<ActionDefinition>
    }


module PlanningDomain = 

    let rec printBooleanFormula (f : BooleanFormula) = 
        match f with 
        | Atom (p, l) ->
            let args = l |> Util.combineStringsWithSeparator " "
            $"(%s{p} %s{args})"
        | And l -> l |> List.map printBooleanFormula |> Util.combineStringsWithSeparator " " |> fun x -> $"(and %s{x})"
        | Or l -> l |> List.map printBooleanFormula |> Util.combineStringsWithSeparator " " |> fun x -> $"(or %s{x})"
        | Not f -> $"(not %s{printBooleanFormula f})"
        | Imply (f1, f2) -> $"(imply %s{printBooleanFormula f1} %s{printBooleanFormula f2})"
        | When (f1, f2) -> $"(when %s{printBooleanFormula f1} %s{printBooleanFormula f2})"
        | Oneof l -> l |> List.map printBooleanFormula |> Util.combineStringsWithSeparator " " |> fun x -> $"(oneof %s{x})"


    let print (dom : PlanningDomain) = 
        let printPredicateDefinition (pred : PredicateDefinition) = 
            "(" + 
            pred.Name + " " +
            (pred.Parameters
            |> List.map (fun (para, t) -> para + " - " + t)
            |> Util.combineStringsWithSeparator " ") +
            ")"

        
        let printActionDefinition (act : ActionDefinition) = 
            "(:action " + 
            act.Name + "\n" + 
            ":parameters (" + (act.Parameters |> List.map (fun (para, t) -> para + " - " + t) |> Util.combineStringsWithSeparator " ") + ")" + "\n" + 
            ":precondition " + printBooleanFormula act.Precondition + "\n" + 
            ":effect " + printBooleanFormula act.Effect + "\n" + 
            ")"

        let typesString = 
            dom.Types
            |> Util.combineStringsWithSeparator " "
            |> fun s -> $"(:types %s{s})"

        let constantsString = 
            dom.Constants
            |> List.map (fun (o, t) -> o + " - " + t)
            |> Util.combineStringsWithSeparator "\n"
            |> fun s -> $"(:constants %s{s})"

        let predicatesString = 
            dom.Predicates
            |> List.map printPredicateDefinition
            |> Util.combineStringsWithSeparator "\n"
            |> fun s -> $"(:predicates %s{s})"

        let actionsString =
            dom.Actions
            |> List.map printActionDefinition
            |> Util.combineStringsWithSeparator "\n\n"

        let isNonDeterminsitic = 
            dom.Actions
            |> List.exists (fun action -> 
                action.Effect
                |> BooleanFormula.containsOneOf
                )
        
        "(define" + "\n" +
        $"(domain %s{dom.Name})" + "\n" +
        (if isNonDeterminsitic then "(:requirements :typing :strips :non-deterministic)" else "(:requirements :typing :strips)") + "\n" + 
        typesString + "\n" + 
        constantsString + "\n" + 
        predicatesString + "\n" + 
        actionsString + "\n" + 
        ")"




type PlanningProblem = 
    {
        Name : string 
        Domain : string
        Objects : list<Object * Type>
        Init : list<Predicate * list<Object>>
        Goal : BooleanFormula
    }
    
module PlanningProblem = 

    let print (prob : PlanningProblem) = 
        let objectsString = 
            prob.Objects
            |> List.map (fun (o, t) -> o + " - " + t)
            |> Util.combineStringsWithSeparator "\n"
            |> fun s -> $"(:objects %s{s})"

        let initString = 
            prob.Init
            |> List.map (fun (o, args) -> 
                "(" + o + " " + (args |> Util.combineStringsWithSeparator " ") + ")"
            )
            |> Util.combineStringsWithSeparator "\n"
            |> fun s -> $"(:init %s{s})"

        let goalString = 
            prob.Goal
            |> PlanningDomain.printBooleanFormula
            |> fun s -> $"(:goal %s{s})"

        "(define" + "\n" +
        $"(problem %s{prob.Name})" + "\n" +
        $"(:domain %s{prob.Domain})" + "\n" +
        objectsString + "\n" + 
        initString + "\n" + 
        goalString + "\n" + 
        ")"