
module Program

open System 
open System.IO 

open TransitionSystemLib.TransitionSystem

open Util
open SolverConfiguration
open PlanningInstance
open ConstructPlanningInstance
open InstanceParsing
open CommandLineParser

let private run (args: array<string>) =

    let swtotal = System.Diagnostics.Stopwatch()
    let sw = System.Diagnostics.Stopwatch()
    swtotal.Start()

    // Parse the command line args
    let cmdArgs =
        match CommandLineParser.parseCommandLineArguments (Array.toList args) with
        | Result.Ok x -> x
        | Result.Error e ->
                raise <| HyPlanException $"%s{e}"

    let solverConfig = SolverConfiguration.getConfig()         
    let config = 
        {
            Configuration.SolverConfig = solverConfig
            ComputeBisimulation = cmdArgs.ComputeBisimulation
            Debug = false
            Logger = 
                {
                    Logger.Log =    
                        fun s -> 
                            if cmdArgs.DebugOutputs then 
                                printf $"%s{s}"
                }    
        }

    sw.Restart()

    let tsList, hyperltl = 
        match cmdArgs.InputFormat with 
        | Explicit -> 
            let files = cmdArgs.InputFiles
            let propertyFile = files[files.Length - 1]
            let systemFiles = files[0..files.Length - 2]

            let tsList, hyperltl = InstanceParsing.readAndParseExplicitInstance systemFiles propertyFile

            tsList, hyperltl
        
        | NuSMV -> 
            let files = cmdArgs.InputFiles
            let propertyFile = files[files.Length - 1]
            let systemFiles = files[0..files.Length - 2]


            let nusmvList, symbolicHyperltl = InstanceParsing.readAndParseSymbolicInstance systemFiles propertyFile

            let tsList, hyperltl = InstanceParsing.convertSymbolicSystemInstance nusmvList symbolicHyperltl

            tsList, hyperltl

    let tsList = 
        if config.ComputeBisimulation then     
            // Compute bisimulation quotient
            sw.Restart()
            let bisim = 
                tsList
                |> List.map (fun ts -> TransitionSystemLib.TransitionSystem.TransitionSystem.computeBisimulationQuotient ts |> fst)

            config.Logger.LogN $"Computed bisimulation quotient (%i{sw.ElapsedMilliseconds}ms)"
            config.Logger.LogN $"System sizes: %A{tsList |> List.map (fun ts -> ts.States.Count)}"

            bisim        
        else 
            tsList

    
    let tsList, hyperltl = 
        if tsList |> List.forall (fun ts -> ts.InitialStates |> Set.count = 1) then 
            tsList, hyperltl
        else 
            // We add a unique initial state
            let modTsList = 
                tsList
                |> List.map (fun ts -> 
                    let newState = (ts.States |> Set.maxElement) + 1

                    {
                        TransitionSystem.States = Set.add newState ts.States
                        InitialStates = Set.singleton newState
                        APs = ts.APs
                        Edges = 
                            ts.Edges
                            |> Map.add newState ts.InitialStates
                        ApEval = 
                            ts.ApEval
                            |> Map.add newState Set.empty
                    }
                
                    )

            let modHyperLTL = 
                { hyperltl with LTLMatrix = FsOmegaLib.LTL.X (hyperltl.LTLMatrix)}

            modTsList, modHyperLTL

    let sizes = tsList |> List.map (fun x -> x.States.Count)

    config.Logger.LogN $"Obtained explicit-state representation in %i{sw.ElapsedMilliseconds}ms (~=%.2f{double(sw.ElapsedMilliseconds) / 1000.0}s)"
    config.Logger.LogN $"Explicit-state size: %A{sizes}"
    sw.Restart()


    let tsMap = 
        if List.length tsList = 1 then 
            hyperltl.QuantifierPrefix
            |> List.map snd 
            |> List.map (fun x -> x, tsList.[0])
            |> Map.ofList
        else 
            if List.length tsList <> List.length hyperltl.QuantifierPrefix then 
                raise <| HyPlanException $"The number of systems and quantifiers must match"

            List.zip tsList hyperltl.QuantifierPrefix
            |> List.map (fun (ts, (_, pi)) -> pi, ts)
            |> Map.ofList


    match cmdArgs.Mode with 
    | ParityGame -> 
        sw.Restart()
        let pg = ConstructParityGame.constructParityGame config tsMap hyperltl

        
        config.Logger.LogN $"Created PG %i{sw.ElapsedMilliseconds}ms (~=%.2f{double(sw.ElapsedMilliseconds) / 1000.0}s)"
        sw.Restart()

        printfn $"|S|: %i{pg.Properties.Count}"
        

        let pgString = ConstructParityGame.ParityGame.convertParityGameToString pg
        let pgPath = "./game.pg"

        try
            File.WriteAllText (pgPath, pgString)
        with 
        | _ -> raise <| HyPlanException $"Could not write to %s{pgPath}"

    | Planning -> 
        let dom, prob = ConstructPlanningInstance.constructPlanningProblem config tsMap hyperltl cmdArgs.AutomatonSemantics

        printfn $"|Predicates|: %i{dom.Predicates.Length}"
        printfn $"|Actions|: %i{dom.Actions.Length}"
        printfn $"|Objects|: %i{prob.Objects.Length + dom.Constants.Length}"

        let domString = PlanningDomain.print dom
        let probString = PlanningProblem.print prob
        let domPath = "./dom.pddl"
        let probPath = "./prob.pddl"

        sw.Restart()

        try
            File.WriteAllText (domPath, domString)
        with 
        | _ -> raise <| HyPlanException $"Could not write to %s{domPath}"

        try
            File.WriteAllText (probPath, probString)
        with 
        | _ -> raise <| HyPlanException $"Could not write to %s{probPath}"

    swtotal.Stop()
    config.Logger.LogN $"\nTotal time: %i{swtotal.ElapsedMilliseconds}ms (~=%.2f{double(swtotal.ElapsedMilliseconds) / 1000.0}s)"

    0

[<EntryPoint>]
let main args =
    try 
        run args
    with 
    | HyPlanException err when true -> 
        printfn $"{err}"
        //reraise()
        -1
    | _  when true -> reraise()
    | HyPlanException err -> 
        printfn "Error during the analysis:"
        printfn "%s" err
        exit -1
    | e -> 
        printfn "Unexpected Error during the analysis:"
        printfn "%s" e.Message
        exit -1