module SolverConfiguration 

open System
open System.IO

open Util
open FsOmegaLib.JSON 


type SolverConfiguration = 
    {
        MainPath : string
        Ltl2tgbaPath: string
    }


let private parseConfigFile (s : string) =
    match FsOmegaLib.JSON.Parser.parseJsonString s with 
    | Result.Error err -> raise <| HyPlanException $"Could not parse config file: %s{err}"
    | Result.Ok x -> 
        {
            MainPath = "./" 
            Ltl2tgbaPath = 
                match JSON.tryLookup "ltl2tgba" x with 
                | Some (JString x) -> x
                | _ -> "Path to ltl2tgba is not defined"
        }

let getConfig() = 
    // By convention the paths.json file is located in the same directory as the HyPA executable
    let configPath = 
        System.IO.Path.Join [|System.IO.Path.GetDirectoryName(System.Reflection.Assembly.GetExecutingAssembly().Location); "paths.json"|]
                     
    // Check if the path to the config file is valid , i.e., the file exists
    if System.IO.FileInfo(configPath).Exists |> not then 
        raise <| HyPlanException "The paths.json file does not exist in the same directory as the executable"            
    
    // Parse the config File
    let configContent = 
        try
            File.ReadAllText configPath
        with 
            | _ -> 
                raise <| HyPlanException "Could not open paths.json file"

    let solverConfig = parseConfigFile configContent

    if System.IO.FileInfo(solverConfig.Ltl2tgbaPath).Exists |> not then 
        raise <| HyPlanException "The given path to the spot's ltl2tgba is incorrect"

    solverConfig


type Logger = 
    {
        Log : string -> unit
    }

    member this.LogN s = this.Log (s + "\n") 

type Configuration = 
    {
        SolverConfig : SolverConfiguration
        ComputeBisimulation : bool
        Debug : bool
        Logger : Logger
    }