import io
import os
import csv
import copy
import json
import base64
from collections.abc import Sequence
from types import FunctionType
from typing import Any, Callable, cast, Dict, List, Optional, Tuple, Union

import cv2
import numpy as np
import torch
import torch.utils.data as data
from PIL import Image
from torch.utils.data import Dataset
from transformers import AutoTokenizer


class ImageReadFromTSV(object):
    def __init__(self,
                 image_tsv,
                 input_key: str = 'img_path',
                 output_key: str = 'img',
                 channel_order: str = 'bgr',
                 meet_corrupt: str = 'raise_error'):
        assert channel_order in ('bgr', 'rgb')
        self.channel_order = channel_order
        self.input_key = input_key
        self.output_key = output_key
        # 如果遇到了损坏的图片，或者图片不存在，应该怎么做：
        # - raise_error：直接报错抛出异常
        # - dummy_image：生成一张空的图片
        assert meet_corrupt in ('raise_error', 'dummy_image')
        self.meet_corrupt = meet_corrupt

        # To tackle '_csv.Error: ImageReadFromTSV: field larger than field limit (131072)'
        import sys
        csv.field_size_limit(sys.maxsize)

        self.image_tsv = {}
        with open(image_tsv, "r") as f:
            for row in csv.reader(f, delimiter="\t"):
                self.image_tsv[str(row[0])] = row[1]

    def __call__(self, results):
        try:
            img_name = results[self.input_key]
            encoded_data = self.image_tsv[str(img_name)]
            image_data = base64.b64decode(encoded_data)
            img = Image.open(io.BytesIO(image_data))

            assert img is not None
            results['imread_succ'] = 1
        except AssertionError:
            if self.meet_corrupt == 'raise_error':
                raise IOError("Fail to load {}".format(results[self.input_key]))
            else:
                width, height = 224, 224  # 尺寸可以根据需要修改
                color = (255, 255, 255)  # 白色
                img = Image.new('RGB', (width, height), color)
                results['imread_succ'] = 0
        # if self.channel_order == 'rgb':
        #     img = np.ascontiguousarray(img[..., ::-1])
        if img is not None:
            results[self.output_key] = img
        return results


class ResizeWithRatioConstraint(object):

    def __init__(self, out_size, max_ratio=2.5):
        self.max_ratio = max_ratio
        if isinstance(out_size, int):
            self.out_size = (out_size, out_size)
        else:
            self.out_size = out_size

    def __call__(self, results: dict) -> dict:
        img = results['img']
        h, w = img.shape[0:2]
        if h <= w * self.max_ratio and w <= h * self.max_ratio:
            # 如果图片长宽比在 max_ratio 规定的范围之内，那么直接 resize
            img = cv2.resize(img, self.out_size, interpolation=cv2.INTER_CUBIC)
        else:
            # 如果图片长宽比太大，则需要先裁剪，再 resize
            if h > w:
                crop_h = min(int(round(w * self.max_ratio)), h)
                crop_w = w
            else:
                crop_h = h
                crop_w = min(int(round(h * self.max_ratio)), w)

            y1 = (h - crop_h) // 2
            x1 = (w - crop_w) // 2
            y2 = y1 + crop_h
            x2 = x1 + crop_w
            cropped_img = img[y1:y2, x1:x2]
            img = cv2.resize(cropped_img, self.out_size, interpolation=cv2.INTER_CUBIC)
        results['img'] = img
        return results


class NormalizeNumpyImage(object):

    def __init__(self, mean, std, shuffle_bgr=False, normalize_scale=False):
        self.mean = np.array(mean, dtype=np.float32)
        self.std = np.array(std, dtype=np.float32)
        self.shuffle_bgr = shuffle_bgr
        self.normalize_scale = normalize_scale

    def __call__(self, results: dict) -> dict:
        img = results['img']
        if img.dtype == np.uint8:
            img = img.astype(np.float32)
        img = img / 255.0 if self.normalize_scale else img

        mean = self.mean.reshape(1, -1)
        stdinv = 1 / self.std.reshape(1, -1)
        if self.shuffle_bgr:
            cv2.cvtColor(img, cv2.COLOR_BGR2RGB, img)  # inplace
        img = (img - mean) * stdinv
        results['img'] = img
        results['img_norm_cfg'] = dict(mean=self.mean,
                                       std=self.std,
                                       shuffle_bgr=self.shuffle_bgr,
                                       normalize_scale=self.normalize_scale)
        return results


class TextDataset(Dataset):
    def __init__(self, ann_file, tokenizer):
        self.ann_file = ann_file
        self.anns = self.load_annotations()

        self.tokenizer = tokenizer

    def load_annotations(self):
        assert os.path.isfile(self.ann_file), f"Cannot read {self.ann_file}"
        anns = []
        with open(self.ann_file, 'r', encoding='utf8') as f:
            # read file line by line
            for line in f:
                anns.append(json.loads(line.strip()))
        return anns

    def __len__(self):
        return len(self.anns)

    def __getitem__(self, item):
        data = copy.deepcopy(self.anns[item])
        text = data['text']

        # ret = self.tokenizer(text,
        #                      padding='max_length',
        #                      truncation=True,
        #                      max_length=72,
        #                      return_tensors="pt")
        ret = self.tokenizer(text)

        results = {}
        results['text'] = ret.squeeze(0)
        # results['text_mask'] = ret['attention_mask'].squeeze(0)

        return results


class ImgTSVDataset(Dataset):
    def __init__(self, ann_file, tsv_path, preprocess):
        self.ann_file = ann_file
        self.anns = self.load_annotations()
        self.preprocess = preprocess
        self.read_op = ImageReadFromTSV(
            image_tsv=tsv_path,
            input_key='img_name',
            channel_order='rgb',
            meet_corrupt='raise_error'
        )

    def load_annotations(self):
        assert os.path.isfile(self.ann_file), f"Cannot read {self.ann_file}"
        anns = []
        with open(self.ann_file, 'r', encoding='utf8') as f:
            # read file line by line
            for line in f:
                anns.append(json.loads(line.strip()))
        return anns

    def __len__(self):
        return len(self.anns)

    def __getitem__(self, item):
        data = copy.deepcopy(self.anns[item])
        data = self.read_op(data)
        img = data['img']
        ret = self.preprocess(img)
        return {'image': ret}


def _log_api_usage_once(obj: Any) -> None:
    module = obj.__module__
    if not module.startswith("torchvision"):
        module = f"torchvision.internal.{module}"
    name = obj.__class__.__name__
    if isinstance(obj, FunctionType):
        name = obj.__name__
    torch._C._log_api_usage_once(f"{module}.{name}")


class VisionDataset(data.Dataset):
    _repr_indent = 4

    def __init__(
            self,
            root: str,
            transforms: Optional[Callable] = None,
            transform: Optional[Callable] = None,
            target_transform: Optional[Callable] = None,
    ) -> None:
        _log_api_usage_once(self)
        if isinstance(root, str):
            root = os.path.expanduser(root)
        self.root = root

        has_transforms = transforms is not None
        has_separate_transform = transform is not None or target_transform is not None
        if has_transforms and has_separate_transform:
            raise ValueError("Only transforms or transform/target_transform can be passed as argument")

        # for backwards-compatibility
        self.transform = transform
        self.target_transform = target_transform

        if has_separate_transform:
            transforms = StandardTransform(transform, target_transform)
        self.transforms = transforms

    def __getitem__(self, index: int) -> Any:
        """
        Args:
            index (int): Index

        Returns:
            (Any): Sample and meta data, optionally transformed by the respective transforms.
        """
        raise NotImplementedError

    def __len__(self) -> int:
        raise NotImplementedError

    def __repr__(self) -> str:
        head = "Dataset " + self.__class__.__name__
        body = [f"Number of datapoints: {self.__len__()}"]
        if self.root is not None:
            body.append(f"Root location: {self.root}")
        body += self.extra_repr().splitlines()
        if hasattr(self, "transforms") and self.transforms is not None:
            body += [repr(self.transforms)]
        lines = [head] + [" " * self._repr_indent + line for line in body]
        return "\n".join(lines)

    def _format_transform_repr(self, transform: Callable, head: str) -> List[str]:
        lines = transform.__repr__().splitlines()
        return [f"{head}{lines[0]}"] + ["{}{}".format(" " * len(head), line) for line in lines[1:]]

    def extra_repr(self) -> str:
        return ""


class StandardTransform:
    def __init__(self, transform: Optional[Callable] = None, target_transform: Optional[Callable] = None) -> None:
        self.transform = transform
        self.target_transform = target_transform

    def __call__(self, input: Any, target: Any) -> Tuple[Any, Any]:
        if self.transform is not None:
            input = self.transform(input)
        if self.target_transform is not None:
            target = self.target_transform(target)
        return input, target

    def _format_transform_repr(self, transform: Callable, head: str) -> List[str]:
        lines = transform.__repr__().splitlines()
        return [f"{head}{lines[0]}"] + ["{}{}".format(" " * len(head), line) for line in lines[1:]]

    def __repr__(self) -> str:
        body = [self.__class__.__name__]
        if self.transform is not None:
            body += self._format_transform_repr(self.transform, "Transform: ")
        if self.target_transform is not None:
            body += self._format_transform_repr(self.target_transform, "Target transform: ")

        return "\n".join(body)


def has_file_allowed_extension(filename: str, extensions: Union[str, Tuple[str, ...]]) -> bool:
    """Checks if a file is an allowed extension.

    Args:
        filename (string): path to a file
        extensions (tuple of strings): extensions to consider (lowercase)

    Returns:
        bool: True if the filename ends with one of given extensions
    """
    return filename.lower().endswith(extensions if isinstance(extensions, str) else tuple(extensions))


def is_image_file(filename: str) -> bool:
    """Checks if a file is an allowed image extension.

    Args:
        filename (string): path to a file

    Returns:
        bool: True if the filename ends with a known image extension
    """
    return has_file_allowed_extension(filename, IMG_EXTENSIONS)


def find_classes(directory: str) -> Tuple[List[str], Dict[str, int]]:
    """Finds the class folders in a dataset.

    See :class:`DatasetFolder` for details.
    """
    classes = sorted(entry.name for entry in os.scandir(directory) if entry.is_dir())
    if not classes:
        raise FileNotFoundError(f"Couldn't find any class folder in {directory}.")

    class_to_idx = {cls_name: i for i, cls_name in enumerate(classes)}
    return classes, class_to_idx


def make_dataset(
        directory: str,
        class_to_idx: Optional[Dict[str, int]] = None,
        extensions: Optional[Union[str, Tuple[str, ...]]] = None,
        is_valid_file: Optional[Callable[[str], bool]] = None,
) -> List[Tuple[str, int]]:
    """Generates a list of samples of a form (path_to_sample, class).

    See :class:`DatasetFolder` for details.

    Note: The class_to_idx parameter is here optional and will use the logic of the ``find_classes`` function
    by default.
    """
    directory = os.path.expanduser(directory)

    if class_to_idx is None:
        _, class_to_idx = find_classes(directory)
    elif not class_to_idx:
        raise ValueError("'class_to_index' must have at least one entry to collect any samples.")

    both_none = extensions is None and is_valid_file is None
    both_something = extensions is not None and is_valid_file is not None
    if both_none or both_something:
        raise ValueError("Both extensions and is_valid_file cannot be None or not None at the same time")

    if extensions is not None:
        def is_valid_file(x: str) -> bool:
            return has_file_allowed_extension(x, extensions)  # type: ignore[arg-type]

    is_valid_file = cast(Callable[[str], bool], is_valid_file)

    instances = []
    available_classes = set()
    for target_class in sorted(class_to_idx.keys()):
        class_index = class_to_idx[target_class]
        target_dir = os.path.join(directory, target_class)
        if not os.path.isdir(target_dir):
            continue
        for root, _, fnames in sorted(os.walk(target_dir, followlinks=True)):
            for fname in sorted(fnames):
                path = os.path.join(root, fname)
                if is_valid_file(path):
                    item = path, class_index
                    instances.append(item)

                    if target_class not in available_classes:
                        available_classes.add(target_class)

    empty_classes = set(class_to_idx.keys()) - available_classes
    if empty_classes:
        msg = f"Found no valid file for the classes {', '.join(sorted(empty_classes))}. "
        if extensions is not None:
            msg += f"Supported extensions are: {extensions if isinstance(extensions, str) else ', '.join(extensions)}"
        raise FileNotFoundError(msg)

    return instances


class DatasetFolder(VisionDataset):
    """A generic data loader.

    This default directory structure can be customized by overriding the
    :meth:`find_classes` method.

    Args:
        root (string): Root directory path.
        loader (callable): A function to load a sample given its path.
        extensions (tuple[string]): A list of allowed extensions.
            both extensions and is_valid_file should not be passed.
        transform (callable, optional): A function/transform that takes in
            a sample and returns a transformed version.
            E.g, ``transforms.RandomCrop`` for images.
        target_transform (callable, optional): A function/transform that takes
            in the target and transforms it.
        is_valid_file (callable, optional): A function that takes path of a file
            and check if the file is a valid file (used to check of corrupt files)
            both extensions and is_valid_file should not be passed.

     Attributes:
        classes (list): List of the class names sorted alphabetically.
        class_to_idx (dict): Dict with items (class_name, class_index).
        samples (list): List of (sample path, class_index) tuples
        targets (list): The class_index value for each image in the dataset
    """

    def __init__(
            self,
            root: str,
            loader: Callable[[str], Any],
            extensions: Optional[Tuple[str, ...]] = None,
            transform: Optional[Callable] = None,
            target_transform: Optional[Callable] = None,
            is_valid_file: Optional[Callable[[str], bool]] = None,
    ) -> None:
        super().__init__(root, transform=transform, target_transform=target_transform)
        classes, class_to_idx = self.find_classes(self.root)
        samples = self.make_dataset(self.root, class_to_idx, extensions, is_valid_file)

        self.loader = loader
        self.extensions = extensions

        self.classes = classes
        self.class_to_idx = class_to_idx
        self.samples = samples
        self.targets = [s[1] for s in samples]

    @staticmethod
    def make_dataset(
            directory: str,
            class_to_idx: Dict[str, int],
            extensions: Optional[Tuple[str, ...]] = None,
            is_valid_file: Optional[Callable[[str], bool]] = None,
    ) -> List[Tuple[str, int]]:
        """Generates a list of samples of a form (path_to_sample, class).

        This can be overridden to e.g. read files from a compressed zip file instead of from the disk.

        Args:
            directory (str): root dataset directory, corresponding to ``self.root``.
            class_to_idx (Dict[str, int]): Dictionary mapping class name to class index.
            extensions (optional): A list of allowed extensions.
                Either extensions or is_valid_file should be passed. Defaults to None.
            is_valid_file (optional): A function that takes path of a file
                and checks if the file is a valid file
                (used to check of corrupt files) both extensions and
                is_valid_file should not be passed. Defaults to None.

        Raises:
            ValueError: In case ``class_to_idx`` is empty.
            ValueError: In case ``extensions`` and ``is_valid_file`` are None or both are not None.
            FileNotFoundError: In case no valid file was found for any class.

        Returns:
            List[Tuple[str, int]]: samples of a form (path_to_sample, class)
        """
        if class_to_idx is None:
            # prevent potential bug since make_dataset() would use the class_to_idx logic of the
            # find_classes() function, instead of using that of the find_classes() method, which
            # is potentially overridden and thus could have a different logic.
            raise ValueError("The class_to_idx parameter cannot be None.")
        return make_dataset(directory, class_to_idx, extensions=extensions, is_valid_file=is_valid_file)

    def find_classes(self, directory: str) -> Tuple[List[str], Dict[str, int]]:
        """Find the class folders in a dataset structured as follows::

            directory/
            ├── class_x
            │   ├── xxx.ext
            │   ├── xxy.ext
            │   └── ...
            │       └── xxz.ext
            └── class_y
                ├── 123.ext
                ├── nsdf3.ext
                └── ...
                └── asd932_.ext

        This method can be overridden to only consider
        a subset of classes, or to adapt to a different dataset directory structure.

        Args:
            directory(str): Root directory path, corresponding to ``self.root``

        Raises:
            FileNotFoundError: If ``dir`` has no class folders.

        Returns:
            (Tuple[List[str], Dict[str, int]]): List of all classes and dictionary mapping each class to an index.
        """
        return find_classes(directory)

    def __getitem__(self, index: int) -> Tuple[Any, Any]:
        """
        Args:
            index (int): Index

        Returns:
            tuple: (sample, target) where target is class_index of the target class.
        """
        path, target = self.samples[index]
        sample = self.loader(path)
        if self.transform is not None:
            sample = self.transform(sample)
        if self.target_transform is not None:
            target = self.target_transform(target)

        return sample, target

    def __len__(self) -> int:
        return len(self.samples)


IMG_EXTENSIONS = (".jpg", ".jpeg", ".png", ".ppm", ".bmp", ".pgm", ".tif", ".tiff", ".webp")


def pil_loader(path: str) -> Image.Image:
    # open path as file to avoid ResourceWarning (https://github.com/python-pillow/Pillow/issues/835)
    with open(path, "rb") as f:
        img = Image.open(f)
        return img.convert("RGB")


# TODO: specify the return type
def accimage_loader(path: str) -> Any:
    import accimage

    try:
        return accimage.Image(path)
    except OSError:
        # Potentially a decoding problem, fall back to PIL.Image
        return pil_loader(path)


def default_loader(path: str) -> Any:
    from torchvision import get_image_backend

    if get_image_backend() == "accimage":
        return accimage_loader(path)
    else:
        return pil_loader(path)


class ImageFolder(DatasetFolder):
    """A generic data loader where the images are arranged in this way by default: ::

        root/dog/xxx.png
        root/dog/xxy.png
        root/dog/[...]/xxz.png

        root/cat/123.png
        root/cat/nsdf3.png
        root/cat/[...]/asd932_.png

    This class inherits from :class:`~torchvision.datasets.DatasetFolder` so
    the same methods can be overridden to customize the dataset.

    Args:
        root (string): Root directory path.
        transform (callable, optional): A function/transform that  takes in an PIL image
            and returns a transformed version. E.g, ``transforms.RandomCrop``
        target_transform (callable, optional): A function/transform that takes in the
            target and transforms it.
        loader (callable, optional): A function to load an image given its path.
        is_valid_file (callable, optional): A function that takes path of an Image file
            and check if the file is a valid file (used to check of corrupt files)

    """

    def __init__(
            self,
            root: str,
            transform: Optional[Callable] = None,
            target_transform: Optional[Callable] = None,
            loader: Callable[[str], Any] = default_loader,
            is_valid_file: Optional[Callable[[str], bool]] = None,
    ):
        super().__init__(
            root,
            loader,
            IMG_EXTENSIONS if is_valid_file is None else None,
            transform=transform,
            target_transform=target_transform,
            is_valid_file=is_valid_file,
        )
        self.imgs = self.samples


class ImgFolderDataset(ImageFolder):
    def __init__(self, root, preprocess, **kwargs):
        super().__init__(root, **kwargs)
        self.kwargs = kwargs
        self.preprocess = preprocess

    def __getitem__(self, item):
        path, target = self.imgs[item]
        image = self.preprocess(Image.open(path))

        return {'image': image, 'target': target}
