#!/usr/bin/env python
# Copyright (c) Facebook, Inc. and its affiliates. All Rights Reserved
import argparse
import builtins
import math
import os
import random
import shutil
import time
import warnings

import torch
import torch.nn as nn
import torch.nn.parallel
import torch.backends.cudnn as cudnn
import torch.distributed as dist
import torch.optim
import torch.multiprocessing as mp
import torch.utils.data
import torch.utils.data.distributed
import torchvision.transforms as transforms
import torchvision.datasets as datasets
import torchvision.models as models

import models.loader
import models.builder
from utils import setup_logger, get_rank, copy_script_file, AverageMeter, ProgressMeter
import pdb

model_names = sorted(name for name in models.__dict__
    if name.islower() and not name.startswith("__")
    and callable(models.__dict__[name]))

parser = argparse.ArgumentParser(description='PyTorch ImageNet Training')

## dataset
parser.add_argument('--dataset', default='Imagenet', type=str, help='dataset')
parser.add_argument('--data', default="/export/share/datasets/vision/imagenet/", type=str, help='path to dataset')


## archtechture
parser.add_argument('--arch', metavar='ARCH', default='resnet50', choices=model_names, help='model architecture: ' +' | '.join(model_names) + ' (default: resnet50)')
parser.add_argument('--moco-dim', default=128, type=int, help='feature dimension (default: 128)')

## distribuion
parser.add_argument('--world-size', default=1, type=int, help='number of nodes for distributed training')
parser.add_argument('--rank', default=0, type=int, help='node rank for distributed training')
parser.add_argument('--dist-url', default='tcp://localhost:10001', type=str, help='url used to set up distributed training')
parser.add_argument('--dist-backend', default='nccl', type=str, help='distributed backend')
parser.add_argument('--multiprocessing-distributed', default=True, type=bool, help='Use multi-processing distributed training to launch N processes per node, which has N GPUs. This is the '
                         'fastest way to use PyTorch for either single node or '
                         'multi node data parallel training')

## optimizer
parser.add_argument('--workers', default=96, type=int, metavar='N', help='number of data loading workers (default: 32)')
parser.add_argument('--lr', '--learning-rate', default=0.03, type=float, metavar='LR', help='initial learning rate', dest='lr')
parser.add_argument('--schedule', default=[120, 160], nargs='*', type=int, help='learning rate schedule (when to drop lr by 10x)')
parser.add_argument('--momentum', default=0.9, type=float, metavar='M', help='momentum of SGD solver')
parser.add_argument('--wd', '--weight-decay', default=1e-4, type=float, metavar='W', help='weight decay (default: 1e-4)', dest='weight_decay')
parser.add_argument('--cos', default=True, type=bool, help='use cosine lr schedule')


parser.add_argument('--epochs', default=200, type=int, metavar='N', help='number of total epochs to run')
parser.add_argument('--start-epoch', default=0, type=int, metavar='N', help='manual epoch number (useful on restarts)')
parser.add_argument('--batch-size', default=512, type=int, metavar='N', help='2880 mini-batch size (default: 256), this is the total ' 'batch size of all GPUs on the current node when ' 'using Data Parallel or Distributed Data Parallel')
parser.add_argument('--moco-k', default=65536, type=int, help='65520 65450 65520(26, 208)  65529(27, 216) 65536(32, 256) 65556(216) queue size; number of negative keys (default: 65536)')

parser.add_argument('--scale_crop', default=False, type=bool, help='whether scale the augmentations')
parser.add_argument('--strong_crop_num', default=0, type=int, help='how many strong augmentations are used in the second loss')
parser.add_argument('--weak_crop_num', default=1, type=int, help='how many weak augmentations are used in the second loss')
parser.add_argument('--crop_size', default=96, type=int, help='the size of augmentations in the second loss')

parser.add_argument('--sys', default=True, type=bool, metavar='N', help='whether the loss is sysmetric')
parser.add_argument('--loss-type', default='CLEAN', type=str, metavar='PATH', help='CLEAN,moco')
parser.add_argument('--teacher_weight', default=0.5, type=float, metavar='PATH', help='contrastive, CLSA, PC')

parser.add_argument('--warmup_epoch', default=0, type=int, metavar='PATH', help='contrastive, CLSA, PC')
parser.add_argument('--warmup_lr', default=0.001, type=int, metavar='PATH', help='contrastive, CLSA, PC')


parser.add_argument('--seed', default='',  help='seed for initializing training. ')
parser.add_argument('--gpu', default='0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15', type=str, help='GPU id to use.')
parser.add_argument('--resume', default='model_last.pth', type=str, metavar='PATH', help='model_last.pth path to latest checkpoint (default: none)')

parser.add_argument('--print-freq', default=1000, type=int, metavar='N', help='print frequency (default: 10)')
parser.add_argument('--save_path', default='ImageNet200_256_www_labelrefine/',  type=str,  help='path to dataset')


parser.add_argument('--moco-m', default=0.999, type=float, help='moco momentum of updating key encoder (default: 0.999)')
parser.add_argument('--moco-t', default=0.15, type=float, help='softmax temperature (default: 0.07)')
parser.add_argument('--teach_t', default=0.15, type=float, help='how many augmentations are used')


#CUDA_VISIBLE_DEVICES=3 python main_moco_weak.py
parser.add_argument('--sharp_probability', default=1.0, type=float, metavar='PATH', help='sharp constant for sharp prediction probability')
parser.add_argument('--lam', default=0.1, type=float, metavar='PATH', help='beta distribution parameter')
parser.add_argument('--prior_confidence', default='cos', type=str, metavar='PATH', help='contrastive, CLSA, PC')
parser.add_argument('--prior_confidence_max', default=1.0, type=float, metavar='PATH', help='contrastive, CLSA, PC')
parser.add_argument('--prior_confidence_min', default=1.0, type=float, metavar='PATH', help='contrastive, CLSA, PC')


def main():
    args = parser.parse_args()

    args.lr = args.batch_size / 256.0 * args.lr

    if args.seed == '':
        args.seed = random.randint(0, 10000)

    if args.seed is not None:
        random.seed(args.seed)
        torch.manual_seed(args.seed)
        cudnn.deterministic = True
        warnings.warn('You have chosen to seed training. '
                      'This will turn on the CUDNN deterministic setting, '
                      'which can slow down your training considerably! '
                      'You may see unexpected behavior when restarting '
                      'from checkpoints.')

    os.environ["CUDA_VISIBLE_DEVICES"] = args.gpu
    if args.gpu is not None:
        warnings.warn('You have chosen a specific GPU. This will completely '
                      'disable data parallelism.')


    name = 'www' if args.strong_crop_num==0 else 'wws'
    args.save_path = './%s/%s_%s%d_sys%d_warm%d_ep%d_cs%d_sharp%.2f_weight%.2f_lam%.1f_temp%.2f_%.2f_weak%d_strong%d_batch%d_prior%s_%.2f_%.2f_dic%d_mom%.3f' % ( \
        args.save_path, args.loss_type, name, args.crop_size, args.sys, args.warmup_epoch, args.epochs, args.scale_crop, args.sharp_probability,\
        args.teacher_weight, args.lam, args.moco_t, args.teach_t,\
        args.weak_crop_num, args.strong_crop_num, args.batch_size, args.prior_confidence, args.prior_confidence_min,\
        args.prior_confidence_max, args.moco_k, args.moco_m)
    print(args.save_path)
    # pdb.set_trace()
    files_to_same = ['main.py', 'main_lincls.py', 'utils.py', 'models/builder.py', 'models/augmentations.py',
                     'models/loader.py' ]
    copy_script_file(args.save_path, files_to_same, name='train')

    if args.dist_url == "env://" and args.world_size == -1:
        args.world_size = int(os.environ["WORLD_SIZE"])

    args.distributed = args.world_size > 1 or args.multiprocessing_distributed

    ngpus_per_node = torch.cuda.device_count()
    if args.multiprocessing_distributed:
        # Since we have ngpus_per_node processes per node, the total world_size
        # needs to be adjusted accordingly
        args.world_size = ngpus_per_node * args.world_size
        # Use torch.multiprocessing.spawn to launch distributed processes: the
        # main_worker process function
        mp.spawn(main_worker, nprocs=ngpus_per_node, args=(ngpus_per_node, args))
    else:
        # Simply call main_worker function
        main_worker(args.gpu, ngpus_per_node, args)


def main_worker(gpu, ngpus_per_node, args):
    args.gpu = gpu

    # suppress printing if not master
    if args.multiprocessing_distributed and args.gpu != 0:
        def print_pass(*args):
            pass
        builtins.print = print_pass

    if args.distributed:
        if args.gpu is not None:
            args.batch_size = int(args.batch_size / ngpus_per_node)
        if args.dist_url == "env://" and args.rank == -1:
            args.rank = int(os.environ["RANK"])
        if args.multiprocessing_distributed:
            # For multiprocessing distributed training, rank needs to be the
            # global rank among all the processes
            args.rank = args.rank * ngpus_per_node + gpu
        dist.init_process_group(backend=args.dist_backend, init_method=args.dist_url,
                                world_size=args.world_size, rank=args.rank)
    # create model
    master_flag = not args.multiprocessing_distributed or (
                args.multiprocessing_distributed and args.rank % ngpus_per_node == 0)
    logger = setup_logger("Training", args.save_path)

    model = models.builder.CLEAN(models.__dict__[args.arch], args, master_flag=master_flag, logger=logger)

    if master_flag:
        logger.info("=> model is constructed as '{}'".format(args.arch))
        if args.gpu is not None:
            logger.info("Use GPU: {} for training".format(args.gpu))


    if args.distributed:
        # For multiprocessing distributed, DistributedDataParallel constructor
        # should always set the single device scope, otherwise,
        # DistributedDataParallel will use all available devices.
        if args.gpu is not None:
            torch.cuda.set_device(args.gpu)
            model.cuda(args.gpu)
            # When using a single GPU per process and per
            # DistributedDataParallel, we need to divide the batch size
            # ourselves based on the total number of GPUs we have
            # args.batch_size = int(args.batch_size / ngpus_per_node)
            args.workers = int((args.workers + ngpus_per_node - 1) / ngpus_per_node)
            model = torch.nn.parallel.DistributedDataParallel(model, device_ids=[args.gpu])
        else:
            model.cuda()
            # DistributedDataParallel will divide and allocate batch_size to all
            # available GPUs if device_ids are not set
            model = torch.nn.parallel.DistributedDataParallel(model)
    elif args.gpu is not None:
        torch.cuda.set_device(args.gpu)
        model = model.cuda(args.gpu)
        # comment out the following line for debugging
        raise NotImplementedError("Only DistributedDataParallel is supported.")
    else:
        # AllGather implementation (batch shuffle, queue update, etc.) in
        # this code only supports DistributedDataParallel.
        raise NotImplementedError("Only DistributedDataParallel is supported.")

    # define loss function (criterion) and optimizer
    optimizer = torch.optim.SGD(model.parameters(), args.lr, momentum=args.momentum, weight_decay=args.weight_decay)


    if args.resume is not None:
        resume_path = '%s/%s'%(args.save_path, args.resume)
        if os.path.isfile(resume_path):
            if master_flag:
                logger.info("=> loading checkpoint '{}'".format(resume_path))
            if args.gpu is None:
                checkpoint = torch.load(resume_path)
            else:
                # Map model to be loaded to specified single gpu.
                loc = 'cuda:{}'.format(args.gpu)
                checkpoint = torch.load(resume_path, map_location=loc)
            args.start_epoch = checkpoint['epoch']
            msg = model.load_state_dict(checkpoint['state_dict'])
            if master_flag:
                logger.info(msg.missing_keys)
                logger.info(msg.unexpected_keys)

            optimizer.load_state_dict(checkpoint['optimizer'])
            if master_flag:
                logger.info("=> loaded checkpoint '{}' (epoch {})".format(resume_path, checkpoint['epoch']))
        else:
            if master_flag:
                logger.info("=> no checkpoint found at '{}'".format(resume_path))

    cudnn.benchmark = True

    # Data loading code
    if args.loss_type == "moco":
        augmentation_w, augmentation_s = models.loader.get_augmentation_weak1_strong1(sizes_w=224, sizes_s=args.crop_size,
                                                                                                        scale_flag=args.scale_crop, strong_aug=args.strong_crop_num>1e-8)

        transforms_func = models.loader.ThreeCropsTransform_moco(augmentation_w, augmentation_s)
    else:
        if args.weak_crop_num == 1 or args.strong_crop_num == 1:
            augmentation_w, augmentation_s = models.loader.get_augmentation_weak1_strong1(sizes_w=224, sizes_s=args.crop_size,
                                                                                                            scale_flag=args.scale_crop, strong_aug=args.strong_crop_num>1e-8)
            transforms_func = models.loader.ThreeCropsTransform(augmentation_w, augmentation_s)
        elif args.weak_crop_num > 1 or args.strong_crop_num > 1:
            augmentation_w, augmentation_s = models.loader.get_augmentation(sizes_w=[224,192,160,128,96], sizes_s=[224,192,160,128,96], strong_aug=args.strong_crop_num>1e-8)
            transforms_func = models.loader.MultiCropsTransform(augmentation_w, augmentation_s, \
                          args.strong_crop_num, args.batch_size)
        else:
            logger.info('ERROR of the Augmentation')
            assert (args.weak_crop_num >= 1 or args.strong_crop_num >= 1)

    train_dataset = datasets.ImageFolder(os.path.join(args.data, 'train'), transforms_func)
    if master_flag: logger.info('training data size: %d' % len(train_dataset.samples))


    if args.distributed:
        train_sampler = torch.utils.data.distributed.DistributedSampler(train_dataset)
    else:
        train_sampler = None

    train_loader = torch.utils.data.DataLoader(train_dataset, batch_size=args.batch_size, shuffle=(train_sampler is None),
        num_workers=args.workers, pin_memory=True, sampler=train_sampler, drop_last=True)


    for epoch in range(args.start_epoch, args.epochs+1):
        t1 = time.time()
        if args.distributed:
            train_sampler.set_epoch(epoch)
        adjust_learning_rate(optimizer, epoch, args)


        # train for one epoch
        train_loss, test_acc, top1, top2, top3, confidence_score1, confidence_score2 = train(train_loader, model, optimizer, epoch, args)
        if master_flag:
            t2 = time.time()
            remaining_time = (args.epochs - epoch) * (t2 - t1) / 3600.0

            lr = optimizer.param_groups[0]['lr']
            results_str = '%d ep, lr %.5f trloss %.6f, test acc %.4f, ac %.4f, ac2 %.4f, ac3 %.4f, c1 %.4f, c2 %.4f, time %.2f h' % (
                epoch, lr, train_loss, test_acc, top1, top2, top3, confidence_score1, confidence_score2, remaining_time)
            logger.info(results_str)
            torch.save(
                {'epoch': epoch, 'testAcc': test_acc, 'arch': args.arch, 'state_dict': model.state_dict(),
                 'optimizer': optimizer.state_dict()}, args.save_path + '/model_last.pth')
            if epoch % 100 == 0:
                torch.save({'epoch': epoch, 'testAcc': test_acc, 'arch': args.arch, 'state_dict': model.module.encoder_q.state_dict()}, args.save_path + '/model_last_q.pth')
                torch.save({'epoch': epoch, 'testAcc': test_acc, 'arch': args.arch, 'state_dict': model.module.encoder_k.state_dict()}, args.save_path + '/model_last_k.pth')

            if epoch % 100 == 0:
                torch.save(
                    {'epoch': epoch, 'testAcc': test_acc, 'arch': args.arch, 'state_dict': model.state_dict(),
                     'optimizer': optimizer.state_dict()}, args.save_path + '/models_%d.pth' % epoch)


def adjust_learning_rate(optimizer, epoch, args):
    """Decay the learning rate based on schedule"""
    if args.warmup_epoch > epoch:
        lr = args.warmup_lr + (args.lr - args.warmup_lr) * epoch / args.warmup_epoch
    else:
        lr = args.lr
        if args.cos:  # cosine lr schedule
            lr *= 0.5 * (1. + math.cos(math.pi * (epoch - args.warmup_epoch) / (args.epochs - args.warmup_epoch)))
        else:  # stepwise lr schedule
            for milestone in args.schedule:
                lr *= 0.1 if epoch >= milestone else 1.
    for param_group in optimizer.param_groups:
        param_group['lr'] = lr

class data_prefetcher():
    def __init__(self, loader, MOCO=False):
        self.loader = iter(loader)
        self.moco_flag = MOCO
        self.stream = torch.cuda.Stream()
        self.mean = torch.tensor([0.485, 0.456, 0.406]).cuda().view(1, 3, 1, 1)
        self.std = torch.tensor([0.229, 0.224, 0.225]).cuda().view(1, 3, 1, 1)
        self.preload()

    def preload(self):
        try:
            self.next_input = next(self.loader)
        except StopIteration:
            self.next_input[0] = None
            self.next_input[1] = None
            return
        with torch.cuda.stream(self.stream):
            self.next_input[0][0] = self.next_input[0][0].cuda(non_blocking=True).float()
            self.next_input[0][1] = self.next_input[0][1].cuda(non_blocking=True).float()

            self.next_input[0][0] = self.next_input[0][0].sub_(self.mean).div_(self.std)
            self.next_input[0][1] = self.next_input[0][1].sub_(self.mean).div_(self.std)
            if self.moco_flag==False:
                self.next_input[0][2] = self.next_input[0][2].cuda(non_blocking=True).float()
                self.next_input[0][2] = self.next_input[0][2].sub_(self.mean).div_(self.std)

    def next(self):
        torch.cuda.current_stream().wait_stream(self.stream)
        input = self.next_input
        self.preload()
        return input

def train(train_loader, model, optimizer, epoch, args):
    batch_time, data_time = AverageMeter('t', ':.3f'), AverageMeter('dt', ':.3f')
    losses, LR = AverageMeter('loss', ':.2f'),  AverageMeter('lr', ':.1e')
    test_acc, top1, top2, top3 = AverageMeter('testAcc', ':.2f'), AverageMeter('Ac1', ':.2f'), AverageMeter('Ac2', ':.2f'), AverageMeter('Ac3', ':.2f')
    confidence_score1, confidence_score2 = AverageMeter('c1', ':0.2f'), AverageMeter('c2', ':.2f')
    progress = ProgressMeter(len(train_loader), [LR, batch_time, data_time, losses, test_acc, top1, top2, top3, confidence_score1, confidence_score2], prefix="Epoch:[{}]".format(epoch))

    if args.prior_confidence == 'cos': ## from min to max
        prior_confidence = args.prior_confidence_max - 0.5 * (args.prior_confidence_max - args.prior_confidence_min) * (
                    1 + math.cos(epoch / args.epochs * math.pi))
    else:
        prior_confidence = args.prior_confidence_min

    # switch to train mode
    model.train()
    end = time.time()

    prefetcher = data_prefetcher(train_loader,args.loss_type=='moco')
    images, _ = prefetcher.next()
    k = 0
    while images is not None:
        k += 1
        data_time.update(time.time() - end)
        optimizer.zero_grad()
        if args.loss_type == 'moco':
            q_w, k_w, q_inputs = images[0], images[1], None
        else:
            q_w, k_w, q_inputs = images[0], images[1], images[2]
        loss, testacc, acc, acc2, acc3, confidence_score_mean, confidence2_score_mean = model(q_w, k_w, q_inputs, prior_confidence)

        n_batch_size = q_w.size(0)
        losses.update(loss.item(), n_batch_size)
        test_acc.update(testacc, n_batch_size)
        top1.update(acc, n_batch_size)
        top2.update(acc2, n_batch_size)
        top3.update(acc3, n_batch_size)
        confidence_score1.update(confidence_score_mean, n_batch_size)
        confidence_score2.update(confidence2_score_mean, n_batch_size)
        LR.update(optimizer.param_groups[0]['lr'])

        # compute gradient and do SGD step
        optimizer.zero_grad()
        loss.backward()
        optimizer.step()

        # measure elapsed time
        batch_time.update(time.time() - end)
        end = time.time()

        if k % args.print_freq == 0:
            progress.display(k)

        images, _ = prefetcher.next()

    return losses.avg, test_acc.avg, top1.avg, top2.avg, top3.avg, confidence_score1.avg, confidence_score2.avg


if __name__ == '__main__':
    main()
