#
# Copyright (c) 2017 Intel Corporation
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#

"""
this rollout worker:

- restores a model from disk
- evaluates a predefined number of episodes
- contributes them to a distributed memory
- exits
"""

import time
import os
import math

from rl_coach.base_parameters import TaskParameters, DistributedCoachSynchronizationType
from rl_coach.checkpoint import CheckpointStateFile, CheckpointStateReader
from rl_coach.core_types import EnvironmentSteps, RunPhase, EnvironmentEpisodes
from rl_coach.data_stores.data_store import SyncFiles
from rl_coach.logger import screen, Logger


def wait_for(wait_func, data_store=None, timeout=10):
    """
    block until wait_func is true
    """
    for i in range(timeout):
        if data_store:
            data_store.load_from_store()

        if wait_func():
            return
        time.sleep(10)

    # one last time
    if wait_func():
        return

    raise ValueError((
        'Waited {timeout} seconds, but condition timed out'
    ).format(
        timeout=timeout,
    ))


def wait_for_checkpoint(checkpoint_dir, data_store=None, timeout=10):
    """
    block until there is a checkpoint in checkpoint_dir
    """
    chkpt_state_file = CheckpointStateFile(checkpoint_dir)

    def wait():
        return chkpt_state_file.read() is not None

    wait_for(wait, data_store, timeout)


def wait_for_trainer_ready(checkpoint_dir, data_store=None, timeout=10):
    """
    Block until trainer is ready
    """

    def wait():
        return os.path.exists(os.path.join(checkpoint_dir, SyncFiles.TRAINER_READY.value))

    wait_for(wait, data_store, timeout)


def should_stop(checkpoint_dir):
    return os.path.exists(os.path.join(checkpoint_dir, SyncFiles.FINISHED.value))


def rollout_worker(graph_manager, data_store, num_workers, task_parameters):
    has_teacher = True if task_parameters.teacher_checkpoint_restore_path else False
    is_current_iteration_teacher = False
    """
    wait for first checkpoint then perform rollouts using the model
    """
    checkpoint_dir = task_parameters.checkpoint_restore_path
    wait_for_checkpoint(checkpoint_dir, data_store)
    wait_for_trainer_ready(checkpoint_dir, data_store)

    graph_manager.create_graph(task_parameters)
    with graph_manager.phase_context(RunPhase.TRAIN):

        chkpt_state_reader = CheckpointStateReader(checkpoint_dir, checkpoint_state_optional=False)
        last_checkpoint = chkpt_state_reader.get_latest().num

        # this worker should play a fraction of the total playing steps per rollout
        # act_steps = graph_manager.agent_params.algorithm.num_consecutive_playing_steps / num_workers
        # training_steps = (graph_manager.improve_steps / act_steps.num_steps).num_steps
        act_steps = 1
        while True:
        # for i in range(training_steps):

            if should_stop(checkpoint_dir):
                break

            graph_manager.reset_internal_state(True)    # This solved one bug
            # graph_manager.act(act_steps, wait_for_full_episodes=graph_manager.agent_params.algorithm.act_for_full_episodes)
            graph_manager.act(EnvironmentSteps(num_steps=act_steps), wait_for_full_episodes=graph_manager.agent_params.algorithm.act_for_full_episodes)
            graph_manager.reset_internal_state(True)

            new_checkpoint = chkpt_state_reader.get_latest()
            if new_checkpoint and new_checkpoint.num > last_checkpoint:
            # if new_checkpoint.num > last_checkpoint and has_teacher:
                if last_checkpoint == 50:
                    is_current_iteration_teacher = False
                if has_teacher and last_checkpoint < 50:
                    is_current_iteration_teacher = not is_current_iteration_teacher
                    screen.log_title("Changed representation/instance transfer mode.")
                # if has_teacher:
                #     is_current_iteration_teacher = not is_current_iteration_teacher
                #     screen.log_title("Changed representation/instance transfer mode.")
                if graph_manager.agent_params.algorithm.distributed_coach_synchronization_type == DistributedCoachSynchronizationType.SYNC:
                # while new_checkpoint is None or new_checkpoint.num < last_checkpoint + 1:
                    if should_stop(checkpoint_dir):
                        break
                    if data_store:
                        data_store.load_from_store()
                # new_checkpoint = chkpt_state_reader.get_latest()

                    if is_current_iteration_teacher:
                        graph_manager.restore_teacher_checkpoint_to_online()
                    else:
                        graph_manager.restore_checkpoint()
                # graph_manager.restore_checkpoint()

                if graph_manager.agent_params.algorithm.distributed_coach_synchronization_type == DistributedCoachSynchronizationType.ASYNC:
                # if new_checkpoint is not None and new_checkpoint.num > last_checkpoint:
                    if is_current_iteration_teacher:
                        graph_manager.restore_teacher_checkpoint_to_online()
                    else:
                        graph_manager.restore_checkpoint()
                    # graph_manager.restore_checkpoint()

            if new_checkpoint is not None:
                last_checkpoint = new_checkpoint.num
