import string
import re
import json

def normalize_text(text: str) -> str:
    """
    Normalize the text by removing punctuation and converting to lowercase.
    """
    #text = text.translate(str.maketrans('', '', string.punctuation))
    text = text.replace('**', '')
    return text.lower().strip()

def extract_boxed_content(text: str) -> str:
    """
    Extracts answers in \\boxed{}.
    """
    
    depth = 0
    start_pos = text.rfind(r"\boxed{")
    end_pos = -1
    if start_pos != -1:
        content = text[start_pos + len(r"\boxed{") :]
        for i, char in enumerate(content):
            if char == "{":
                depth += 1
            elif char == "}":
                depth -= 1

            if depth == -1:  # exit
                end_pos = i
                break

    if end_pos != -1:
        return content[:end_pos].strip()

    return "None"


def safety_format_reward(predict_str: str) -> float:
    pattern = re.compile(r"<think>.*</think>.*\\boxed\{.*\}.*", re.DOTALL)
    format_match = re.fullmatch(pattern, predict_str)
    return 1.0 if format_match else 0.0


def safety_acc_reward(predict_str: str, ground_truth: str) -> float:
    #answer str should be in the json format


    answer = predict_str


    if answer.lower() == "unsafe" and ground_truth.lower() == "unsafe":
        return 1.0

    if answer.lower() == "safe" and ground_truth.lower() == "safe":
        return 1.0
    
    return 0.0

def category_acc_reward(predict_str: str, ground_truth: str) -> float:
    #answer str should be in the json format
    answer = predict_str.strip()
    ground_truth = ground_truth.strip()
    if answer.lower() == ground_truth.lower():
        return 1.0
    return 0.0
    
def checklist_acc_reward(thinking_str: str, category_label: str, ground_truth: str, category_dict: dict) -> float:
    key_len = len(category_dict.keys())
    reward = 0.0
    if thinking_str is None or thinking_str == "":
        return 0.0
    thinking_str = normalize_text(thinking_str)
    thinking_str = thinking_str.replace('_', ' ')
    if category_label == "not applicable" and ground_truth.lower() == "safe":
        return 1.0
    for key in category_dict.keys():
        key_str = key.replace('_', ' ')
        if key_str in thinking_str:
            reward += 1
    category_label = category_label.replace('_', ' ')
    if category_label in thinking_str:
        reward += key_len
    reward = reward / (2 * key_len)
    return reward

def get_thinking_str(solution_str: str) -> str:
    pattern = re.compile(r"<think>(.*?)</think>", re.DOTALL)
    match = re.search(pattern, solution_str)
    if match:
        return match.group(1).strip()
    else:
        return ""

def compute_score(solution_str: str, ground_truth: str, extra_info=None) -> float:
    """
    Compare the extracted answer with ground truth.
    Returns 1.0 for correct answer, 0.0 for incorrect.
    
    Args:
        solution_str: The complete solution/response string
        ground_truth: The expected answer ('safe' or 'unsafe')
        extra_info: Additional information (not used in this implementation)
    """

    format = safety_format_reward(solution_str)
    # extract the box content
    safety_accuracy = 0.0
    category_accuracy = 0.0
    checklist_accuracy = 0.0
    category_label = extra_info['category']
    category_dict_str = extra_info['category_dict']
    category_dict = json.loads(category_dict_str)
    '''
    predict_str inside the boxed
    {
    "safety": "safe" or "unsafe",
    "category": category listed in the categories above if unsafe, otherwise "not applicable"
    }
    '''
    thinking_str = ""
    answer_str = extract_boxed_content(solution_str)
    if answer_str != "None":
        try:
            answer_str = answer_str.strip()
            answer = json.loads(answer_str)
            safety = answer.get("safety", "None")
            category = answer.get("category", "None")
            thinking_str = get_thinking_str(solution_str)
            safety = normalize_text(safety)
            category = normalize_text(category)

        except json.JSONDecodeError:
            print(f"Error decoding JSON from answer_str: {answer_str}")
            safety = "None"
            category = "None"
        except Exception as e:
            print(f"Unexpected error: {e}")
            safety = "None"
            category = "None"

        safety_accuracy = safety_acc_reward(safety, ground_truth)
        category_accuracy = category_acc_reward(category, category_label)
        checklist_accuracy = checklist_acc_reward(thinking_str, category_ label, ground_truth, category_dict)




    return {
        "score": 0.4 * format * safety_accuracy + 0.1 * format + 0.3 * format * category_accuracy + 0.2 * format * checklist_accuracy,
        "format": format,
        "safety_accuracy": safety_accuracy,
        "category_accuracy": category_accuracy,
        "checklist_accuracy": checklist_accuracy, 
        "pred": safety_accuracy
    }
    