import math
import torch
import torch.nn as nn
import torch.nn.functional as F
from six import add_metaclass
from torch.autograd import Variable
from reparam_function import ReparamModule


class Bottleneck(nn.Module):
    def __init__(self, in_planes, growth_rate):
        super(Bottleneck, self).__init__()
        self.gn1 = nn.GroupNorm(2, in_planes)  # BN -> GN
        self.conv1 = nn.Conv2d(in_planes, 4 * growth_rate, kernel_size=1, bias=False)
        self.gn2 = nn.GroupNorm(2, 4 * growth_rate)  # BN -> GN
        self.conv2 = nn.Conv2d(4 * growth_rate, growth_rate, kernel_size=3, padding=1, bias=False)

    def forward(self, x):
        out = self.conv1(F.relu(self.gn1(x)))
        out = self.conv2(F.relu(self.gn2(out)))
        out = torch.cat([out, x], 1)
        return out


class Transition(nn.Module):
    def __init__(self, in_planes, out_planes):
        super(Transition, self).__init__()
        self.gn = nn.GroupNorm(2, in_planes)  # BN -> GN
        self.conv = nn.Conv2d(in_planes, out_planes, kernel_size=1, bias=False)

    def forward(self, x):
        out = self.conv(F.relu(self.gn(x)))
        out = F.avg_pool2d(out, 2)
        return out


class DenseNet(nn.Module):
    def __init__(self, block, nblocks, growth_rate=12, reduction=0.5, num_classes=10):
        super(DenseNet, self).__init__()
        self.growth_rate = growth_rate

        num_planes = 2 * growth_rate
        self.conv1 = nn.Conv2d(3, num_planes, kernel_size=3, padding=1, bias=False)

        self.dense1 = self._make_dense_layers(block, num_planes, nblocks[0])
        num_planes += nblocks[0] * growth_rate
        out_planes = int(math.floor(num_planes * reduction))
        self.trans1 = Transition(num_planes, out_planes)
        num_planes = out_planes

        self.dense2 = self._make_dense_layers(block, num_planes, nblocks[1])
        num_planes += nblocks[1] * growth_rate
        out_planes = int(math.floor(num_planes * reduction))
        self.trans2 = Transition(num_planes, out_planes)
        num_planes = out_planes

        self.dense3 = self._make_dense_layers(block, num_planes, nblocks[2])
        num_planes += nblocks[2] * growth_rate
        out_planes = int(math.floor(num_planes * reduction))
        self.trans3 = Transition(num_planes, out_planes)
        num_planes = out_planes

        self.dense4 = self._make_dense_layers(block, num_planes, nblocks[3])
        num_planes += nblocks[3] * growth_rate

        self.gn = nn.GroupNorm(2, num_planes)  # BN -> GN
        self.linear = nn.Linear(num_planes, num_classes)

    def _make_dense_layers(self, block, in_planes, nblock):
        layers = []
        for i in range(nblock):
            layers.append(block(in_planes, self.growth_rate))
            in_planes += self.growth_rate
        return nn.Sequential(*layers)

    def forward(self, x):
        out = self.conv1(x)
        out = self.trans1(self.dense1(out))
        out = self.trans2(self.dense2(out))
        out = self.trans3(self.dense3(out))
        out = self.dense4(out)
        out = F.avg_pool2d(F.relu(self.gn(out)), 4)
        out = out.view(out.size(0), -1)
        out = self.linear(out)
        return out


class DenseNet_fedlaw(ReparamModule):
    def __init__(self, block, nblocks, growth_rate=12, reduction=0.5, num_classes=10):
        super(DenseNet_fedlaw, self).__init__()
        self.growth_rate = growth_rate

        num_planes = 2 * growth_rate
        self.conv1 = nn.Conv2d(3, num_planes, kernel_size=3, padding=1, bias=False)

        self.dense1 = self._make_dense_layers(block, num_planes, nblocks[0])
        num_planes += nblocks[0] * growth_rate
        out_planes = int(math.floor(num_planes * reduction))
        self.trans1 = Transition(num_planes, out_planes)
        num_planes = out_planes

        self.dense2 = self._make_dense_layers(block, num_planes, nblocks[1])
        num_planes += nblocks[1] * growth_rate
        out_planes = int(math.floor(num_planes * reduction))
        self.trans2 = Transition(num_planes, out_planes)
        num_planes = out_planes

        self.dense3 = self._make_dense_layers(block, num_planes, nblocks[2])
        num_planes += nblocks[2] * growth_rate
        out_planes = int(math.floor(num_planes * reduction))
        self.trans3 = Transition(num_planes, out_planes)
        num_planes = out_planes

        self.dense4 = self._make_dense_layers(block, num_planes, nblocks[3])
        num_planes += nblocks[3] * growth_rate

        self.gn = nn.GroupNorm(2, num_planes)  # BN -> GN
        self.avgpool = nn.AdaptiveAvgPool2d((1, 1))  # <<< 修改这里
        self.linear = nn.Linear(num_planes, num_classes)

    def _make_dense_layers(self, block, in_planes, nblock):
        layers = []
        for i in range(nblock):
            layers.append(block(in_planes, self.growth_rate))
            in_planes += self.growth_rate
        return nn.Sequential(*layers)

    def forward(self, x):
        out = self.conv1(x)
        out = self.trans1(self.dense1(out))
        out = self.trans2(self.dense2(out))
        out = self.trans3(self.dense3(out))
        out = self.dense4(out)
        out = F.relu(self.gn(out))
        out = self.avgpool(out)       # -> (N, C, 1, 1)
        out = torch.flatten(out, 1)   # -> (N, C)
        out = self.linear(out)
        return out



# Factory functions
def DenseNet121(num_classes):
    return DenseNet(Bottleneck, [6, 12, 24, 16], growth_rate=32, num_classes=num_classes)

def DenseNet169(num_classes):
    return DenseNet(Bottleneck, [6, 12, 32, 32], growth_rate=32, num_classes=num_classes)

def DenseNet201(num_classes):
    return DenseNet(Bottleneck, [6, 12, 48, 32], growth_rate=32, num_classes=num_classes)

def DenseNet161(num_classes):
    return DenseNet(Bottleneck, [6, 12, 36, 24], growth_rate=48, num_classes=num_classes)

def densenet_cifar(num_classes):
    return DenseNet(Bottleneck, [6, 12, 24, 16], growth_rate=12, num_classes=num_classes)

def DenseNet121_fedlaw(num_classes):
    return DenseNet_fedlaw(Bottleneck, [6, 12, 24, 16], growth_rate=32, num_classes=num_classes)

def DenseNet169_fedlaw(num_classes):
    return DenseNet_fedlaw(Bottleneck, [6, 12, 32, 32], growth_rate=32, num_classes=num_classes)

def DenseNet201_fedlaw(num_classes):
    return DenseNet_fedlaw(Bottleneck, [6, 12, 48, 32], growth_rate=32, num_classes=num_classes)

def DenseNet161_fedlaw(num_classes):
    return DenseNet_fedlaw(Bottleneck, [6, 12, 36, 24], growth_rate=48, num_classes=num_classes)

def densenet_cifar_fedlaw(num_classes):
    return DenseNet_fedlaw(Bottleneck, [6, 12, 24, 16], growth_rate=12, num_classes=num_classes)
