import multiprocessing

def _exec_code(code: str, result_var_name: str, return_dict, context_vars=None):
    local_vars = {}
    if context_vars:
        local_vars.update(context_vars)
    try:
        exec(code, {}, local_vars)
        return_dict['result'] = local_vars.get(result_var_name, 'Error: No explicit result found.')
    except Exception as e:
        return_dict['result'] = f'Error: {str(e)}'

def clean_code(code: str) -> str:
    code = code.strip()
    if code.startswith("```"):
        if code.startswith("```python"):
            code = code[len("```python"):].lstrip()
        else:
            code = code[len("```"):].lstrip()
        if code.endswith("```"):
            code = code[:code.rfind("```")].rstrip()
    return code

def execute_python_code(code: str, result_var_name: str = 'answer', timeout: int = 10, table_data=None):
    """
    Executes Python code in a separate process with optional table_data passed in.
    Kills the process if it exceeds the timeout.
    """
    code = clean_code(code)
    manager = multiprocessing.Manager()
    return_dict = manager.dict()

    # Prepare context: pass table_data if available
    context = {}
    if table_data is not None:
        context['table_data'] = table_data

    process = multiprocessing.Process(
        target=_exec_code,
        args=(code, result_var_name, return_dict, context)
    )
    process.start()
    process.join(timeout)

    if process.is_alive():
        process.terminate()
        process.join()
        return f'Error: Code execution exceeded {timeout} seconds and was terminated.'

    return return_dict.get('result', 'Error: No result returned.')

# Test
if __name__ == "__main__":
    table_data = [
        ["id", "name", "score"],
        ["1", "Alice", "85"],
        ["2", "Bob", "90"],
        ["3", "Charlie", "75"]
    ]
    code = """
```python
# Accessing table_data
answer = table_data[2][2]  # Should be '90'
```"""
    result = execute_python_code(code, table_data=table_data)
    print(result)