import sqlite3
import pandas as pd
from typing import List, Union

def execute_sql_on_table(sql_str: str, table: List[List[str]]) -> Union[List[List[str]], str]:
    """
    Executes a SQL query on a single table with name 'TMP_TABLE'.

    Args:
        sql_str: SQL query string to execute.
        table: List of lists where the first row is the header and the rest are data.

    Returns:
        Query result as a list of lists (first row is column names), or an error message.
    """
    try:
        if len(table) < 2:
            return "Error: Table must have at least one header row and one data row."

        header = table[0]
        data = table[1:]
        df = pd.DataFrame(data, columns=header)

        # Create in-memory SQLite DB
        conn = sqlite3.connect(":memory:")
        df.to_sql("TMP_TABLE", conn, index=False, if_exists='replace')

        # Execute query
        result_df = pd.read_sql_query(sql_str, conn)
        conn.close()

        # Flatten the result to a 1D list
        result = result_df.values.flatten().tolist()
        return result

    except Exception as e:
        return f"Error: {str(e)}"


if __name__ == "__main__":
    table = [
        ["id", "name", "score"],
        ["1", "Alice", "85"],
        ["2", "Bob", "90"],
        ["3", "Charlie", "75"]
    ]

    sql = "SELECT name, score FROM TMP_TABLE WHERE CAST(score AS INT) >= 80;"
    result = execute_sql_on_table(sql, table)
    print(result)