from common.constants import ALI_API_MODELS, OPENAI_API_MODELS, ANTHROPIC_API_MODELS, \
                             OPENAI_LINEAGE, DEEPSEEK_API_MODELS, GOOGLE_API_MODELS, \
                             XAI_API_MODELS, ZHIPU_API_MODELS, MINIMAX_API_MODELS, \
                             MOONSHOT_API_MODELS, SPARK_API_MODELS, ARK_API_MODELS

def get_response(**kwargs):
    model_name = kwargs.get('model_name')
    if model_name in ALI_API_MODELS:
        return get_response_from_ali(**kwargs)
    elif model_name in OPENAI_API_MODELS:
        return get_response_from_openai(**kwargs) 
    elif model_name in ANTHROPIC_API_MODELS:
        return get_response_from_anthropic(**kwargs)
    elif model_name in DEEPSEEK_API_MODELS:
        return get_response_from_deepseek(**kwargs)
    elif model_name in GOOGLE_API_MODELS:
        return get_response_from_google(**kwargs)
    elif model_name in XAI_API_MODELS:
        return get_response_from_xai(**kwargs)
    elif model_name in ZHIPU_API_MODELS:
        return get_response_from_zhipuai(**kwargs)
    elif model_name in MINIMAX_API_MODELS: 
        return get_response_from_minimax(**kwargs)
    elif model_name in MOONSHOT_API_MODELS:
        return get_response_from_moonshot(**kwargs)
    elif model_name in SPARK_API_MODELS:
        return get_response_from_spark(**kwargs)
    elif model_name in ARK_API_MODELS:
        return get_response_from_ark(**kwargs)


def _build_common_settings(kwargs, include_system=True):
    messages = [
        {'role': 'user', 'content': kwargs['prompt']}
    ] 
    if include_system:
        messages.insert(0, {
            'role': 'system', 
            'content': kwargs.get('system_prompt', 'You are a helpful assistant.')
        })
    common = {
        "model": kwargs['model_name'],
        "messages": messages
    }
    if kwargs["model_name"] in OPENAI_LINEAGE or kwargs["model_name"] in ARK_API_MODELS:
        common.update({"max_completion_tokens": kwargs["max_tokens"]} if kwargs.get("max_tokens") != "Default" else {})
    else:
        common.update({"max_tokens": kwargs["max_tokens"]} if kwargs.get("max_tokens") != "Default" else {})
    common.update({"temperature": kwargs["temperature"]} if kwargs.get("temperature") != "Default" else {})
    common.update({"top_p": kwargs["top_p"]} if kwargs.get("top_p") != "Default" else {})

    return common


def get_response_from_ali(**kwargs):
    client, model_name, prompt, temperature, top_p,  max_tokens, thinking_budget, enable_intrinsic_reasoning = \
        kwargs.get('client'), kwargs.get('model_name'), kwargs.get('prompt'), \
        kwargs.get('temperature', None), kwargs.get('top_p', None), kwargs.get('max_tokens', 8192),  kwargs.get('thinking_budget', 8192), \
        kwargs.get('enable_intrinsic_reasoning', False)
    model_response = ""
    model_intrinsic_reasoning = ""
    completion_settings = _build_common_settings(kwargs) 
    try:
        completion_settings.update({
            "stream": True,
            "extra_body": {
                "enable_thinking": True,
                **({"thinking_budget": thinking_budget} if thinking_budget != "Default" else {})
            } if enable_intrinsic_reasoning else {"enable_thinking": False},
            "stream_options": {"include_usage": True},
        })
        completion = client.chat.completions.create(
            **completion_settings,
        )
    except Exception as e:
        if hasattr(e, "args") and e.args:
            err_msg = str(e.args[0])
            if "data_inspection_failed" in err_msg:
                print("ERROR: Data inspection failed. ")
                return f"{err_msg}", "", {"error": f"{err_msg}"}, completion_settings
        else:
            raise Exception(f"Error while getting response from Ali API: {e}")
    try:
        usage = completion.usage if hasattr(completion, 'usage') else None 
        for chunk in completion:
            if not chunk.choices:
                if hasattr(chunk, "usage") and chunk.usage is not None:
                    usage = chunk.usage.model_dump()
                continue
            delta = chunk.choices[0].delta
            if hasattr(delta, "reasoning_content") and delta.reasoning_content is not None:
                model_intrinsic_reasoning += delta.reasoning_content
            if hasattr(delta, "content") and delta.content:
                model_response += delta.content
    except Exception as e:
        if "Output data may contain inappropriate content." in str(e):
            print("ERROR: Output data may contain inappropriate content. ")
            return f"{e}", "", {"error": f"{e}"}, completion_settings
        else:
            raise Exception(f"Error while streaming response from Ali API: {e}")
    return model_response, model_intrinsic_reasoning, usage, completion_settings


def get_response_from_openai(**kwargs):
    client, model_name, prompt, temperature, top_p,  max_tokens, thinking_budget, enable_intrinsic_reasoning = \
        kwargs.get('client'), kwargs.get('model_name'), kwargs.get('prompt'), \
        kwargs.get('temperature', None), kwargs.get('top_p', None), kwargs.get('max_tokens', 8192),  kwargs.get('thinking_budget', 8192), \
        kwargs.get('enable_intrinsic_reasoning', False)
    model_response = None
    model_intrinsic_reasoning = None
    usage = None
    completion_settings = _build_common_settings(kwargs)
    if enable_intrinsic_reasoning:
        completion_settings["reasoning_effort"] = "medium"
    
    try:
        completion = client.chat.completions.create(
            **completion_settings,
        )
        model_response = completion.choices[0].message.content 
        usage = completion.usage.model_dump() if hasattr(completion, 'usage') else None
    except Exception as e:
        raise Exception(f"Error while getting response from OpenAI: {e}")
    return model_response, model_intrinsic_reasoning, usage, completion_settings 
    

def get_response_from_deepseek(**kwargs):
    client, model_name, prompt, temperature, top_p,  max_tokens, thinking_budget, enable_intrinsic_reasoning = \
        kwargs.get('client'), kwargs.get('model_name'), kwargs.get('prompt'), \
        kwargs.get('temperature', None), kwargs.get('top_p', None), kwargs.get('max_tokens', 8192),  kwargs.get('thinking_budget', 8192), \
        kwargs.get('enable_intrinsic_reasoning', False)
    model_response = None
    model_intrinsic_reasoning = None
    usage = None
    completion_settings = _build_common_settings(kwargs)
    try:
        completion = client.chat.completions.create(
            **completion_settings,
        )
        model_response = completion.choices[0].message.content 
        model_intrinsic_reasoning = completion.choices[0].message.reasoning_content if hasattr(completion.choices[0].message, 'reasoning_content') else None
        usage = completion.usage.model_dump() if hasattr(completion, 'usage') else None
    except Exception as e:
        raise Exception(f"Error while getting response from DeepSeek: {e}")
    return model_response, model_intrinsic_reasoning, usage, completion_settings 


def get_response_from_anthropic(**kwargs):
    client, model_name, prompt, temperature, top_p, max_tokens, thinking_budget, enable_intrinsic_reasoning = \
    kwargs.get('client'), kwargs.get('model_name'), kwargs.get('prompt'), \
    kwargs.get('temperature', None), kwargs.get('top_p', None), kwargs.get('max_tokens', 8192), kwargs.get('thinking_budget', 8192), \
    kwargs.get('enable_intrinsic_reasoning', False)

    model_response = ""
    model_intrinsic_reasoning = ""
    usage = None

    message_settings = _build_common_settings(kwargs, include_system=False)

    if enable_intrinsic_reasoning:
        message_settings["thinking"] = {
            "type": "enabled",
            "budget_tokens": thinking_budget,
        } if thinking_budget != "Default" else {"type": "enabled"}

    message_settings.update({
        "model": model_name,
        "messages": [{"role": "user", "content": prompt}],
        "max_tokens": max_tokens,
        "system": "You are a helpful assistant.",
    })

    with client.messages.stream(**message_settings) as stream:
        try:
            for event in stream:
                if event.type == "content_block_delta":
                    if event.delta.type == "thinking_delta":
                        model_intrinsic_reasoning += event.delta.thinking
                    elif event.delta.type == "text_delta":
                        model_response += event.delta.text
                elif event.type == "message_start":
                    usage = event.message.usage
                    usage = usage.model_dump() if usage else None
                elif event.type == "message_delta":
                    output_usage = getattr(event, "usage", None)
                    output_usage = output_usage.model_dump()
                    usage.update(output_usage)
        except Exception as e:
            raise Exception(f"Error while streaming response from Anthropic API: {e}")
    return model_response, model_intrinsic_reasoning, usage, message_settings
    

def get_response_from_google(**kwargs):
    client, model_name, prompt, temperature, top_p, max_tokens, thinking_budget, enable_intrinsic_reasoning = \
        kwargs.get('client'), kwargs.get('model_name'), kwargs.get('prompt'), \
        kwargs.get('temperature', None), kwargs.get('top_p', None), kwargs.get('max_tokens', 8192),  kwargs.get('thinking_budget', 8192), \
        kwargs.get('enable_intrinsic_reasoning', False)
    
    from google.genai import types
    model_response = ""
    model_intrinsic_reasoning = ""
    usage = None

    completion_settings = {}
    if temperature != "Default":
        completion_settings["temperature"] = temperature
    if top_p != "Default":
        completion_settings["top_p"] = top_p
    completion_settings["max_output_tokens"] = max_tokens
    completion_settings["system_instruction"] = "You are a helpful assistant."
      
    try:
        response = client.models.generate_content(
            model=model_name,
            contents=prompt,
            config=types.GenerateContentConfig(
                thinking_config=types.ThinkingConfig(thinking_budget=thinking_budget) if enable_intrinsic_reasoning \
                                else types.ThinkingConfig(thinking_budget=0),
                **completion_settings
            ),
        )
        # print(f"Response: {response}")
        model_response = response.text
        usage = {
            "prompt_tokens": response.usage_metadata.prompt_token_count,
            "thoughts_tokens": response.usage_metadata.thoughts_token_count,
            "output_tokens": response.usage_metadata.candidates_token_count,
            "total_tokens": response.usage_metadata.total_token_count
        }
    except Exception as e:
        raise Exception(f"Error while getting response from Google API: {e}")
    return model_response, model_intrinsic_reasoning, usage, completion_settings

    
def get_response_from_xai(**kwargs):
    from xai_sdk.chat import user, system
    client, model_name, prompt, temperature, top_p, max_tokens, thinking_budget, enable_intrinsic_reasoning = \
        kwargs.get('client'), kwargs.get('model_name'), kwargs.get('prompt'), \
        kwargs.get('temperature', None), kwargs.get('top_p', None), kwargs.get('max_tokens', 8192),  kwargs.get('thinking_budget', 8192), \
        kwargs.get('enable_intrinsic_reasoning', False)
    
    model_response = ""
    model_intrinsic_reasoning = ""
    usage = None

    completion_settings = _build_common_settings(kwargs)
    try:
        chat = client.chat.create(
            model=completion_settings["model"],
            messages=[
                system(completion_settings["messages"][0]["content"]),
                user(completion_settings["messages"][1]["content"]),
            ],
            temperature=completion_settings.get("temperature"),
            top_p=completion_settings.get("top_p"),
            max_tokens=completion_settings.get("max_tokens"),
        )
        response = chat.sample()
        model_response = response.content 
        model_intrinsic_reasoning = response.reasoning_content if hasattr(response, 'reasoning_content') else None
        usage = response.usage if hasattr(response, 'usage') else None
        usage = {
            'completion_tokens': getattr(usage, 'completion_tokens', 0),
            'prompt_tokens': getattr(usage, 'prompt_tokens', 0),
            'total_tokens': getattr(usage, 'total_tokens', 0),
            'prompt_text_tokens': getattr(usage, 'prompt_text_tokens', 0),
            'reasoning_tokens': getattr(usage, 'reasoning_tokens', 0),
            'cached_prompt_text_tokens': getattr(usage, 'cached_prompt_text_tokens', 0)
        }
    except Exception as e:
        raise Exception(f"Error while getting response from XAI API: {e}")
    
    return model_response, model_intrinsic_reasoning, usage, completion_settings


def get_response_from_zhipuai(**kwargs):
    client, model_name, prompt, temperature, top_p,  max_tokens, thinking_budget, enable_intrinsic_reasoning = \
        kwargs.get('client'), kwargs.get('model_name'), kwargs.get('prompt'), \
        kwargs.get('temperature', None), kwargs.get('top_p', None), kwargs.get('max_tokens', 8192),  kwargs.get('thinking_budget', 8192), \
        kwargs.get('enable_intrinsic_reasoning', False)
     
    model_response = ""
    model_intrinsic_reasoning = ""
    usage = None
    completion_settings = _build_common_settings(kwargs)

    try:
        completion = client.chat.completions.create(
            **completion_settings,
            thinking={
                "type": "enabled",
            } if enable_intrinsic_reasoning else {"type": "disabled"},
            stream=True,
        )
        for chunk in completion:
            if not chunk.choices:
                continue
            choice = chunk.choices[0]
            delta = chunk.choices[0].delta
            if hasattr(chunk, 'usage') and chunk.usage:
                usage = chunk.usage.model_dump()
            if hasattr(delta, "reasoning_content") and delta.reasoning_content is not None:
                model_intrinsic_reasoning += delta.reasoning_content
            if hasattr(delta, "content") and delta.content:
                model_response += delta.content
            if hasattr(choice, "finish_reason") and choice.finish_reason == "sensitive":
                model_response = "sensitive content"
                usage = {
                    "prompt_tokens": 0,
                    "completion_tokens": 0,
                    "completion_tokens_details": 0,
                    "total_tokens": 0
                }
                return model_response, model_intrinsic_reasoning, usage, completion_settings

    except Exception as e:
        raise Exception(f"Error while getting response from {model_name}: {e}")
    return model_response, model_intrinsic_reasoning, usage, completion_settings 


def get_response_from_minimax(**kwargs):
    client, model_name, prompt, temperature, top_p,  max_tokens, thinking_budget, enable_intrinsic_reasoning = \
        kwargs.get('client'), kwargs.get('model_name'), kwargs.get('prompt'), \
        kwargs.get('temperature', None), kwargs.get('top_p', None), kwargs.get('max_tokens', 8192),  kwargs.get('thinking_budget', 8192), \
        kwargs.get('enable_intrinsic_reasoning', False)
     
    model_response = ""
    model_intrinsic_reasoning = ""
    usage = None
    completion_settings = _build_common_settings(kwargs)

    try:
        completion = client.chat.completions.create(
            **completion_settings,
            stream=True,
            stream_options={"include_usage": True},
        )
        if hasattr(completion, 'usage'):
            usage = completion.usage.model_dump()
        
        for chunk in completion:
            input_sensitive = getattr(chunk, "input_sensitive", False)
            output_sensitive = getattr(chunk, "output_sensitive", False)

            if input_sensitive or output_sensitive: 
                model_response = "sensitive content"
                usage = {
                    "completion_tokens": 0,
                    "prompt_tokens": 0,
                    "total_tokens": 0,
                    "completion_tokens_details": None,
                    "prompt_tokens_details": None,
                    "total_characters": 0
                }
                return model_response, model_intrinsic_reasoning, usage, completion_settings 

            if chunk.choices:
                delta = chunk.choices[0].delta
                if getattr(delta, "reasoning_content", None):
                    model_intrinsic_reasoning += delta.reasoning_content
                if getattr(delta, "content", None):
                    model_response += delta.content
            if hasattr(chunk, 'usage') and chunk.usage:
                usage = chunk.usage.model_dump()

    except Exception as e:
        raise Exception(f"Error while getting response from {model_name}: {e}")
    return model_response, model_intrinsic_reasoning, usage, completion_settings


def get_response_from_moonshot(**kwargs):
    client, model_name, prompt, temperature, top_p,  max_tokens, thinking_budget, enable_intrinsic_reasoning = \
        kwargs.get('client'), kwargs.get('model_name'), kwargs.get('prompt'), \
        kwargs.get('temperature', None), kwargs.get('top_p', None), kwargs.get('max_tokens', 8192),  kwargs.get('thinking_budget', 8192), \
        kwargs.get('enable_intrinsic_reasoning', False)
     
    model_response = ""
    model_intrinsic_reasoning = ""
    usage = None
    completion_settings = _build_common_settings(kwargs)

    try:
        completion = client.chat.completions.create(
            **completion_settings,
            stream=True,
            stream_options={"include_usage": True},
        )
        if hasattr(completion, 'usage'):
            usage = completion.usage.model_dump()
        
        for chunk in completion:
            if chunk.choices:
                delta = chunk.choices[0].delta
                if getattr(delta, "reasoning_content", None):
                    model_intrinsic_reasoning += delta.reasoning_content
                if getattr(delta, "content", None):
                    model_response += delta.content
            if hasattr(chunk, 'usage') and chunk.usage:
                usage = chunk.usage.model_dump()

    except Exception as e:
        raise Exception(f"Error while getting response from {model_name}: {e}")
    return model_response, model_intrinsic_reasoning, usage, completion_settings


def get_response_from_spark(**kwargs):
    client, model_name, prompt, temperature, top_p,  max_tokens, thinking_budget, enable_intrinsic_reasoning = \
        kwargs.get('client'), kwargs.get('model_name'), kwargs.get('prompt'), \
        kwargs.get('temperature', None), kwargs.get('top_p', None), kwargs.get('max_tokens', 8192),  kwargs.get('thinking_budget', 8192), \
        kwargs.get('enable_intrinsic_reasoning', False)
     
    model_response = ""
    model_intrinsic_reasoning = ""
    usage = None
    completion_settings = _build_common_settings(kwargs)

    try:
        completion = client.chat.completions.create(
            **completion_settings,
            stream=True,
            stream_options={"include_usage": True},
        )
        if hasattr(completion, 'usage'):
            usage = completion.usage.model_dump()
        
        for chunk in completion:
            if hasattr(chunk, "code") and chunk.code in [10013, 10014, 10019]:
                model_response = "sensitive content"
                usage = {
                    "completion_tokens": 0,
                    "prompt_tokens": 0,
                    "total_tokens": 0,
                    "completion_tokens_details": None,
                    "prompt_tokens_details": None
                }
                return model_response, model_intrinsic_reasoning, usage, completion_settings
            
            if chunk.choices:
                delta = chunk.choices[0].delta
                if getattr(delta, "reasoning_content", None):
                    model_intrinsic_reasoning += delta.reasoning_content
                if getattr(delta, "content", None):
                    model_response += delta.content
            if hasattr(chunk, 'usage') and chunk.usage:
                usage = chunk.usage.model_dump()
            
    except Exception as e:
        raise Exception(f"Error while getting response from {model_name}: {e}")
    return model_response, model_intrinsic_reasoning, usage, completion_settings


def get_response_from_ark(**kwargs):
    client, model_name, prompt, temperature, top_p,  max_tokens, thinking_budget, enable_intrinsic_reasoning = \
        kwargs.get('client'), kwargs.get('model_name'), kwargs.get('prompt'), \
        kwargs.get('temperature', None), kwargs.get('top_p', None), kwargs.get('max_tokens', 8192),  kwargs.get('thinking_budget', 8192), \
        kwargs.get('enable_intrinsic_reasoning', False)
     
    model_response = ""
    model_intrinsic_reasoning = ""
    usage = None
    completion_settings = _build_common_settings(kwargs)

    try:
        completion = client.chat.completions.create(
            **completion_settings,
            extra_body={
                "thinking": {
                    "type": "enabled",
                } if enable_intrinsic_reasoning else {"type": "disabled"},
            },
            stream=True,
            stream_options={"include_usage": True},
        )
        
        for chunk in completion:
            if chunk.choices:
                delta = chunk.choices[0].delta
                if getattr(delta, "reasoning_content", None):
                    model_intrinsic_reasoning += delta.reasoning_content
                if getattr(delta, "content", None):
                    model_response += delta.content
            if hasattr(chunk, 'usage') and chunk.usage:
                usage = chunk.usage.model_dump()
            
    except Exception as e:
        raise Exception(f"Error while getting response from {model_name}: {e}")
    return model_response, model_intrinsic_reasoning, usage, completion_settings