import argparse, os, sys, glob
from ossaudiodev import SNDCTL_SEQ_CTRLRATE
from ast import parse
import cv2
import torch
import numpy as np
from omegaconf import OmegaConf
from PIL import Image
from tqdm import tqdm, trange
from imwatermark import WatermarkEncoder
from itertools import islice
from einops import rearrange
from torchvision.utils import make_grid
import time
from pytorch_lightning import seed_everything
from torch import autocast
from contextlib import contextmanager, nullcontext
from ldm.modules.attention import SpatialTransformer

from ldm.util import instantiate_from_config
from ldm.models.diffusion.ddim import DDIMSampler
from ldm.models.diffusion.plms import PLMSSampler

from diffusers.pipelines.stable_diffusion.safety_checker import StableDiffusionSafetyChecker
from transformers import AutoFeatureExtractor

import stanza
from nltk.tree import Tree
nlp = stanza.Pipeline(lang='en', processors='tokenize,pos,constituency')
import pdb
import json

# load safety model
safety_model_id = ""
safety_feature_extractor = AutoFeatureExtractor.from_pretrained(safety_model_id)
safety_checker = StableDiffusionSafetyChecker.from_pretrained(safety_model_id)


def preprocess_prompts(prompts):
    if isinstance(prompts, (list, tuple)):
        return [p.lower().strip().strip(".").strip() for p in prompts]
    elif isinstance(prompts, str):
        return prompts.lower().strip().strip(".").strip()
    else:
        raise NotImplementedError


def get_all_nps(tree, full_sent=None):
    start = 0
    end = len(tree.leaves())

    def get_sub_nps(tree, left, right):
        if isinstance(tree, str) or len(tree.leaves()) == 1:
            return []
        sub_nps = []
        n_leaves = len(tree.leaves())
        n_subtree_leaves = [len(t.leaves()) for t in tree]
        offset = np.cumsum([0] + n_subtree_leaves)[:len(n_subtree_leaves)]
        assert right - left == n_leaves
        if tree.label() == 'NP' and n_leaves > 1:
            sub_nps.append([" ".join(tree.leaves()), (int(left), int(right))])
        for i, subtree in enumerate(tree):
            sub_nps += get_sub_nps(subtree, left=left+offset[i], right=left+offset[i]+n_subtree_leaves[i])
        return sub_nps
    
    all_nps = get_sub_nps(tree, left=start, right=end)
    lowest_nps = []
    for i in range(len(all_nps)):
        span = all_nps[i][1]
        lowest = True
        for j in range(len(all_nps)):
            span2 = all_nps[j][1]
            if span2[0] >= span[0] and span2[1] <= span[1]:
                lowest = False
                break
        if lowest:
            lowest_nps.append(all_nps[i])

    all_nps, spans = map(list, zip(*all_nps))
    if full_sent and full_sent not in all_nps:
        all_nps = [full_sent] + all_nps
        spans = [(start, end)] + spans

    return all_nps, spans, lowest_nps

def expand_sequence(seq, length, dim=1):
    seq = seq.transpose(0, dim)
    max_length = seq.size(0)
    n_repeat = (max_length - 2) // length
    repeat_size = (n_repeat,) + (1, ) * (len(seq.size()) -1)

    eos = seq[length+1, ...].clone()
    segment = seq[1:length+1, ...].repeat(*repeat_size)
    seq[1:len(segment)+1] = segment
    seq[len(segment)+1] = eos

    return seq.transpose(0, dim)


# def align_sequence(main_seq, seq, span, eos_loc=None, dim=1):
#     seq = seq.transpose(0, dim)
#     main_seq = main_seq.transpose(0, dim)
#     start, end = span[0]+1, span[1]+1
#     seg_length = end - start

#     main_seq[start:end] = seq[1:1+seg_length]

#     if eos_loc is not None:
#         pad_length = len(main_seq) - eos_loc
#         main_seq[eos_loc:] = seq[1+seg_length:1+seg_length+pad_length]

#     return main_seq.transpose(0, dim)

def align_sequence(main_seq, seq, span, eos_loc, dim=1, zero_out=False, replace_pad=False):
    seq = seq.transpose(0, dim)
    main_seq = main_seq.transpose(0, dim)
    start, end = span[0]+1, span[1]+1
    seg_length = end - start

    main_seq[start:end] = seq[1:1+seg_length]
    if zero_out:
        main_seq[1:start] = 0
        main_seq[end:eos_loc] = 0

    if replace_pad:
        pad_length = len(main_seq) - eos_loc
        main_seq[eos_loc:] = seq[1+seg_length:1+seg_length+pad_length]
    

    return main_seq.transpose(0, dim)


def get_actions(tree, SHIFT = 0, REDUCE = 1, OPEN='(', CLOSE=')'):
    #input tree in bracket form: ((A B) (C D))
    #output action sequence: S S R S S R R
    actions = []
    tree = tree.strip()
    i = 0
    num_shift = 0
    num_reduce = 0
    left = 0
    right = 0
    while i < len(tree):
        if tree[i] != ' ' and tree[i] != OPEN and tree[i] != CLOSE: #terminal
            if tree[i-1] == OPEN or tree[i-1] == ' ':
                actions.append(SHIFT)
                num_shift += 1
        elif tree[i] == CLOSE:
            actions.append(REDUCE)
            num_reduce += 1
            right += 1
        elif tree[i] == OPEN:
            left += 1
        i += 1
    pdb.set_trace()
    assert(num_shift == num_reduce + 1)
    return actions


def chunk(it, size):
    it = iter(it)
    return iter(lambda: tuple(islice(it, size)), ())


def numpy_to_pil(images):
    """
    Convert a numpy image or a batch of images to a PIL image.
    """
    if images.ndim == 3:
        images = images[None, ...]
    images = (images * 255).round().astype("uint8")
    pil_images = [Image.fromarray(image) for image in images]

    return pil_images


def load_model_from_config(config, ckpt, verbose=False):
    print(f"Loading model from {ckpt}")
    pl_sd = torch.load(ckpt, map_location="cpu")
    if "global_step" in pl_sd:
        print(f"Global Step: {pl_sd['global_step']}")
    sd = pl_sd["state_dict"]
    model = instantiate_from_config(config.model)
    m, u = model.load_state_dict(sd, strict=False)
    if len(m) > 0 and verbose:
        print("missing keys:")
        print(m)
    if len(u) > 0 and verbose:
        print("unexpected keys:")
        print(u)

    model.cuda()
    model.eval()
    return model


def put_watermark(img, wm_encoder=None):
    if wm_encoder is not None:
        img = cv2.cvtColor(np.array(img), cv2.COLOR_RGB2BGR)
        img = wm_encoder.encode(img, 'dwtDct')
        img = Image.fromarray(img[:, :, ::-1])
    return img


def load_replacement(x):
    try:
        hwc = x.shape
        y = Image.open("assets/rick.jpeg").convert("RGB").resize((hwc[1], hwc[0]))
        y = (np.array(y)/255.0).astype(x.dtype)
        assert y.shape == x.shape
        return y
    except Exception:
        return x


def check_safety(x_image):
    safety_checker_input = safety_feature_extractor(numpy_to_pil(x_image), return_tensors="pt")
    x_checked_image, has_nsfw_concept = safety_checker(images=x_image, clip_input=safety_checker_input.pixel_values)
    assert x_checked_image.shape[0] == len(has_nsfw_concept)
    for i in range(len(has_nsfw_concept)):
        if has_nsfw_concept[i]:
            x_checked_image[i] = load_replacement(x_checked_image[i])
    return x_checked_image, has_nsfw_concept


def main():
    parser = argparse.ArgumentParser()

    parser.add_argument(
        "--prompt",
        type=str,
        nargs="?",
        default="a painting of a virus monster playing guitar",
        help="the prompt to render"
    )
    parser.add_argument(
        "--outdir",
        type=str,
        nargs="?",
        help="dir to write results to",
        default="outputs/txt2img-samples"
    )
    parser.add_argument(
        "--skip_grid",
        action='store_true',
        help="do not save a grid, only individual samples. Helpful when evaluating lots of samples",
    )
    parser.add_argument(
        "--skip_save",
        action='store_true',
        help="do not save individual samples. For speed measurements.",
    )
    parser.add_argument(
        "--ddim_steps",
        type=int,
        default=50,
        help="number of ddim sampling steps",
    )
    parser.add_argument(
        "--plms",
        action='store_true',
        help="use plms sampling",
    )
    parser.add_argument(
        "--laion400m",
        action='store_true',
        help="uses the LAION400M model",
    )
    parser.add_argument(
        "--fixed_code",
        action='store_true',
        help="if enabled, uses the same starting code across samples ",
    )
    parser.add_argument(
        "--ddim_eta",
        type=float,
        default=0.0,
        help="ddim eta (eta=0.0 corresponds to deterministic sampling",
    )
    parser.add_argument(
        "--n_iter",
        type=int,
        default=1,
        help="sample this often",
    )
    parser.add_argument(
        "--H",
        type=int,
        default=512,
        help="image height, in pixel space",
    )
    parser.add_argument(
        "--W",
        type=int,
        default=512,
        help="image width, in pixel space",
    )
    parser.add_argument(
        "--C",
        type=int,
        default=4,
        help="latent channels",
    )
    parser.add_argument(
        "--f",
        type=int,
        default=8,
        help="downsampling factor",
    )
    parser.add_argument(
        "--n_samples",
        type=int,
        default=3,
        help="how many samples to produce for each given prompt. A.k.a. batch size",
    )
    parser.add_argument(
        "--n_rows",
        type=int,
        default=0,
        help="rows in the grid (default: n_samples)",
    )
    parser.add_argument(
        "--scale",
        type=float,
        default=7.5,
        help="unconditional guidance scale: eps = eps(x, empty) + scale * (eps(x, cond) - eps(x, empty))",
    )
    parser.add_argument(
        "--from-file",
        type=str,
        help="if specified, load prompts from this file",
    )
    parser.add_argument(
        "--config",
        type=str,
        default="configs/stable-diffusion/v1-inference.yaml",
        help="path to config which constructs model",
    )
    parser.add_argument(
        "--ckpt",
        type=str,
        default="models/ldm/stable-diffusion-v1/model.ckpt",
        help="path to checkpoint of model",
    )
    parser.add_argument(
        "--seed",
        type=int,
        default=42,
        help="the seed (for reproducible sampling)",
    )
    parser.add_argument(
        "--precision",
        type=str,
        help="evaluate at this precision",
        choices=["full", "autocast"],
        default="autocast"
    )

    parser.add_argument(
        "--eval",
        action="store_true"
    )

    parser.add_argument(
        "--prompt_formation",
        choices=['no_pad', 'random', 'cls', 'first', 'none', 'half', 'last'],
        default='none'
    )

    parser.add_argument(
        "--start_idx",
        type=int,
        default=0
    )

    parser.add_argument(
        "--end_idx",
        type=int,
        default=-1,
    )

    parser.add_argument(
        "--save-x_T",
        action='store_true'
    )
    
    parser.add_argument(
        "--load_init",
        type=str,
        default=None
    )

    parser.add_argument(
        "--load_parsed_prompts",
        type=str,
        default=None
    )

    parser.add_argument(
        "--struct_attn",
        type=str,
        choices=['none', 'extend_str', 'extend_seq', 'align_seq'],
        default='none'
    )

    opt = parser.parse_args()

    if opt.eval:
        opt.n_iter = 1
        opt.skip_grid = True
        # if opt.prompt_formation == 'no_pad':
        #     opt.n_samples = 1

    if opt.load_init is not None:
        init_code = np.load(opt.load_init)

    if opt.laion400m:
        print("Falling back to LAION 400M model...")
        opt.config = "configs/latent-diffusion/txt2img-1p4B-eval.yaml"
        opt.ckpt = "models/ldm/text2img-large/model.ckpt"
        opt.outdir = "outputs/txt2img-samples-laion400m"

    seed_everything(opt.seed)

    config = OmegaConf.load(f"{opt.config}")
    model = load_model_from_config(config, f"{opt.ckpt}")

    device = torch.device("cuda") if torch.cuda.is_available() else torch.device("cpu")
    model = model.to(device)

    if opt.plms:
        sampler = PLMSSampler(model)
    else:
        sampler = DDIMSampler(model)

    os.makedirs(opt.outdir, exist_ok=True)
    outpath = opt.outdir

    print("Creating invisible watermark encoder (see...")
    wm = "StableDiffusionV1"
    wm_encoder = WatermarkEncoder()
    wm_encoder.set_watermark('bytes', wm.encode('utf-8'))

    batch_size = opt.n_samples
    n_rows = opt.n_rows if opt.n_rows > 0 else batch_size
    if not opt.from_file:
        prompt = opt.prompt
        assert prompt is not None
        data = [batch_size * [prompt]]
    else:
        print(f"reading prompts from {opt.from_file}")
        with open(opt.from_file, "r") as f:
            data = f.read().splitlines()
            try:
                opt.end_idx = len(data) if opt.end_idx == -1 else opt.end_idx
                data = data[:opt.end_idx]
                data, filenames = zip(*[d.strip("\n").split("\t") for d in data])
                data = list(chunk(data, batch_size))
            except:
                data = [batch_size * [d] for d in data]
        # data = list(chunk(data, batch_size))

    if opt.load_parsed_prompts:
        with open(opt.load_parsed_prompts, "r") as file:
            parsed_data = [json.loads(l.strip("\n")) for l in file.readlines()]
        batch_size = 1
        opt.n_samples = 1
        data, all_nps, all_spans = zip(*parsed_data)
        data = [[d] for d in data]
        with open("coco/val2014_10K.txt", "r") as f:
            filenames = f.read().splitlines()
            filenames = [d.strip("\n").split("\t")[1] for d in filenames]

    sample_path = os.path.join(outpath, "samples") if not opt.eval else outpath
    os.makedirs(sample_path, exist_ok=True)
    base_count = len(os.listdir(sample_path))
    grid_count = len(os.listdir(outpath)) - 1

    start_code = None
    if opt.fixed_code:
        start_code = torch.randn([opt.n_samples, opt.C, opt.H // opt.f, opt.W // opt.f], device=device)

    precision_scope = autocast if opt.precision=="autocast" else nullcontext
    with torch.no_grad():
        with precision_scope("cuda"):
            with model.ema_scope():
                tic = time.time()
                all_samples = list()
                for n in trange(opt.n_iter, desc="Sampling"):
                    for bid, prompts in enumerate(tqdm(data, desc="data")):
                        prompts = preprocess_prompts(prompts)

                        uc = None
                        if opt.scale != 1.0:
                            uc = model.get_learned_conditioning(batch_size * [""])
                        if isinstance(prompts, tuple):
                            prompts = list(prompts)
                        c = model.get_learned_conditioning(prompts)

                        # no use
                        # if opt.load_init is not None:
                            # start_code = torch.from_numpy(init_code[bid*batch_size:(bid+1)*batch_size]).to(device)
                        
                        if opt.eval:
                            # ['no_pad', 'random', 'cls', 'first', 'none']
                            seq_len = torch.tensor([len(k) for k in model.cond_stage_model.tokenizer(prompts).input_ids], device=c.device)
                            eos_loc = [l-1 for l in seq_len]
                            if opt.prompt_formation == 'no_pad':
                                seq_len = max(seq_len)
                                c = c[:, :seq_len]
                            elif opt.prompt_formation == 'random':
                                sample_token_idx = np.random.randint(1, seq_len.min().item())
                                c = torch.stack([
                                                 c[:, 0], 
                                                 c[:, sample_token_idx], 
                                                 c[range(batch_size), eos_loc]
                                                ], dim=1)
                            elif opt.prompt_formation == 'cls':
                                c = torch.stack([
                                                 c[:, 0], 
                                                 c[range(batch_size), eos_loc]
                                                ], dim=1)
                            elif opt.prompt_formation == 'first':
                                c = torch.stack([
                                                 c[:, 0], 
                                                 c[:, 1], 
                                                 c[range(batch_size), eos_loc]
                                                ], dim=1)
                            elif opt.prompt_formation == 'half':
                                # approximated implementation
                                # half_seq_len = int(seq_len.float().mean() // 2)
                                # if half_seq_len > seq_len.min(): half_seq_len = seq_len.min()
                                # c = torch.cat([
                                #                 c[:, :half_seq_len-1], 
                                #                 c[range(batch_size), eos_loc].unsqueeze(1) 
                                #               ], dim=1)

                                half_seq_len = [l//2 for l in seq_len]
                                max_seq_len = max(half_seq_len) + 1
                                half_c = [torch.cat((c[i, :half_l], c[i, loc:loc+max_seq_len-half_l]), dim=0) for i, (half_l, loc) in enumerate(zip(half_seq_len, eos_loc))]
                                c = torch.stack(half_c, dim=0)
                            elif opt.prompt_formation == 'parse':
                                pass
                            elif opt.prompt_formation == 'last':
                                c = c[:, :max(seq_len)]
                                for i in range(batch_size):
                                    c[i, eos_loc[i]:max(seq_len)] = uc[i, 1:1+max(seq_len)-eos_loc[i]]
                            else:
                                pass
                            uc = uc[:, :c.size(1)]
                            while len(uc.size()) < 3:
                                uc = uc.unsqueeze(1)
                        else:
                            if opt.load_parsed_prompts:
                                nps = all_nps[bid]
                                spans = all_spans[bid]
                            else:
                                doc = nlp(prompts[0])
                                mytree = Tree.fromstring(str(doc.sentences[0].constituency))
                                nps, spans, noun_chunk = get_all_nps(mytree, prompts[0])
                            
                            if opt.struct_attn == 'extend_str':    
                                # repeat each NP in string to expand the embed seq
                                for i in range(1, len(nps)):
                                    nps[i] = (" " + nps[i]) * (model.cond_stage_model.max_length // len(nps[i].split()))
                                nps = [[np]*len(prompts) for np in nps]
                                c = [model.get_learned_conditioning(np) for np in nps]
                            elif opt.struct_attn == 'extend_seq':
                                # repeat each NP after embedding
                                nps_length = [len(ids)-2 for ids in model.cond_stage_model.tokenizer(nps).input_ids] # not including bos eos
                                nps = [[np]*len(prompts) for np in nps]
                                c = [model.get_learned_conditioning(np) for np in nps]
                                c = [c[0]] + [expand_sequence(seq, l) for seq, l in zip(c[1:], nps_length[1:])]
                            elif opt.struct_attn == 'align_seq':
                                # re-align sequence base on spans
                                # nps = nps[:1] + [np for np, span in zip(nps[1:], spans[1:]) if span[0] >= 1]
                                # spans = spans[:1] + [span for span in spans[1:] if span[0] >= 1]

                                nps_length = [len(ids)-2 for ids in model.cond_stage_model.tokenizer(nps).input_ids]
                                nps = [[np]*len(prompts) for np in nps]
                                c = [model.get_learned_conditioning(np) for np in nps]
                                k_c = [c[0]] + [align_sequence(c[0].clone(), seq, span, nps_length[0]+1) for seq, span in zip(c[1:], spans[1:])]
                                v_c = [c[0]] + [align_sequence(c[0].clone(), seq, span, nps_length[0]+1) for seq, span in zip(c[1:], spans[1:])]
                                c = {'k': k_c, 'v': v_c}
                            else:
                                pass

                            # c = {'k': c[-1], 'v': c[-1]} # k heavily rely on the eos token, v is more important!!!

                        shape = [opt.C, opt.H // opt.f, opt.W // opt.f]
                        samples_ddim, intermediates = sampler.sample(S=opt.ddim_steps,
                                                         conditioning=c,
                                                         batch_size=opt.n_samples,
                                                         shape=shape,
                                                         verbose=False,
                                                         unconditional_guidance_scale=opt.scale,
                                                         unconditional_conditioning=uc,
                                                         eta=opt.ddim_eta,
                                                         x_T=start_code,
                                                         skip=(bid+1)*batch_size<opt.start_idx)

                        x_samples_ddim = model.decode_first_stage(samples_ddim)
                        x_samples_ddim = torch.clamp((x_samples_ddim + 1.0) / 2.0, min=0.0, max=1.0)
                        x_samples_ddim = x_samples_ddim.cpu().permute(0, 2, 3, 1).numpy()

                        x_checked_image = x_samples_ddim

                        x_checked_image_torch = torch.from_numpy(x_checked_image).permute(0, 3, 1, 2)

                        if not opt.skip_save and (bid+1)*batch_size>=opt.start_idx:
                            for sid, x_sample in enumerate(x_checked_image_torch):
                                x_sample = 255. * rearrange(x_sample.cpu().numpy(), 'c h w -> h w c')
                                img = Image.fromarray(x_sample.astype(np.uint8))          
                                try:
                                    count = bid * opt.n_samples + sid
                                    safe_filename = (filenames[count][:-4])[:150] + ".jpg"
                                    img.save(os.path.join(sample_path, f"{n}-{count}-{safe_filename}"))
                                except:
                                    img.save(os.path.join(sample_path, f"{base_count:05}-{n}-{prompts[0]}.png"))
                                base_count += 1  

                        if not opt.skip_grid:
                            all_samples.append(x_checked_image_torch)

                        if opt.save_x_T:
                            torch.save(intermediates['x_inter'][0].cpu(), f"eval_test/init_code/{bid}.pt")

                if not opt.skip_grid:
                    # additionally, save as grid
                    grid = torch.stack(all_samples, 0)
                    grid = rearrange(grid, 'n b c h w -> (n b) c h w')
                    grid = make_grid(grid, nrow=n_rows)

                    # to image
                    grid = 255. * rearrange(grid, 'c h w -> h w c').cpu().numpy()
                    img = Image.fromarray(grid.astype(np.uint8))
                    img = put_watermark(img, wm_encoder)
                    img.save(os.path.join(outpath, f'grid-{grid_count:04}.png'))
                    grid_count += 1

                toc = time.time()

    print(f"Your samples are ready and waiting for you here: \n{outpath} \n"
          f" \nEnjoy.")


if __name__ == "__main__":
    main()
