#!/bin/bash

SEEDS=(1234 2234 3234 4234 5234)
CONFIG='configs/config_complete.json'
DATASET='mosei'
MODEL='multimodaltransformer'
TRAIN_MODE='regression'
TRAIN_RATIOS=($(seq 0.1 0.1 1.0))

MODALITIES=(3) # 0: AT, 1: VT, 2: VA, 3: VAT
EMB_DIM=512 # [128, 256, 512], fixed to 512
 
DEVICE="cuda"  
GPU_ID=1

DATE=$(date +%Y%m%d_%H%M%S)  
LOG_DIR="logs/${DATE}"  
SAVE_DIR="results/saved_models/${DATE}"  

mkdir -p $LOG_DIR
mkdir -p $SAVE_DIR

BATCH_SIZE=128 # (fixed) mosi: 32; mosei: 128
EPOCH=100 # fixed

LEARNING_RATE=0.0001 # tune [0.001, 0.00001]
WEIGHT_DECAY=0.005 # tune [0.01, 0.001]
PATIENCE=5 # tune [5, 10]
EARLY_STOP=10
CLIP_VAL=1.0


export CUDA_LAUNCH_BLOCKING=1
export CUBLAS_WORKSPACE_CONFIG=:4096:8

    
for train_ratio in "${TRAIN_RATIOS[@]}"; do
    for mod_com in "${MODALITIES[@]}"; do
        exp_name="Modcom${mod_com}_ratio${train_ratio}_lr${LEARNING_RATE}_wd${WEIGHT_DECAY}"
        log_file="${LOG_DIR}/${exp_name}.log"
        
        echo "Starting experiment: ${exp_name} on GPU ${GPU_ID}"

        modal_params=""
    
        case $mod_com in
            0)  #  no vision
                modal_params="--no_vision"
                num_modal=2
                ;;
            1)  # no audio
                modal_params="--no_audio"
                num_modal=2
                ;;
            2)  # no text
                modal_params="--no_text"
                num_modal=2
                ;;
            3)  # all modalities
                modal_params=""
                num_modal=3
                ;;
        esac
        
        CUDA_VISIBLE_DEVICES=$GPU_ID python affect_multimodal.py \
            --config_path ${CONFIG} \
            --dataset_name ${DATASET} \
            --model_name ${MODEL} \
            --train_ratio $train_ratio \
            --batch_size ${BATCH_SIZE} \
            --epoch ${EPOCH} \
            $modal_params \
            --num_modal $num_modal \
            --embed_dim ${EMB_DIM} \
            --lr ${LEARNING_RATE} \
            --weight_decay ${WEIGHT_DECAY} \
            --early_stop  ${EARLY_STOP} \
            --patience ${PATIENCE} \
            --device ${DEVICE} \
            --save_dir ${SAVE_DIR} \
            --seeds ${SEEDS[@]} \
            2>&1 | tee $log_file
        
        sleep 3
    done
done

                   
echo "Experiments completed. Results saved in ${LOG_DIR}"
