#!/bin/bash

SEEDS=(1234 2234 3234 4234 5234) 
CONFIG='configs/config_complete.json'
DATASET='mosei'
MODEL='unimodaltransformer'
TRAIN_RATIO=1.0
TRAIN_MODE='regression'

MODALITIES=(0 1 2) #(0 1 2)  # 0:vision, 1:audio, 2:text 
INPUT_DIMS=(35 74 768) #(20 5 768); (35 74 768)
HIDDEN_DIMS=(512 256)

DEVICE="cuda"  
GPU_ID=7
DATE=$(date +%Y%m%d_%H%M%S)  
LOG_DIR="logs/${DATE}"  
SAVE_DIR="results/saved_models/${DATE}"  

mkdir -p $LOG_DIR
mkdir -p $SAVE_DIR


BATCH_SIZE=128 #32
EPOCH=100
LEARNING_RATE=0.0001 # tune [0.001, 0.00001]
WEIGHT_DECAY=0.005 # tune [0.01, 0.001]
PATIENCE=8 # tune [5, 10]
CLIP_VAL=1.0 # observe during training


set_random_seed() {
    local seed=$1
    export PYTHONHASHSEED=$seed
    export CUDA_LAUNCH_BLOCKING=1
    export CUBLAS_WORKSPACE_CONFIG=:4096:8
}


for seed in "${SEEDS[@]}"; do
    set_random_seed $seed
    
    for modal_idx in "${!MODALITIES[@]}"; do
        modality=${MODALITIES[$modal_idx]}
        input_dim=${INPUT_DIMS[$modal_idx]}
        exp_name="modal${modality}_ratio${TRAIN_RATIO}_seed${seed}"
        log_file="${LOG_DIR}/${exp_name}.log"
        
        echo "Starting experiment: ${exp_name} on GPU ${GPU_ID}"
        
        
        CUDA_VISIBLE_DEVICES=$GPU_ID python affect_unimodal.py \
            --config_path ${CONFIG} \
            --dataset_name ${DATASET} \
            --model_name ${MODEL} \
            --train_ratio ${TRAIN_RATIO} \
            --modality_num $modality \
            --input_dim $input_dim \
            --encoder_hidden_dim ${HIDDEN_DIMS[0]} \
            --head_hidden_dim ${HIDDEN_DIMS[1]} \
            --batch_size ${BATCH_SIZE} \
            --epoch ${EPOCH} \
            --lr ${LEARNING_RATE} \
            --early_stop \
            --patience ${PATIENCE} \
            --weight_decay ${WEIGHT_DECAY} \
            --clip_value ${CLIP_VAL} \
            --device ${DEVICE} \
            --save_dir ${SAVE_DIR} \
            --seed $seed \
            2>&1 | tee $log_file
        
        sleep 3
    done
done
                  
echo "Experiments completed. Results saved in ${LOG_DIR}"