# Copyright 2020-present, Pietro Buzzega, Matteo Boschini, Angelo Porrello, Davide Abati, Simone Calderara.
# All rights reserved.
# This source code is licensed under the license found in the
# LICENSE file in the root directory of this source tree.

import torch
import torch.nn as nn
import torch.nn.functional as F
from torch.nn.functional import relu, avg_pool2d
from typing import List


def conv3x3(in_planes: int, out_planes: int, stride: int = 1) -> F.conv2d:
    """
    Instantiates a 3x3 convolutional layer with no bias.
    :param in_planes: number of input channels
    :param out_planes: number of output channels
    :param stride: stride of the convolution
    :return: convolutional layer
    """
    return nn.Conv2d(in_planes, out_planes, kernel_size=3, stride=stride,
                     padding=1, bias=False)


class BasicBlock(nn.Module):
    """
    The basic block of ResNet.
    """
    expansion = 1

    def __init__(self, in_planes: int, planes: int, stride: int = 1) -> None:
        """
        Instantiates the basic block of the network.
        :param in_planes: the number of input channels
        :param planes: the number of channels (to be possibly expanded)
        """
        super(BasicBlock, self).__init__()
        self.conv1 = conv3x3(in_planes, planes, stride)
        self.bn1 = nn.BatchNorm2d(planes)
        self.conv2 = conv3x3(planes, planes)
        self.bn2 = nn.BatchNorm2d(planes)

        self.shortcut = nn.Sequential()
        if stride != 1 or in_planes != self.expansion * planes:
            self.shortcut = nn.Sequential(
                nn.Conv2d(in_planes, self.expansion * planes, kernel_size=1,
                          stride=stride, bias=False),
                nn.BatchNorm2d(self.expansion * planes)
            )

    def forward(self, x: torch.Tensor) -> torch.Tensor:
        """
        Compute a forward pass.
        :param x: input tensor (batch_size, input_size)
        :return: output tensor (10)
        """
        out = relu(self.bn1(self.conv1(x)))
        out = self.bn2(self.conv2(out))
        out += self.shortcut(x)
        out = relu(out)
        return out


class ResNet(nn.Module):
    """
    ResNet network architecture. Designed for complex datasets.
    """

    def __init__(self, block: BasicBlock, num_blocks: List[int],
                 num_classes: int, nf: int) -> None:
        """
        Instantiates the layers of the network.
        :param block: the basic ResNet block
        :param num_blocks: the number of blocks per layer
        :param num_classes: the number of output classes
        :param nf: the number of filters
        """
        super(ResNet, self).__init__()
        self.in_planes = nf
        self.block = block
        self.num_classes = num_classes
        self.nf = nf
        self.conv1 = conv3x3(3, nf * 1)
        self.bn1 = nn.BatchNorm2d(nf * 1)
        self.layer1 = self._make_layer(block, nf * 1, num_blocks[0], stride=1)
        self.layer2 = self._make_layer(block, nf * 2, num_blocks[1], stride=2)
        self.layer3 = self._make_layer(block, nf * 4, num_blocks[2], stride=2)
        self.layer4 = self._make_layer(block, nf * 8, num_blocks[3], stride=2)
        self.linear = nn.Linear(nf * 8 * block.expansion, num_classes)
       # self.linear2 = nn.Linear(nf * 8 * block.expansion, num_classes)
  #      self.simclr1 = nn.Linear(64, 128)
   #     self.simclr2 = nn.Linear(128, 128)
    #    self.simclr3 = nn.Linear(256, 128)
        self.simclr=nn.Linear(nf * 8 * block.expansion, 128)
        self.envs_predictor = nn.Linear(nf * 8 * block.expansion, 4)
        self.mask = ((torch.rand_like(torch.zeros([1,512])) > 0.8).float())#.repeat(out.shape[0], 1)
    #    self.simclr2 = nn.Linear(nf * 8 * block.expansion, 128)
        #self.predict = nn.Linear(nf * 8 * block.expansion,4)
      #  self.simclr = nn.Sequential(torch.nn.ReLU(),nn.Linear(nf * 8 * block.expansion, 1000),torch.nn.ReLU(),nn.Linear(1000,128))
      #  self.simclr2 = nn.Sequential(torch.nn.ReLU(),nn.Linear(nf * 8 * block.expansion, 1000),torch.nn.ReLU(),nn.Linear(1000,128))
        self._features = nn.Sequential(self.conv1,
                                       self.bn1,
                                       self.layer1,
                                       self.layer2,
                                       self.layer3,
                                       self.layer4
                                       )
        self.classifier = self.linear
        self.out_dim=nf * 8 * block.expansion
        self.num_classes=num_classes
   #     self.label_layer=nn.Linear(1,128)
    def f_train_feat_map(self, x: torch.Tensor,mask=None) -> torch.Tensor:

        out = relu(self.bn1(self.conv1(x)))
       # pdb.set_trace()
        out = self.layer1(out)#,None)#,mask)  # 64, 32, 32
        out = self.layer2(out)#,None)#,mask)  # 128, 16, 16
        out = self.layer3(out)#,None)  # 256, 8, 8
       # pdb.set_trace()

        #out = self.layer4.BasicBlock0
        out = self.layer4(out)#,None)  # 512, 4, 4
        #out = avg_pool2d(out, out.shape[2])  # 512, 1, 1
        #out = out.view(out.size(0), -1)  # 512
        return out
    def change_output_dim(self, new_dim, second_iter=False):
      #  self.prev_weights = nn.Linear(self.out_dim, self.num_classes + new_dim)
        in_features = self.out_dim
        out_features = self.num_classes + new_dim
        new_weights = nn.Linear(in_features, out_features)
        print("in_features:", in_features, "out_features:", out_features)
        new_weights.weight.data[:self.num_classes] = self.linear.weight.data
        new_weights.bias.data[:self.num_classes] = self.linear.bias.data
        self.linear = new_weights.cuda()
        self.linear.requires_grad = True
        self.num_classes = out_features

    def add_pseudo_dim(self, new_dim, second_iter=False):
        #self.prev_weights = nn.Linear(self.out_dim, self.num_classes + new_dim)
        in_features = self.out_dim
        out_features = 2*self.num_classes -new_dim
        new_weights = nn.Linear(in_features, out_features)
        print("in_features:", in_features, "out_features:", out_features)
        new_weights.weight.data[:self.num_classes] = self.linear.weight.data
        new_weights.bias.data[:self.num_classes] = self.linear.bias.data
        self.linear = new_weights.cuda()
        self.linear.requires_grad = True
    def del_pseudo_dim(self):
       # self.prev_weights = nn.Linear(self.out_dim, self.num_classes + new_dim)
        in_features = self.out_dim
        out_features = self.num_classes #- new_dim
        new_weights = nn.Linear(in_features, out_features)
        print("in_features:", in_features, "out_features:", out_features)
        new_weights.weight.data[:self.num_classes] = self.linear.weight.data[:self.num_classes]
        new_weights.bias.data[:self.num_classes] = self.linear.bias.data[:self.num_classes]
        self.linear = new_weights.cuda()
        self.linear.requires_grad = True
        #self.num_classes = out_features
    def _make_layer(self, block: BasicBlock, planes: int,
                    num_blocks: int, stride: int) -> nn.Module:
        """
        Instantiates a ResNet layer.
        :param block: ResNet basic block
        :param planes: channels across the network
        :param num_blocks: number of blocks
        :param stride: stride
        :return: ResNet layer
        """
        strides = [stride] + [1] * (num_blocks - 1)
        layers = []
        for stride in strides:
            layers.append(block(self.in_planes, planes, stride))
            self.in_planes = planes * block.expansion
        return nn.Sequential(*layers)

    def f_train(self, x: torch.Tensor) -> torch.Tensor:

        out = relu(self.bn1(self.conv1(x)))
        out = self.layer1(out)  # 64, 32, 32
        out = self.layer2(out)  # 128, 16, 16
        out = self.layer3(out)  # 256, 8, 8
        out = self.layer4(out)  # 512, 4, 4
        out = avg_pool2d(out, out.shape[2])  # 512, 1, 1
        out = out.view(out.size(0), -1)  # 512
        return out
    def f_train_2(self, x: torch.Tensor) -> torch.Tensor:

        with torch.no_grad():
            out = relu(self.bn1(self.conv1(x)))
            out = self.layer1(out)  # 64, 32, 32
            out = self.layer2(out)  # 128, 16, 16
            #out = self.layer3(out)
        out = out.requires_grad_()
        out = self.layer3(out)  # 256, 8, 8
        out = self.layer4(out)  # 512, 4, 4
        out = avg_pool2d(out, out.shape[2])  # 512, 1, 1
        out = out.view(out.size(0), -1)  # 512
        return out
    def f_train_feat_map(self, x: torch.Tensor,mask=None) -> torch.Tensor:

        out = relu(self.bn1(self.conv1(x)))
       # pdb.set_trace()
        out = self.layer1(out)#,None)#,mask)  # 64, 32, 32
        out = self.layer2(out)#,None)#,mask)  # 128, 16, 16
        out = self.layer3(out)#,None)  # 256, 8, 8
       # pdb.set_trace()

        #out = self.layer4.BasicBlock0
        out = self.layer4(out)#,None)  # 512, 4, 4
        #out = avg_pool2d(out, out.shape[2])  # 512, 1, 1
        #out = out.view(out.size(0), -1)  # 512
        return out
    def forward_2(self, x: torch.Tensor, is_simclr=False,is_predict=False,is_simclr_multi=False,is_sparse=False,y_range=None) :
        with torch.no_grad():
            out = relu(self.bn1(self.conv1(x)))
            out = self.layer1(out)  # 64, 32, 32
            out = self.layer2(out)  # 128, 16, 16
      #      out = self.layer3(out)
        out=out.requires_grad_()
        out = self.layer3(out)  # 256, 8, 8
        out = self.layer4(out)  # 512, 4, 4
        out = avg_pool2d(out, out.shape[2])  # 512, 1, 1
        out = out.view(out.size(0), -1)  # 512
        if is_simclr:
            feature=out
            return feature,self.simclr(out)
        else:
            return self.linear(out)
    def forward(self, x: torch.Tensor, is_simclr=False,is_predict=False,is_simclr_multi=False,is_sparse=False,y_range=None) :
        """
        Compute a forward pass.
        :param x: input tensor (batch_size, *input_shape)
        :return: output tensor (output_classes)
        """
        ''' 
        out = relu(self.bn1(self.conv1(x)))
        out = self.layer1(out)  # 64, 32, 32
        out = self.layer2(out)  # 128, 16, 16
        out = self.layer3(out)  # 256, 8, 8
        '''
        out = self.f_train(x)
        ''' 
        out = self.layer4(out)  # 512, 4, 4
        out = avg_pool2d(out, out.shape[2]) # 512, 1, 1
        out = out.view(out.size(0), -1)  # 512
        '''
        if is_sparse:
           # import pdb
           # pdb.set_trace()
            linear_no_sparse=self.linear(out)
            rate=torch.sqrt(1-F.softmax(linear_no_sparse).mean(dim=0)[y_range[0]:y_range[1]].sum())
           # print(rate)
            mask=((torch.rand_like(out[0])>rate).float()).repeat(out.shape[0],1)
            linear_sparse=self.linear(out*mask)#self.mask.repeat(out.shape[0], 1))
            class_mask=torch.zeros_like(linear_no_sparse)
            class_mask[:,y_range[0]:y_range[1]]=1
            return linear_sparse*(~(class_mask.bool()))+linear_no_sparse*(class_mask.bool())
           # linear_sparse[:, y_range] =  linear_no_sparse[:, y_range]
           # linear_no_sparse[:, y_range]
            #self.linear(out)[:, y_range]

            #inal_pred=torch.cat([self.linear(out)[:,y_range],])

           # return  self.envs_predictor(out)
        if is_simclr:
            feature=out
            return feature,self.simclr(out)
        else:
            return self.linear(out)

        #return out

    def features(self, x: torch.Tensor) -> torch.Tensor:
        """
        Returns the non-activated output of the second-last layer.
        :param x: input tensor (batch_size, *input_shape)
        :return: output tensor (??)
        """
        out = self._features(x)
        out = avg_pool2d(out, out.shape[2])
        feat = out.view(out.size(0), -1)
        return feat
   # def get_label_embedding(self,y):
    #    y_embedding = self.label_layer(y)
     #   return y_embedding

    def get_params(self) -> torch.Tensor:
        """
        Returns all the parameters concatenated in a single tensor.
        :return: parameters tensor (??)
        """
        params = []
        for pp in list(self.parameters()):
            params.append(pp.view(-1))
        return torch.cat(params)

    def set_params(self, new_params: torch.Tensor) -> None:
        """
        Sets the parameters to a given value.
        :param new_params: concatenated values to be set (??)
        """
        assert new_params.size() == self.get_params().size()
        progress = 0
        for pp in list(self.parameters()):
            cand_params = new_params[progress: progress +
                                               torch.tensor(pp.size()).prod()].view(pp.size())
            progress += torch.tensor(pp.size()).prod()
            pp.data = cand_params

    def get_grads(self) -> torch.Tensor:
        """
        Returns all the gradients concatenated in a single tensor.
        :return: gradients tensor (??)
        """
        grads = []
        for pp in list(self.parameters()):
            grads.append(pp.grad.view(-1))
        return torch.cat(grads)


def resnet18(nclasses: int, nf: int = 64) -> ResNet:
    """
    Instantiates a ResNet18 network.
    :param nclasses: number of output classes
    :param nf: number of filters
    :return: ResNet network
    """
    return ResNet(BasicBlock, [2, 2, 2, 2], nclasses, nf=64)
