import fcntl
import os
import json
import pickle as pkl
import hashlib
from collections import defaultdict

RESULT_PATH = '.'


class ResultLogger:
    Default_data = {}

    def __init__(self, fn, log_dir=RESULT_PATH, meta_data=None, file_format='json', no_lock=False):
        self.meta_data = meta_data if meta_data is not None else {}
        self.log_dir = log_dir
        self.file_format = file_format
        assert file_format in ['json', 'pickle', 'pkl']
        if not os.path.exists(log_dir):
            os.makedirs(log_dir, exist_ok=True)
        self.fp = os.path.join(log_dir, fn)
        self.no_lock = no_lock
        print(self.fp)
        if not os.path.exists(self.fp):
            open(self.fp, 'w', encoding='utf-8')
            self.fd = open(self.fp, 'r', encoding='utf-8')
            self.data = ResultLogger.Default_data.copy()
            self._save()
        else:
            self.fd = open(self.fp, 'r', encoding='utf-8')
            self.refresh()

        # self.lock = Lock()
        # self.mp_lock = multiprocessing.Lock()
        # self.mode_to_lock = {
        #     'mt': self.lock,
        #     'mp': self.mp_lock
        # }

    def _load(self):
        # must paired with save or unlock
        if not self.no_lock:
            fcntl.flock(self.fd, fcntl.LOCK_EX)
        if self.file_format == 'json':
            with open(self.fp, 'r', encoding='utf-8') as f:
                self.data = json.load(f)
        elif self.file_format in ['pickle', 'pkl']:
            with open(self.fp, 'rb') as f:
                self.data = pkl.load(f)

    def _save(self):
        # must paired with load or lock
        if self.file_format == 'json':
            with open(self.fp, 'w', encoding='utf-8') as f:
                json.dump(self.data, f)
        elif self.file_format in ['pickle', 'pkl']:
            with open(self.fp, 'wb') as f:
                pkl.dump(self.data, f)
        if not self.no_lock:
            fcntl.flock(self.fd, fcntl.LOCK_UN)

    def refresh(self):
        self._load()
        if not self.no_lock:
            fcntl.flock(self.fd, fcntl.LOCK_UN)

    def update(self, updater):
        self._load()
        # self.exp_params[uuid][int(exp_id)].update(status)
        self.data = updater(self.data)
        self._save()


class AttackResultLogger(ResultLogger):
    def __init__(self, fn, log_dir=RESULT_PATH, meta_data=None, file_format='json', no_lock=False):
        super(AttackResultLogger, self).__init__(fn, log_dir, meta_data, file_format, no_lock)

    def add_record(self, layout, agent_name, metadata, record):
        # metadata = {epi, horizon, n_games}
        self._load()
        if layout not in self.data:
            self.data[layout] = {}
        if agent_name not in self.data[layout]:
            self.data[layout][agent_name] = {}
        if metadata not in self.data[layout][agent_name]:
            self.data[layout][agent_name][metadata] = []

        self.data[layout][agent_name][metadata].append(record)

        self._save()

    def del_records(self, layout, agent_name, metadata):
        self._load()
        if layout in self.data and agent_name in self.data[layout]:
            self.data[layout][agent_name][metadata] = []
        self._save()

    def get_scores(self, layout, agent_name=None, metadata=None, show=True):
        agent_scores = self.data[layout]
        output = {}
        for an in ([agent_name] if agent_name is not None else agent_scores.keys()):
            if metadata is None:
                output[an] = agent_scores[an].copy()
            else:
                output[an] = {metadata: agent_scores[an][metadata].copy()}
        return output
