import random

FUNCTION_POOL = {
    "LOAD_SPATIOTEMPORAL_DATA": {
        "Call signature": "LOAD_SPATIOTEMPORAL_DATA(location, time, feature, region, time_int, period, unit, task)",
        "Type": "function",
        "Docstring": """
            Load spatiotemporal data for a specific location and time period.
       
            Parameters:
            - location (str):  The geographical location identifier.
            - time (datetime): The current time.
            - feature (str): The feature of interest(e.g., 'traffic speed', 'air quality').
            - region (str): The broader geographical area.
            - time_int (int): Interval in minutes between data points.
            - period (int): Duration for which data is loaded.
            - unit (str): The time unit of the period (e.g., 'hours', 'days').
            - task(str): Task of the query(e.g., 'analysis', 'anomaly detection', 'prediction'). 

            Returns:
            - DataFrame: Containing spatiotemporal data.
        """
    },
    "LOAD_SPATIAL_AUX_DATA": {
        "Call signature": "LOAD_SPATIAL_AUX_DATA(spatial_var, location, time, feature, region, time_int, period, unit, constraint)",
        "Type": "function",
        "Docstring": """
            Load auxiliary spatial data to support anomaly detection or analysis.

            Parameters:
            - spatial_var (str): Variable that defines the type of spatial data.
            - location (str):  The geographical location identifier.
            - time (datetime): The current time.
            - feature (str): The feature of interest(e.g., 'traffic speed', 'air quality').
            - region (str): The broader geographical area.
            - time_int (int): Interval in minutes between data points.
            - period (int): Duration for which data is loaded.
            - unit (str): The time unit of the period (e.g., 'hours', 'days').
            - constraint(str):  Additional constraint for data retrieval(e.g., 'weekdays only, weekends only'). Defaults to "None" which represents everyday.

            Returns:
            - DataFrame: Containing Spatial auxiliary data.
        """
    },
    "LOAD_TEMPORAL_AUX_DATA": {
        "Call signature": "LOAD_TEMPORAL_AUX_DATA(temp_var, location, time, feature, region, time_int, period, unit, constraint)",
        "Type": "function",
        "Docstring": """
            Load auxiliary temporal data to support anomaly detection or analysis.
            
            Parameters:
            - temp_var (str): Variable representing the type of temporal data.
            - location (str): The geographical location identifier.
            - time (datetime): The current time.
            - feature (str): The feature of interest(e.g., 'traffic speed', 'air quality').
            - region (str): The broader geographical area.
            - time_int (int): Interval in minutes between data points.
            - period (int): Duration for which data is loaded.
            - unit (str): The time unit of the period (e.g., 'hours', 'days').
            - constraint(str):  Additional constraint for data retrieval(e.g., 'weekdays only, weekends only'). Defaults to "None" which represents everyday.            
            Returns:
            - DataFrame: Containing Temporal auxiliary data.
        """
    },
    "IMPOSE_CONSTRAINTS": {
        "Call signature": "IMPOSE_CONSTRAINTS(data, time, time_int, period, unit, task, constraint, constraint_val, preds)",
        "Type": "function",
        "Docstring": """
            Impose data retrieval constraints for anomaly detection or analysis tasks or impose threshold constraints for prediction task.
            
            Parameters:
            - data (DataFrame): Data to which the data retrieval constraints are applied.
            - time (datetime): The current time.
            - time_int (int): Interval in minutes between data points.
            - period (int): Duration for which data is loaded.
            - unit (str): The time unit of the period (e.g., 'hours', 'days').
            - task(str): Task of the query(e.g., 'analysis', 'anomaly detection', 'prediction'). 
            - constraint(str): Additional constraint for data retrieval in analysis or anomaly detection tasks(e.g., 'weekdays only, weekends only'), defaults to "None" which represents everyday or Additional constraints for the forecasts task (e.g., 'traffic speed threshold'), defaults to "None".
            - constraint_val(int):  Value of the threshold contraint. Defaults to "None".
            - preds(str):  Predicted Values without constraints.

            Returns:
            - DataFrame: Filtered Data/ Results with constraint adherence.
        """
    },
    "ANALYZE_TREND": {
        "Call signature": "ANALYZE_TREND(data, feature, location, time_int, constraint, output_var)",
        "Type": "function",
        "Docstring": """
            Detect and Analyze trends in spatiotemporal data.
            
            Parameters:
            - data (DataFrame): Data on which to perform the trend analysis.
            - feature (str): The feature of interest(e.g., 'traffic speed', 'air quality').
            - location (str): The geographical location identifier.
            - time_int (int): Interval in minutes between data points.
            - constraint(str):  Additional constraint for data retrieval(e.g., 'weekdays only, weekends only'). Defaults to "None" which represents everyday.            
            Returns:
            - str: Summary of the trend analysis in the data.
        """
    },
    "ANALYZE_SEASONALITY": {
        "Call signature": "ANALYZE_SEASONALITY(data, time_int, constraint)",
        "Type": "function",
        "Docstring": """
            Detect and Analyze seasonality in spatiotemporal data.
            
            Parameters:
            - data (DataFrame): Data on which to perform the trend analysis.
            - time_int (int): Interval in minutes between data points.
            - constraint(str):  Additional constraint for data retrieval(e.g., 'weekdays only, weekends only'). Defaults to "None" which represents everyday.            
            Returns:
            - str: Summary of the seasonality analysis in the data.
        """
    },
    "ANALYZE_NEIGHBOURHOOD": {
        "Call signature": "ANALYZE_NEIGHBOURHOOD(feature, location, region)",
        "Type": "function",
        "Docstring": """
            Detect and Analyze seasonality in spatiotemporal data.
            
            Parameters:
            - feature (str): The feature of interest(e.g., 'traffic speed', 'air quality').
            - location (str):  The geographical location identifier.
            - region (str): The broader geographical area.
            
            Returns:
            - str: Summary of the spatial neighbourhood analysis in the data.
        """
    },
   "GEN_EXPLANATION": {
        "Call signature": "GEN_EXPLANATION(task, data, feature, location, region, time_int, horizon, horizon_unit, constraint, constraint_val, trend, seasonality, neighbourhood, preds, sensitivity)",
        "Type": "function",
        "Docstring": """
            Generate a comprehensive explanation of analyzed data.

            Parameters:
            - task(str): Task of the query(e.g., 'analysis', 'anomaly detection', 'prediction'). 
            - data (DataFrame): Data on which to generate the explanation.
            - feature (str): The feature of interest(e.g., 'traffic speed', 'air quality').
            - location (str): The geographical location identifier.
            - region (str): The broader geographical area.
            - time_int (int): Interval in minutes between data points.
            - horizon (int): Duration for which forecasts are made.
            - horizon_unit (str): The time unit of the horizon (e.g., 'hours', 'days').
            - constraint(str): Additional constraint for data retrieval in analysis or anomaly detection tasks(e.g., 'weekdays only, weekends only'), defaults to "None" which represents everyday or Additional constraints for the forecasts task (e.g., 'traffic speed threshold'), defaults to "None".
            - constraint_val(int):  Value of the threshold contraint. Defaults to "None".
            - trend (str): Trend result if trend analysis conducted. Defaults to "None".
            - seasonality (str): Seasonality result if seasonality analysis conducted. Defaults to "None".
            - neighbourhood (str): Neighbourhood result if neighbourhood analysis conducted. Defaults to "None".
            - preds(str): Prediction result if forecast task is executed. Defaults to "None".
            - sensitivity(str): Sensitivity result if sensitivity analysis on forecast task is conducted. Defaults to "None".
            
            Returns:
            - str: Detailed explanation of the results.
        """
    },
    "DETECT_ANOMALY_ST_DATA": {
        "Call signature": "DETECT_ANOMALY_ST_DATA(data, spatial_aux_data, temp_aux_data, temp_reasoning, spatial_reasoning, location, feature,time_int, constraint)",
        "Type": "function",
        "Docstring": """
            Detect anomalies in spatiotemporal data using optional auxiliary data.

            Parameters:
            - data (DataFrame): Data on which to perform the anomaly detection an analysis.
            - spatial_aux_data (DataFrame): Auxiliary spatial data.
            - temp_aux_data (DataFrame): Auxiliary temporal data.
            - temp_reasoning (bool): Whether to include temporal reasoning based on temporal auxiliary data.
            - spatial_reasoning (bool): Whether to include spatial reasoning based in spatial auxiliary data.
            - location (str): The geographical location identifier.
            - feature (str): The feature of interest(e.g., 'traffic speed', 'air quality').
            - time_int (int): Interval in minutes between data points.
            - constraint(str):  Additional constraint for data retrieval(e.g., 'weekdays only, weekends only'). Defaults to "None" which represents everyday.
            
            Returns:
            - str: Detailed explanation of detailed anomalies and their details.
        """
    },
    "FORECAST": {
        "Call signature": "FORECAST(data, location, time, feature, region, time_int, period, unit, horizon, horizon_unit)",
        "Type": "function",
        "Docstring": """
            Forecast spatiotemporal data.

            Parameters:
            - data (DataFrame): Data which is used to make the forecasts.
            - location (str):  The geographical location identifier.
            - time (datetime): The current time.
            - feature (str): The feature of interest(e.g., 'traffic speed', 'air quality').
            - region (str): The broader geographical area.
            - time_int (int): Interval in minutes between data points.
            - period (int): Duration for which data is loaded.
            - unit (str): The time unit of the period (e.g., 'hours', 'days').
            - horizon (int): Duration for which forecasts are made.
            - horizon_unit (str): The time unit of the horizon (e.g., 'hours', 'days').
           
            Returns:
            - str: Final Result of the Forecasts.
        """
    },
    "CONDUCT_SENSITIVITY_ANALYSIS": {
        "Call signature": "CONDUCT_SENSITIVITY_ANALYSIS(data, preds, location, time, feature, region, time_int, period, unit, horizon, horizon_unit)",
        "Type": "function",
        "Docstring": """
            Conduct spatial and temporal sensitvity analysis on the forecasts.

            Parameters:
            - data (DataFrame): Data which is used to make the forecasts.
            - preds(str): Prediction result.
            - location (str):  The geographical location identifier.
            - time (datetime): The current time.
            - feature (str): The feature of interest(e.g., 'traffic speed', 'air quality').
            - region (str): The broader geographical area.
            - time_int (int): Interval in minutes between data points.
            - period (int): Duration for which data is loaded.
            - unit (str): The time unit of the period (e.g., 'hours', 'days').
            - horizon (int): Duration for which forecasts are made.
            - horizon_unit (str): The time unit of the horizon (e.g., 'hours', 'days').
           
            Returns:
            - str: Detailed explanation of spatial and temporal sensitivity analysis.
        """
    },
    "REFINE_OUTPUT": {
            "Call signature": "REFINE_OUTPUT(var)",
            "Type": "function",
            "Docstring": """
                Return the final result of an analysis, anomaly detection or prediction task.
                
                Parameters:
                - var (any): The data or analysis outcome to be returned as the result.
                
                Returns:
                - any: The final result, which could be of any type depending on the task, such as a DataFrame, a list, a string, a dictionary, or even a simple scalar value.
            """
        },
}

ST_ANALYSIS_EXAMPLES = [
    """Question: Analyze the trend in the historical traffic speed data for the past 24 hours at location 76542 in the LA region, focusing specifically on weekdays only. The current time is 2017-03-01 01:00:00.The data is recorded at 5-minute intervals. Include the Trend Significance in the analysis.
Program:
DATA0 = LOAD_SPATIOTEMPORAL_DATA(location="76542", time="2017-03-01 01:00:00", feature='traffic speed', region='LA', time_int=5, period=24, unit="hours", task="analysis")
DATA0_CONST = IMPOSE_CONSTRAINTS(data=DATA0, time="2017-03-01 01:00:00", time_int=5, period=24, unit="hours", task="analysis", constraint="weekdays only", constraint_val="None", preds="None")
TREND_ANALYSIS = ANALYZE_TREND(data=DATA0_CONST, time_int=5, feature='traffic speed', location="76542", constraint="weekdays only")
EXPLANATION = GEN_EXPLANATION(task="analysis", data=DATA0_CONST, feature='traffic speed', location="76542", region="LA", time_int=5, horizon="None", horizon_unit="None", constraint="weekdays only", constraint_val="None", trend=TREND_ANALYSIS, seasonality="None", neighbourhood="None", preds="None", sensitivity="None")
FINAL_RESULT = REFINE_OUTPUT(var=EXPLANATION)
""",    
    """Question: Perform a seasonality analysis of the historical air quality data for the past 50 days at location tianjin_aq in the Shenzhen region focusing specifically on weekends only. The current time is 2020-09-05 02:05:00.The data is recorded at 60-minute intervals. Analyze the Daily Seasonality Strength and Weekly Seasonality Strength. 
Program:
DATA0 = LOAD_SPATIOTEMPORAL_DATA(location="tianjin_aq", time="2020-09-05 02:05:00", feature="air quality", region='Shenzhen', time_int=60, period=50, unit = "days", task="analysis")
DATA0_CONST = IMPOSE_CONSTRAINTS(data=DATA0, time="2020-09-05 02:05:00", time_int=60, period=50, unit = "days", task="analysis", constraint="weekends only", constraint_val="None", preds="None")
SEASONALITY_ANALYSIS = ANALYZE_SEASONALITY(data=DATA0_CONST, time_int=60, constraint="weekends only")
EXPLANATION = GEN_EXPLANATION(task="analysis", data=DATA0_CONST, feature='air quality', location="tianjin_aq", region="Shenzhen", time_int=60, horizon="None", horizon_unit="None", constraint="weekends only", constraint_val="None", trend="None", seasonality="SEASONALITY_ANALYSIS", neighbourhood="None", preds="None", sensitivity="None")
FINAL_RESULT = REFINE_OUTPUT(var=EXPLANATION)
""",
    """Question: Conduct a neighbourhood analysis of the historical air quality data for the past 21 days at location dongsi_aq in the Beijing region, focusing specifically on weekends only. The current time is 2017-09-05 02:05:00. The data is recorded at 60-minute intervals.Analyze the spatial Neighbours. 
Program:
DATA0 = LOAD_SPATIOTEMPORAL_DATA(location="dongsi_aq", time="2017-09-05 02:05:00", feature="air quality", region='Beijing', time_int=60, period=21, unit="days", task="analysis")
DATA0_CONST = IMPOSE_CONSTRAINTS(data=DATA0, time="2017-09-05 02:05:00", time_int=60, period=21, unit="days", task="analysis", constraint="weekends only", constraint_val="None", preds="None")
NEIGHBOURHOOD_ANALYSIS = ANALYZE_NEIGHBOURHOOD(feature='air quality', location="dongsi_aq", region='Beijing')
EXPLANATION = GEN_EXPLANATION(task="analysis", data=DATA0_CONST, feature='air quality', location="dongsi_aq", region="Beijing", time_int=60, horizon="None", horizon_unit="None", constraint="weekends only", constraint_val="None", trend="None", seasonality="None", neighbourhood=NEIGHBOURHOOD_ANALYSIS, preds="None", sensitivity="None")
FINAL_RESULT = REFINE_OUTPUT(var=EXPLANATION)
""",
    """Question: Analyze the trend in the historical traffic speed data for the past 12 hours at location 98021 in the BAY region, focusing on daily data. The current time is 2016-03-01 05:00:00. The data is recorded at 5-minute intervals. Analyze the trend significance. 
Program:
DATA0 = LOAD_SPATIOTEMPORAL_DATA(location="98021", time="2016-03-01 05:00:00", feature='traffic speed', region='BAY', time_int=5, period=12, unit="hours", task="analysis")
DATA0_CONST = IMPOSE_CONSTRAINTS(data=DATA0, time="2016-03-01 05:00:00", time_int=5, period=12, unit="hours", task="analysis", constraint="None", constraint_val="None", preds="None")
TREND_ANALYSIS = ANALYZE_TREND(data=DATA0_CONST, time_int=5, feature='traffic speed', location="98021", constraint="None")
EXPLANATION = GEN_EXPLANATION(task="analysis", data=DATA0_CONST, feature='traffic speed', location="98021", region="BAY", time_int=5, horizon="None", horizon_unit="None", constraint="None", constraint_val="None", trend=TREND_ANALYSIS, seasonality="None", neighbourhood="None", preds="None", sensitivity="None")
FINAL_RESULT = REFINE_OUTPUT(var=EXPLANATION)
""",       
    """Question: Conduct a seasonality analysis on the historical traffic speed data over the last 24 hours at the location 8991 in the LA region. The current timestamp is 2016-09-05 04:05:00. This data is collected in 5-minute intervals. Include the Daily Seasonality Strength and Weekly Seasonality Strength in the analysis. 
Program:
DATA0 = LOAD_SPATIOTEMPORAL_DATA(location="8991", time="2016-09-05 04:05:00", feature="traffic speed", region='LA', time_int=5, period=24, unit = "hours", task="analysis")
DATA0_CONST = IMPOSE_CONSTRAINTS(data=DATA0, time="2016-09-05 04:05:00", time_int=5, period=24, unit = "hours", task="analysis", constraint="None", constraint_val="None", preds="None")
SEASONALITY_ANALYSIS = ANALYZE_SEASONALITY(data=DATA0_CONST, time_int=5, constraint="None")
EXPLANATION = GEN_EXPLANATION(task="analysis", data=DATA0_CONST, feature='traffic speed', location="8991", region="LA", time_int=5, horizon="None", horizon_unit="None", constraint="None", constraint_val="None", trend="None", seasonality=SEASONALITY_ANALYSIS, neighbourhood="None", preds="None", sensitivity="None")
FINAL_RESULT = REFINE_OUTPUT(var=EXPLANATION)
""",
    """Question: Perform an analysis on the neighborhood effects concerning historical air quality over the last 60 days at the tiantian station in the Tianjin area, focusing everyday. The current date and time are 2016-03-11 11:05:00, with data being recorded every 60 minutes. Analyze the neighbours. Please offer a detailed explanation.
Program:
DATA0 = LOAD_SPATIOTEMPORAL_DATA(location="tiantian", time="2016-03-11 11:05:00", feature="air quality", region='Tianjin', time_int=60, period=60, unit="days", task="analysis")
DATA0_CONST = IMPOSE_CONSTRAINTS(data=DATA0, time="2016-03-11 11:05:00", time_int=60, period=60, unit="days", task="analysis", constraint="None", constraint_val="None", preds="None")
NEIGHBOURHOOD_ANALYSIS = ANALYZE_NEIGHBOURHOOD(feature='air quality', location="tiantian", region='Tianjin')
EXPLANATION = GEN_EXPLANATION(task="analysis", data=DATA0_CONST, feature='air quality', location="tiantian", region="Tianjin", time_int=60, horizon="None", horizon_unit="None", constraint="None", constraint_val="None", trend="None", seasonality="None", neighbourhood=NEIGHBOURHOOD_ANALYSIS, preds="None", sensitivity="None")
FINAL_RESULT = REFINE_OUTPUT(var=EXPLANATION)
""",
]

ST_ANOMALY_EXAMPLES = [
    """Question: Analyze the historical traffic speed data for the past 48 hours at location 76542 in the LA region focusing everyday, and detect any unusual traffic events. The current time is 2017-03-01 01:00:00. The data is recorded at 5-minute intervals. Provide spatial reasoning observing anomalies in nearby locations.
Program:
DATA0 = LOAD_SPATIOTEMPORAL_DATA(location="76542", time="2017-03-01 01:00:00", feature='traffic speed', region='LA', time_int=5, period=48, unit="hours", task="anomaly detection")
DATA0_CONST = IMPOSE_CONSTRAINTS(data=DATA0, time="2017-03-01 01:00:00", time_int=5, period=48, unit="hours", task="anomaly detection", constraint="None", constraint_val="None", preds="None")
DATA1 = LOAD_SPATIAL_AUX_DATA(spatial_var="neighbour", location="76542", time="2017-03-01 01:00:00", feature='traffic speed', region='LA', time_int=5, period=48, unit="hours", constraint="None")
EVENTS = DETECT_ANOMALY_ST_DATA(data=DATA0_CONST, spatial_aux_data=DATA1, temp_aux_data="None", temp_reasoning="False", spatial_reasoning="True", location ="76542", feature='traffic speed', time_int=5, constraint="None")
FINAL_RESULT = REFINE_OUTPUT(var=EVENTS)
""",
    """Question: Analyze the historical air quality data for the past 21 days at location dongsi_aq in the Beijing region,focusing specifically on weekends only and detect any unusual air quality patterns. The current time is 2017-09-05 02:05:00. The data is recorded at 60-minute intervals. Provide temporal reasoning observing patterns of corresponding weather data.
Program:
DATA0 = LOAD_SPATIOTEMPORAL_DATA(location="dongsi_aq", time="2017-09-05 02:05:00", feature='air quality', region='Beijing', time_int=60, period=21, unit="days", task="anomaly detection")
DATA0_CONST = IMPOSE_CONSTRAINTS(data=DATA0, time="2017-09-05 02:05:00", time_int=60, period=21, unit="days", task="anomaly detection", constraint="weekends only", constraint_val="None", preds="None")
DATA1 = LOAD_TEMPORAL_AUX_DATA(temp_var="weather", location="dongsi_aq", time="2017-09-05 02:05:00", feature='air quality', region='Beijing', time_int=60, period=21, unit="days", constraint="weekends only")
EVENTS = DETECT_ANOMALY_ST_DATA(data=DATA0_CONST, spatial_aux_data="None", temp_aux_data=DATA1, temp_reasoning="True", spatial_reasoning="False", location="dongsi_aq", feature='air quality',time_int=60, constraint="weekends only")
FINAL_RESULT = REFINE_OUTPUT(var=EVENTS)
""",
    """Question: Analyze the historical air quality data for the past 7 days at location 4007 in the Shenzhen region focusing everyday, and detect any unusual air quality patterns. The current time is 2015-04-06 06:00:00. The data is recorded at 60-minute intervals. Provide temporal reasoning observing patterns of corresponding weather data and spatial reasoning observing anomalies in nearby locations.
Program:
DATA0 = LOAD_SPATIOTEMPORAL_DATA(location="4007", time="2015-04-06 06:00:00", feature='air quality', region='Shenzhen', time_int=60, period=7, unit="days", task="anomaly detection")
DATA0_CONST = IMPOSE_CONSTRAINTS(data=DATA0, time="2015-04-06 06:00:00", time_int=60, period=7, unit="days", task="anomaly detection", constraint="None", constraint_val="None", preds="None")
DATA1 = LOAD_TEMPORAL_AUX_DATA(temp_var="weather", location="4007", time="2015-04-06 06:00:00", feature = 'air quality', region='Shenzhen', time_int=60, period=7, unit="days", constraint="None")
DATA2 = LOAD_SPATIAL_AUX_DATA(spatial_var="neighbour", location="4007", time="2015-04-06 06:00:00", feature='air quality', region='Shenzhen', time_int=60, period=7, unit="days", constraint="None")
EVENTS = DETECT_ANOMALY_ST_DATA(data=DATA0_CONST, spatial_aux_data=DATA2, temp_aux_data=DATA1, temp_reasoning="True", spatial_reasoning="True", location="4007", feature='air quality', time_int=60, constraint="None")
FINAL_RESULT = REFINE_OUTPUT(var=EVENTS)
""",
    """Question: Analyze the historical rainfall data for the past 3 weekdays at location 4421 in the Miami region, focusing on daily patterns, and detect any unusual rainfall patterns. The current time is 2020-08-01 12:00:00. The data is recorded at 20-minute intervals. Provide temporal reasoning observing patterns of corresponding weather data and spatial reasoning observing anomalies in nearby locations.
Program:
DATA0 = LOAD_SPATIOTEMPORAL_DATA(location="4421", time="2020-08-01 12:00:00", feature='rainfall', region='Miami', time_int=20, period=3, unit="days", task="anomaly detection")
DATA0_CONST = IMPOSE_CONSTRAINTS(data=DATA0, time="2020-08-01 12:00:00", time_int=20, period=3, unit="days", task="anomaly detection", constraint="None", constraint_val="weekdays only", preds="None")
DATA1 = LOAD_TEMPORAL_AUX_DATA(temp_var="weather", location="4421", time="2020-08-01 12:00:00", feature='rainfall', region='Miami', time_int=20, period=3, unit="days", constraint = "weekdays only")
DATA2 = LOAD_SPATIAL_AUX_DATA(spatial_var="neighbour", location="4421", time="2020-08-01 12:00:00", feature='rainfall', region='Miami', time_int=20, period=3, unit="days", constraint="weekdays only")
EVENTS = DETECT_ANOMALY_ST_DATA(data=DATA0_CONST, spatial_aux_data=DATA2, temp_aux_data=DATA1, temp_reasoning="True", spatial_reasoning="True", location="4421", feature='rainfall', time_int=20, constraint="weekdays only")
FINAL_RESULT = REFINE_OUTPUT(var=EVENTS)
""",
    """Question: Analyze the historical traffic congestion data for the past 48 hours at location 2309 in the New York region, and detect any unusual congestion patterns. The current time is 2018-05-20 09:00:00. The data is recorded at 10-minute intervals. Provide spatial reasoning observing anomalies in nearby locations.
Program:
DATA0 = LOAD_SPATIOTEMPORAL_DATA(location="2309", time="2018-05-20 09:00:00", feature='traffic congestion', region='New York', time_int=10, period=48, unit="hours", task="anomaly detection")
DATA0_CONST = IMPOSE_CONSTRAINTS(data=DATA0, time="2018-05-20 09:00:00", time_int=10, period=48, unit="hours", task="anomaly detection", constraint="None", constraint_val="None", preds="None")
DATA1 = LOAD_SPATIAL_AUX_DATA(spatial_var="neighbour", location="2309", time="2018-05-20 09:00:00", feature='traffic congestion', region='New York', time_int=10, period=48, unit="hours", constraint = "None")
EVENTS = DETECT_ANOMALY_ST_DATA(data=DATA0, spatial_aux_data=DATA1, temp_aux_data="None", temp_reasoning="False", spatial_reasoning="True", location="2309", feature='traffic congestion', time_int=10, constraint="None")
FINAL_RESULT = REFINE_OUTPUT(var=EVENTS)
""",
]

ST_FORECAST_EXAMPLES = [
    """Question: The current time is 2012-03-01 01:00:00. Predict the traffic speed of sensor ID 767542 in the LA region for the next 1 hour using the historical data of the past 1 hour with data points recorded at 5-minute intervals. Ensure the predicted traffic speed does not exceed the 100 km/h safety threshold. Analyze how daily timestamps and neighbouring sensors influence the traffic speed predictions.
Program:
DATA0 = LOAD_SPATIOTEMPORAL_DATA(location="767542", time="2012-03-01 01:00:00", feature='traffic speed', region="LA", time_int=5, period=1, unit="hours", task="prediction")
PREDICTION = FORECAST(data=DATA0, location="767542", time="2012-03-01 01:00:00", feature='traffic speed', region="LA", time_int=5, period=1, unit="hours", horizon=1, horizon_unit="hours")
PREDICTION_CONST = IMPOSE_CONSTRAINTS(data=DATA0, time="2012-03-01 01:00:00", time_int=5, period=1, unit="hours", task="prediction", constraint="traffic speed threshold", constraint_val=100, preds=PREDICTION)
SPATIOTEMPORAL_SENSITIVITY = CONDUCT_SENSITIVITY_ANALYSIS(data=DATA0, preds=PREDICTION_CONST, location="767542", time="2012-03-01 01:00:00", feature='traffic speed', region="LA", time_int=5, period=1, unit="hours", horizon=1, horizon_unit="hours")
EXPLANATION = GEN_EXPLANATION(task="prediction", data=DATA0, feature='traffic speed', location="767542", region="LA", time_int=5, horizon=1, horizon_unit="hours", constraint="traffic speed threshold", constraint_val=100, trend="None", seasonality="None", neighbourhood="None", preds=PREDICTION_CONST, sensitivity=SPATIOTEMPORAL_SENSITIVITY)
FINAL_RESULT = REFINE_OUTPUT(var=EXPLANATION)
""",
    """Question: The current time is 2012-03-05 02:05:00. Predict the traffic speed of sensor ID 769806 in the BAY region for the the next 15 minutes using the historical data of the past 1 hour with data points recorded at 5-minute intervals. Ensure the predicted traffic speed does not exceed the 85 km/h safety threshold. Analyze how daily timestamps and neighbouring sensors influence the traffic speed predictions.
Program:
DATA0 = LOAD_SPATIOTEMPORAL_DATA(location="769806", time="2012-03-05 02:05:00", feature='traffic speed', region="BAY", time_int=5, period=1, unit="hours", task="prediction")
PREDICTION = FORECAST(data=DATA0, location="769806", time="2012-03-05 02:05:00", feature='traffic speed', region="BAY", time_int=5, period=1, unit="hours", horizon=15, horizon_unit="minutes")
PREDICTION_CONST = IMPOSE_CONSTRAINTS(data=DATA0, time="2012-03-05 02:05:00", time_int=5, period=1, unit="hours", task="prediction", constraint="traffic speed threshold", constraint_val=85, preds=PREDICTION)
SPATIOTEMPORAL_SENSITIVITY = CONDUCT_SENSITIVITY_ANALYSIS(data=DATA0, preds=PREDICTION_CONST, location="769806", time="2012-03-05 02:05:00", feature='traffic speed', region="BAY", time_int=5, period=1, unit="hours", horizon=15, horizon_unit="minutes")
EXPLANATION = GEN_EXPLANATION(task="prediction", data=DATA0, feature='traffic speed', location="769806", region="BAY", time_int=5, horizon=15, horizon_unit="minutes", constraint="traffic speed threshold", constraint_val=85, trend="None", seasonality="None", neighbourhood="None", preds=PREDICTION_CONST, sensitivity=SPATIOTEMPORAL_SENSITIVITY)
FINAL_RESULT = REFINE_OUTPUT(var=EXPLANATION)
""",
    """Question: The current time is 2017-01-04 07:00:00. Predict the air quality of location dongsi_aq in the Beijing region for the the next 6 hours using the historical data of the past 24 hour with data points recorded at 60-minute intervals. Ensure the predicted air quality does not exceed 75 µg per cubic metre safety threshold. Analyze how daily timestamps and neighbouring sensors influence the air quality predictions.
Program:
DATA0 = LOAD_SPATIOTEMPORAL_DATA(location="dongsi_aq", time="2017-01-04 07:00:00", feature='air quality', region="Beijing", time_int=60, period=24, unit="hours", task="prediction")
PREDICTION = FORECAST(data=DATA0, location="dongsi_aq", time="2017-01-04 07:00:00", feature='air quality', region="Beijing", time_int=60, period=24, unit="hours", horizon=6, horizon_unit="hours")
PREDICTION_CONST = IMPOSE_CONSTRAINTS(data=DATA0, time="2017-01-04 07:00:00", time_int=60, period=24, unit="hours", task="prediction", constraint="air quality threshold", constraint_val=75, preds=PREDICTION)
SPATIOTEMPORAL_SENSITIVITY = CONDUCT_SENSITIVITY_ANALYSIS(data=DATA0, preds=PREDICTION_CONST, location="dongsi_aq", time="2017-01-04 07:00:00", feature='air quality', region="Beijing", time_int=60, period=24, unit="hours", horizon=6, horizon_unit="hours")
EXPLANATION = GEN_EXPLANATION(task="prediction", data=DATA0, feature='air quality', location="dongsi_aq", region="Beijing", time_int=60, horizon=6, horizon_unit="hours", constraint="air quality threshold", constraint_val=75, trend="None", seasonality="None", neighbourhood="None", preds=PREDICTION_CONST, sensitivity=SPATIOTEMPORAL_SENSITIVITY)
FINAL_RESULT = REFINE_OUTPUT(var=EXPLANATION)
""",
     """Question: The current time is 2018-05-10 15:30:00. Predict the water level of sensor ID 325698 in the Nile Delta region for the next 2 hours using the historical data of the past 3 hours with data points recorded at 30-minute intervals. Ensure the predicted water level does not exceed the 5 meters safety threshold. Analyze how daily timestamps and neighbouring sensors influence the water level predictions.
Program:
DATA0 = LOAD_SPATIOTEMPORAL_DATA(location="325698", time="2018-05-10 15:30:00", feature='water level', region="Nile Delta", time_int=30, period=3, unit="hours", task="prediction")
PREDICTION = FORECAST(data=DATA0, location="325698", time="2018-05-10 15:30:00", feature='water level', region="Nile Delta", time_int=30, period=3, unit="hours", horizon=2, horizon_unit="hours")
PREDICTION_CONST = IMPOSE_CONSTRAINTS(data=DATA0, time="2018-05-10 15:30:00", time_int=30, period=3, unit="hours", task="prediction", constraint="water level threshold", constraint_val=5, preds=PREDICTION)
SPATIOTEMPORAL_SENSITIVITY = CONDUCT_SENSITIVITY_ANALYSIS(data=DATA0, preds=PREDICTION_CONST, location="325698", time="2018-05-10 15:30:00", feature='water level', region="Nile Delta", time_int=30, period=3, unit="hours", horizon=2, horizon_unit="hours")
EXPLANATION = GEN_EXPLANATION(task="prediction", data=DATA0, feature='water level, location="325698", region="Nile Delta", time_int=30, horizon=2, horizon_unit="hours", constraint="water level threshold", constraint_val=5, trend="None", seasonality="None", neighbourhood="None", preds=PREDICTION_CONST, sensitivity=SPATIOTEMPORAL_SENSITIVITY)
FINAL_RESULT = REFINE_OUTPUT(var=EXPLANATION)
""",
]

ST_EXAMPLES = ST_ANALYSIS_EXAMPLES + ST_ANOMALY_EXAMPLES + ST_FORECAST_EXAMPLES

PROMPT_EXAMPLES = {
    'comb_tasks': ST_EXAMPLES,
    'analysis': ST_ANALYSIS_EXAMPLES,
    'anomaly': ST_ANOMALY_EXAMPLES,
    'forecast': ST_FORECAST_EXAMPLES
}


def create_prompt(inputs, num_prompts=8, method='random',func_pool=False, example_type = "comb_tasks", seed=42, group=0):
    if method=='all':
        prompt_examples = PROMPT_EXAMPLES[example_type]
    elif method=='random':
        random.seed(seed)
        prompt_examples = random.sample(PROMPT_EXAMPLES[example_type], num_prompts)
    else:
        raise NotImplementedError

    # print(prompt_examples)

    # if func_pool:
    #     function_details = FUNCTION_POOL
    #     full_prompt = f"Think step by step to answer the question. Output only the corresponding Program.\nFunction Details:\n{function_details}\n\nExamples:\n{prompt_examples}"
    # else:
    #     full_prompt = f"Think step by step to answer the question. Output only the corresponding Program.\n\nExamples:\n{prompt_examples}"

    if func_pool:
        function_details = FUNCTION_POOL
        full_prompt = f"Use the following details of functions and examples to answer the question. Think step by step and output only the corresponding Program for the given question.\nFunction Details:\n{function_details}\n\nExamples:\n{prompt_examples}"
    else:
        full_prompt = f"Follow the below examples to answer the question. Think step by step and output only the corresponding Program for the given question.\n\nExamples:\n{prompt_examples}"

    # print(full_prompt + "\nQuestion: {question}\nProgram:".format(**inputs))
    
    return full_prompt + "\nQuestion: {question}\nProgram:".format(**inputs)


def create_examples(num_prompts=8, method='random',func_pool=False, example_type = "comb_tasks", seed=42):
    if method=='all':
        prompt_examples = PROMPT_EXAMPLES[example_type]
    elif method=='random':
        random.seed(seed)
        prompt_examples = random.sample(PROMPT_EXAMPLES[example_type], num_prompts)
    else:
        raise NotImplementedError

    return prompt_examples
