import os
import numpy as np

def augment_noise_with_peak_and_save_positions(
    peak_dir: str,
    noise_dir: str,
    output_dir: str,
    positions_dir: str,
    window_len: int = 7,
    scale: float = 0.3,      # 根据实际情况决定
    num_positions: int = 3,
    random_seed: int = None
):
    """
    从 peak_dir 中每个 .npy 文件随机选取 num_positions 个
    长度为 window_len 的窗口，将其 * scale 后叠加到 noise_dir
    对应文件的相同位置，保存结果到 output_dir，并将每个窗口
    的索引数组保存到 positions_dir（.npy 格式）。
    """
    if random_seed is not None:
        np.random.seed(random_seed)

    os.makedirs(output_dir, exist_ok=True)
    os.makedirs(positions_dir, exist_ok=True)

    for fname in os.listdir(peak_dir):
        if not fname.lower().endswith('.npy'):
            continue

        peak_path = os.path.join(peak_dir, fname)
        noise_path = os.path.join(noise_dir, fname)
        out_path = os.path.join(output_dir, fname)
        pos_path = os.path.join(positions_dir, fname)

        # 加载数据
        peak_data = np.load(peak_path)
        noise_data = np.load(noise_path)
        if peak_data.shape != noise_data.shape:
            raise ValueError(f"Shape mismatch for {fname}: "
                             f"{peak_data.shape} vs {noise_data.shape}")

        length = peak_data.shape[0]
        max_start = length - window_len
        if max_start < 0:
            raise ValueError(f"File {fname} too short (len={length}) for window_len={window_len}")

        # 随机选取不重叠的窗口起始索引
        starts = np.random.choice(
            np.arange(0, max_start + 1),
            size=num_positions,
            replace=False
        )

        # 生成并保存每个窗口的索引数组
        positions = np.vstack([np.arange(s, s + window_len) for s in starts])
        # positions shape: (num_positions, window_len)
        np.save(pos_path, positions)

        # 叠加 peak 到 noise
        result = noise_data.copy()
        for s in starts:
            window = peak_data[s:s + window_len] * scale
            result[s:s + window_len] += window

        # 保存叠加后的结果
        np.save(out_path, result)
        print(f"[OK] {fname} → saved to {out_path}; positions saved to {pos_path}")

if __name__ == '__main__':
    peak_dir      = r""
    noise_dir     = r""
    output_dir    = r""
    positions_dir = r""

    augment_noise_with_peak_and_save_positions(
        peak_dir=peak_dir,
        noise_dir=noise_dir,
        output_dir=output_dir,
        positions_dir=positions_dir,
        window_len=7,
        scale=0.3,
        num_positions=3,
        random_seed=42
    )

