import os
import numpy as np
from pathlib import Path

import os
import numpy as np
from pathlib import Path

def add_gyro_noise(gyro_data, fs=100, arw_level=0.005, bi_level=1e-4, qn_level=0.001, seed=None):
    """
    对一段陀螺仪时序数据 gyro_data (shape: (N,)) 添加 ARW + Bias Instability + Quantization Noise (QN) 噪声。
      gyro_data: 原始数据 (例如静止时的陀螺仪输出)，长度 N
      fs:        采样频率(Hz)
      arw_level: 角随机游走 (deg/sqrt(hr))，示例值，请根据需求调整
      bi_level:  零偏不稳定度 (deg/h)，示例值，请根据需求调整
      qn_level:  量化噪声标准差 (deg)，示例值，请根据需求调整
    """
    if seed is not None:
        np.random.seed(seed)

    N = len(gyro_data)
    dt = 1.0 / fs  # 采样间隔

    # ---- 1) 角随机游走 ARW: 视为白噪声
    # 如果 arw_level 单位为 deg/sqrt(hr)，则 sigma_arw = arw_level * sqrt(fs/3600)
    sigma_arw = arw_level * np.sqrt(fs / 3600.0)
    arw_noise = np.random.normal(0, sigma_arw, N)

    # ---- 2) Bias Instability (BI): 模拟为低频随机游走
    sigma_bi = bi_level * np.sqrt(dt)
    bi_noise = np.zeros(N)
    for i in range(1, N):
        bi_noise[i] = bi_noise[i-1] + np.random.normal(0, sigma_bi)

    # ---- 3) Quantization Noise (QN): 模拟为白噪声
    # 这里可以使用正态分布，也可以改为均匀分布模拟量化误差
    qn_noise = np.random.normal(0, qn_level, N)

    # 总噪声为各部分的叠加
    total_noise = arw_noise + bi_noise + qn_noise
    noisy_gyro = ((gyro_data*0.0011154172318002256 + 0.029607261518762952 + total_noise)-0.029607261518762952)/0.0011154172318002256
    return noisy_gyro

def main():
    # 1) 输入/输出文件夹
    input_folder = r""
    output_folder = r""
    os.makedirs(output_folder, exist_ok=True)

    # 2) 收集所有 .npy 文件，并排序
    input_path = Path(input_folder)
    output_path = Path(output_folder)
    npy_files = sorted(input_path.glob("*.npy"))

    # 3) 分批: 每200个文件合成一次 512s 数据 (200 x 256 = 51200点)
    chunk_size = 20
    fs = 100  # 采样频率 100Hz
    arw_level = 0.006073  # 根据需求调整
    bi_level = 0.000232908  # 根据需求调整
    qn_level = 0.002   # 根据需求调整

    total_files = len(npy_files)
    print(f"Total npy files: {total_files}")

    # 假设 total_files 是 chunk_size 的整数倍；否则需要额外处理
    for i in range(0, total_files, chunk_size):
        chunk_files = npy_files[i : i + chunk_size]
        # 读取并拼接
        segments = []
        for f in chunk_files:
            arr = np.load(f)   # 每个文件 shape: (256,)
            segments.append(arr)
        # 拼成一个大数组 shape: (200*256,)
        big_data = np.concatenate(segments, axis=0)

        # 4) 添加噪声 (ARW + BI + QN)
        noisy_data = add_gyro_noise(big_data, fs=fs,
                                    arw_level=arw_level,
                                    bi_level=bi_level,
                                    qn_level=qn_level,
                                    seed=None)  # seed 可设定固定值确保复现

        # 5) 再切回 200 份，每份 (256,)
        splitted = np.split(noisy_data, chunk_size)

        # 6) 依次保存到新文件夹
        for j, seg_data in enumerate(splitted):
            original_file = chunk_files[j]   # Path 对象
            out_file = output_path / original_file.name
            np.save(out_file, seg_data)
            print(f"Saved: {out_file.name}")

    print("All done.")

if __name__ == "__main__":
    main()
