from multiprocessing import Process, Pipe
from telnetlib import IP
from copy import deepcopy
import IPython       

import numpy as np

from . import VecEnv, CloudpickleWrapper


def worker(remote, parent_remote, env_fn_wrapper):
    parent_remote.close()
    env = env_fn_wrapper.x()
    try:
        while True:
            cmd, data = remote.recv()
            
            if cmd == 'step':
                ob, reward, done, info = env.step(data)
                remote.send((ob, reward, done, info))
            elif cmd == 'reset':
                # ob = env.reset()
                ob = env.reset(data)
                remote.send(ob)
            elif cmd == 'reset_mdp':
                ob = env.reset_mdp()
                remote.send(ob)
            elif cmd == 'render':
                remote.send(env.render(mode='rgb_array'))
            elif cmd == 'close':
                remote.close()
                break
            elif cmd == 'get_spaces':
                remote.send((env.observation_space, env.action_space))
            elif cmd == 'get_task':
                remote.send(env.get_task())
            elif cmd == 'get_cluster':
                remote.send(env.get_cluster())
            elif cmd == 'sample_cluster_task':
                remote.send(env.sample_cluster_task(data[0], data[1]))
            elif cmd == 'task_dim':
                remote.send(env.task_dim)
            elif cmd == 'get_belief':
                remote.send(env.get_belief())
            elif cmd == 'belief_dim':
                remote.send(env.belief_dim)
            elif cmd == 'reset_task':
                env.unwrapped.reset_task(data)
            else:
                # try to get the attribute directly
                remote.send(getattr(env.unwrapped, cmd))
    except KeyboardInterrupt:
        print('SubprocVecEnv worker: got KeyboardInterrupt')
    finally:
        env.close()


class SubprocVecEnv(VecEnv):
    """
    VecEnv that runs multiple envs in parallel in subproceses and communicates with them via pipes.
    Recommended to use when num_envs > 1 and step() can be a bottleneck.
    """

    def __init__(self, env_fns):
        """
        Arguments:

        env_fns: iterable of callables -  functions that create envs to run in subprocesses. Need to be cloud-pickleable
        """
        self.waiting = False
        self.closed = False
        nenvs = len(env_fns)
        self.remotes, self.work_remotes = zip(*[Pipe() for _ in range(nenvs)])
        self.ps = [Process(target=worker, args=(work_remote, remote, CloudpickleWrapper(env_fn)))
                   for (work_remote, remote, env_fn) in zip(self.work_remotes, self.remotes, env_fns)]
        for p in self.ps:
            p.daemon = True  # if the main process crashes, we should not cause things to hang
            p.start()
        for remote in self.work_remotes:
            remote.close()

        self.remotes[0].send(('get_spaces', None))
        observation_space, action_space = self.remotes[0].recv()
        self.viewer = None
        VecEnv.__init__(self, len(env_fns), observation_space, action_space)

    def step_async(self, actions):
        self._assert_not_closed()
        for remote, action in zip(self.remotes, actions):
            remote.send(('step', action))
        self.waiting = True

    def step_wait(self):
        self._assert_not_closed()
        results = [remote.recv() for remote in self.remotes]
        self.waiting = False
        obs, rews, dones, infos = zip(*results)
        return np.stack(obs), np.stack(rews), np.stack(dones), infos

    def reset(self, task=None):
        self._assert_not_closed()
        if task is not None:
            for remote, t in zip(self.remotes, task):
                remote.send(('reset', t))
        else:
            for remote in self.remotes:
                remote.send(('reset', None))
        self.waiting = True
        
        self._assert_not_closed()
        results = np.stack([remote.recv() for remote in self.remotes])
        self.waiting = False
        return results

    def close_extras(self):
        self.closed = True
        if self.waiting:
            for remote in self.remotes:
                remote.recv()
        for remote in self.remotes:
            remote.send(('close', None))
        for p in self.ps:
            p.join()

    def get_images(self):
        self._assert_not_closed()
        for pipe in self.remotes:
            pipe.send(('render', None))
        imgs = [pipe.recv() for pipe in self.remotes]
        return imgs

    def _assert_not_closed(self):
        assert not self.closed, "Trying to operate on a SubprocVecEnv after calling close()"

    def get_env_attr(self, attr):
        self.remotes[0].send((attr, None))
        return self.remotes[0].recv()

    def get_task(self):
        self._assert_not_closed()
        for remote in self.remotes:
            remote.send(('get_task', None))
        return np.stack([remote.recv() for remote in self.remotes])
    
    def get_cluster(self):
        self._assert_not_closed()
        for remote in self.remotes:
            remote.send(('get_cluster', None))
        return np.stack([remote.recv() for remote in self.remotes])
    
    def get_belief(self):
        self._assert_not_closed()
        for remote in self.remotes:
            remote.send(('get_belief', None))
        return np.stack([remote.recv() for remote in self.remotes])

    def sample_cluster_task(self, n_tasks=10, seed=123):
        self._assert_not_closed()
        self.remotes[0].send(('sample_cluster_task', (n_tasks, seed)))
        return self.remotes[0].recv()