from contextlib import contextmanager
import os
from typing import TypeVar, Generator
import bbs.bimodal.plot as plot
from bbs.bimodal.experiment import BatchUtil, Epsilon
from bbs.bimodal.search import bimodal
from bbs.stats import plot_distribution


def epsilon():
    batch = Epsilon.run()
    df_exp = BatchUtil.experiments_df(batch)
    df_ks = BatchUtil.ks_test_df(batch)
    plot.Epsilon.show(df_exp, df_ks, write_html_name="1000")


T = TypeVar("T")


@contextmanager
def yield_self(value: T) -> Generator[T, None, None]:
    print(f"🏋️Generating:\t{value}")
    yield value
    print(f"✅Saved:\t{value}")


def generate_figures():
    """
    Save figures for paper.

    figures/bimodal/latex/
      line_chart.pdf
      table.txt
    """
    path = "./figures/bimodal/latex"
    os.makedirs(path, exist_ok=True)

    batch = Epsilon.run()
    df_exp = BatchUtil.experiments_df(batch)
    df_ks = BatchUtil.ks_test_df(batch)

    # line chart
    with yield_self(os.path.join(path, "line_chart.pdf")) as filename:
        Epsilon.mline_chart(df_exp, pdf_name=filename)

    # latex table
    with yield_self(os.path.join(path, "table.txt")) as filename:
        table = BatchUtil.latex_aggregate_table(df_ks)
        with open(filename, "w") as file:
            file.write(table)


def plot_dist():
    MU1 = 0
    STD_DEV1 = 1000

    MU2 = 4000
    STD_DEV2 = 1000

    WEIGHT1 = 0.5

    rv = bimodal(
        mu1=MU1, std_dev1=STD_DEV1, mu2=MU2, std_dev2=STD_DEV2, weight1=WEIGHT1
    )
    plot_distribution(rv)


def run():
    generate_figures()
