from envs.colour_grid_world import ColourGridWorld
from algorithms.q_learning import Q_Learning
import matplotlib.pyplot as plt 
from model_checking import pctl
from model_checking.helpers import construct_product_state_space, init_model_checker, init_dynamics_model
import numpy as np
import argparse
import sys
import importlib.util
import time
import jax 
import tensorflow as tf
 
"""task Q learning parameters"""
task_learning_rate = 0.1
task_discount_factor = 0.95
task_exploration_type = "boltzmann"
task_exploration_parameter = 0.05

"""safe Q learning parameters"""
safe_learning_rate = 0.1
safe_discount_factor = 0.95
safe_exploration_type = "boltzmann"
safe_exploration_parameter = 0.01

"""training parameters"""
property_path = "./properties/property_1.py"
steps = 100
counter_factual_experiences = True
cost_coefficient = 10.0

"""shielding parameters"""
model_checking_type = "exact"
approximate_model = False
shielding_type = "task_prod"
num_samples = 512
satisfaction_probability = 0.9
safe_policy_mode = "explore"
prior_type = "none"

"""environment parameters"""
random_action_probability = 0.05
episode_length = 1000

"""misc"""
device_type = "cpu"
seed = 0
logdir = "./logdir/"


def get_string_args():
    if approximate_model:
        template = "Property {}, Model Checking Type {}, Approximate Model {}, Satisfaction Probability {}, Prior Type {}, Shielding Type {}"
        return template.format(property_path.split('/')[-1],
                            model_checking_type,
                            approximate_model,
                            satisfaction_probability,
                            prior_type,
                            shielding_type)
    else:
        template = "Property {}, Model Checking Type {}, Approximate Model {}, Satisfaction Probability {}, Shielding Type {}"
        return template.format(property_path.split('/')[-1],
                            model_checking_type,
                            approximate_model,
                            satisfaction_probability,
                            shielding_type)
    
if __name__ == "__main__":

    parser = argparse.ArgumentParser()
    """task policy Q learning parameters"""
    parser.add_argument("--tp_lr", default=task_learning_rate)
    parser.add_argument("--tp_df", default=task_discount_factor)
    parser.add_argument("--tp_exploration", default=task_exploration_type)
    parser.add_argument("--tp_expl_parameter", default=task_exploration_parameter)

    """safe policy Q learning parameters"""
    parser.add_argument("--sp_lr", default=safe_learning_rate)
    parser.add_argument("--sp_df", default=safe_discount_factor)
    parser.add_argument("--sp_exploration", default=safe_exploration_type)
    parser.add_argument("--sp_expl_parameter", default=safe_exploration_parameter)

    """training parameters"""
    parser.add_argument("--property", default=property_path)
    parser.add_argument("--steps", default=steps)
    parser.add_argument("--cf", default=counter_factual_experiences)
    parser.add_argument("--cost_coeff", default=cost_coefficient)

    """shielding parameters"""
    parser.add_argument("--model_checking_type", default=model_checking_type)
    parser.add_argument("--approximate_model", default=approximate_model)
    parser.add_argument("--shielding_type", default=shielding_type)
    parser.add_argument("--num_samples", default=num_samples)
    parser.add_argument("--sat_prob", default=satisfaction_probability)
    parser.add_argument("--safe_policy_mode", default=safe_policy_mode)
    parser.add_argument("--prior_type", default=prior_type)

    """environment parameters"""
    parser.add_argument("--random_action_probability", default=random_action_probability)
    parser.add_argument("--episode_length", default=episode_length)

    """misc"""
    parser.add_argument("--device_type", default=device_type)
    parser.add_argument("--seed", default=seed)
    parser.add_argument("--logdir", default=logdir)
    args = parser.parse_args()

    try:
        """task Q learning parameters"""
        task_learning_rate = float(args.tp_lr)
        task_discount_factor = float(args.tp_df)
        task_exploration_type = str(args.tp_exploration)
        task_exploration_parameter = float(args.tp_expl_parameter)

        """safe Q learning parameters"""
        safe_learning_rate = float(args.sp_lr)
        safe_discount_factor = float(args.sp_df)
        safe_exploration_type = str(args.sp_exploration)
        safe_exploration_parameter = float(args.sp_expl_parameter)

        """training parameters"""
        property_path = str(args.property)
        steps = int(args.steps)
        counter_factual_experiences = bool(args.cf)
        cost_coefficient = float(args.cost_coeff)

        """shielding parameters"""
        model_checking_type = str(args.model_checking_type)
        approximate_model = bool(args.approximate_model)
        shielding_type = str(args.shielding_type)
        num_samples = int(args.num_samples)
        satisfaction_probability = float(args.sat_prob)
        safe_policy_mode = str(args.safe_policy_mode)
        prior_type = str(args.prior_type)

        """environment parameters"""
        random_action_probability = float(args.random_action_probability)
        episode_length = int(args.episode_length)

        """misc"""
        device_type = str(args.device_type)
        seed = int(args.seed)
        logdir = str(args.logdir)
    except:
        raise TypeError

    string_args = get_string_args()
    print(string_args)

    # setup JAX device
    device = jax.devices(device_type)[0]
    cpu_device = jax.devices('cpu')[0]

    key = jax.random.PRNGKey(seed)
    np.random.seed(seed)

    # setup tensorboard
    summary_writer = tf.summary.create_file_writer(logdir)

    # setup environment
    env = ColourGridWorld(random_action_probability=random_action_probability, episode_length=episode_length)
    n_states = env.n_states
    n_actions = env.n_actions

    # load cost function, automaton and pctl and lt properties
    spec=importlib.util.spec_from_file_location("property", property_path)
 
    # creates a new module based on spec
    properties = importlib.util.module_from_spec(spec)
    
    # executes the module in its own namespace
    # when a module is imported or reloaded.
    spec.loader.exec_module(properties)

    # get the cost funtion and corresponding automaton
    cost_function = properties.cost_function
    automaton = properties.automaton

    # define the product state space
    n_automaton_states = len(automaton.states)
    prod_state_space = construct_product_state_space(n_states, len(automaton.states))
    n_prod_states = np.prod(prod_state_space.shape)

    # setup q learning agents
    task_agent = Q_Learning(n_states, n_actions, alpha=task_learning_rate, discount=task_discount_factor, exploration=task_exploration_type, expl_parameter=task_exploration_parameter)
    safe_agent = Q_Learning(n_prod_states, n_actions, alpha=safe_learning_rate, discount=safe_discount_factor, exploration=safe_exploration_type, expl_parameter=safe_exploration_parameter)

    # returns a model checker with model_checking_type that checks the corresponding property specified in the properties module
    model_checker = init_model_checker(env, properties, device, model_checking_type=model_checking_type, satisfaction_probability=satisfaction_probability, shielding_type=shielding_type)

    # kwargs to pass the the model checker
    mc_kwargs = {'num_samples': num_samples}

    # returns a dynamics model for the state space being checked if approximate then the model is learned from experience
    dynamics_model = init_dynamics_model(env, automaton, approximate_model=approximate_model, shielding_type=shielding_type, prior_type=prior_type)

    # if the dynamics model is approximate and on the product state space then it is updated with counter factual experiences
    update_dynamics_with_cf = (approximate_model and (shielding_type in ['action_cond_safe', 'task_prod']))

    # training loop
    with summary_writer.as_default():
        for episode in range(steps):

            episode_reward = 0.0
            episode_cost = 0.0
            episode_overrides = 0.0

            terminated = False
            state, info = env.reset()
            labels = info["labels"]
            
            cost_function.reset()
            _, automaton_state = cost_function.step(labels)

            start_time = time.time()

            while not terminated:

                prod_state = prod_state_space[automaton_state, state]

                if shielding_type == "task_prod":
                    action = task_agent.step(state)
                    # task agent is defined on the state space we need to map it to the product state space
                    key, sat = model_checker.check(key, np.repeat(task_agent.get_policy(), n_automaton_states, axis=0), dynamics_model.get_model(), prod_state, action, **mc_kwargs)
                elif shielding_type == "action_cond_safe":
                    action = task_agent.step(state)
                    key, sat = model_checker.check(key, safe_agent.get_policy(), dynamics_model.get_model(), prod_state, action, **mc_kwargs)
                else:
                    assert NotImplementedError

                if not sat:
                    action = safe_agent.step(prod_state, mode=safe_policy_mode)
                    episode_overrides += 1.0
                else:
                    action = action if action is not None else task_agent.step(state)
                
                # play shielded action
                next_state, reward, terminated, info = env.step(action)
                labels = info["labels"]

                cost, next_automaton_state = cost_function.step(labels)
            
                episode_cost += cost
                episode_reward += reward
                
                # update the task agent
                tup = (state, action, reward, next_state, terminated)
                task_agent.update(tup)

                # update the safe agent
                if counter_factual_experiences or update_dynamics_with_cf:
                    # run counter factual experiences and update the safe agent and/or dynamics model
                    # simulate state, action, next_state, from all automaton states
                    for counter_fac_automaton_state in automaton.states:
                        counter_fac_prod_state = prod_state_space[counter_fac_automaton_state, state]
                        counter_fac_next_automaton_state = automaton.transition(counter_fac_automaton_state, labels)
                        counter_fac_next_prod_state = prod_state_space[counter_fac_next_automaton_state, next_state]
                        if counter_factual_experiences:
                            counter_fac_cost = float(counter_fac_next_automaton_state in automaton.accepting)
                            penalty = counter_fac_cost * cost_coefficient
                            counter_fac_terminated = terminated
                            tup = (counter_fac_prod_state, action, -penalty, counter_fac_next_prod_state, counter_fac_terminated)
                            safe_agent.update(tup)
                        if update_dynamics_with_cf:
                            dynamics_model.update(counter_fac_next_prod_state, counter_fac_prod_state, action)
                else:
                    next_prod_state = prod_state_space[next_automaton_state, next_state]
                    penalty = cost * cost_coefficient
                    simulated_terminated = terminated
                    tup = (prod_state, action, -penalty, next_prod_state, simulated_terminated)
                    safe_agent.update(tup)

                # update the dynamics model
                if not update_dynamics_with_cf:
                    dynamics_model.update(next_state, state, action) 
                
                state = next_state
                automaton_state = next_automaton_state

                if terminated:
                    tf.summary.scalar('reward', episode_reward, step=episode)
                    tf.summary.scalar('cost', episode_cost, step=episode)
                    tf.summary.scalar('overrides', episode_overrides, step=episode)
                    template = "Episode {} completed in {} second. Reward {}, Cost {}, Overrides {}, FPS {}"
                    current_time = time.time()
                    print(template.format(episode+1,
                                          current_time-start_time,
                                          episode_reward,
                                          episode_cost,
                                          episode_overrides,
                                          episode_length/(current_time-start_time)))

    

