# ML4OPF

ML4OPF is a Python package for developing machine learning proxy models for optimal power flow. It is built on top of PyTorch and PyTorch Lightning, and is designed to be modular and extensible. The main components are:

- [`formulations`](ml4opf/formulations): The main interface to the OPF formulations [ACOPF](ml4opf/formulations/ac), [DCOPF](ml4opf/formulations/dc), and [Economic Dispatch](ml4opf/formulations/ed).

  - Each OPF formulation has three main component classes: `OPFProblem`, `OPFViolation`, and `OPFModel`. The `OPFProblem` class loads and parses data from disk, the `OPFViolation` class calculates constraints residuals, incidence matrices, objective value, etc., and the `OPFModel` class is an abstract base class for proxy models.

<!-- - [`functional`](ml4opf/functional): The functional interface underlying the object-oriented interface in `formulations`, which makes less assumptions (i.e., all problem data may vary but the user is responsible for keeping track of it). -->

- [`loss_functions`](ml4opf/loss_functions): Various loss functions including [LDFLoss](ml4opf/loss_functions/ldf.py) and the self-supervised [ObjectiveLoss](ml4opf/loss_functions/objective.py).

- [`layers`](ml4opf/layers): Various feasibility recovery layers including [BoundRepair](ml4opf/layers/bound_repair.py) and [HyperSimplexRepair](ml4opf/layers/hypersimplex_repair.py).

- [`models`](ml4opf/models): Various proxy models including [BasicNeuralNet](ml4opf/models/basic_nn), [LDFNeuralNet](ml4opf/models/ldf_nn), and [E2ELR](ml4opf/models/e2elr).

- [`parsers`](ml4opf/parsers): Parsers for data generated by PGLearn.jl

- [`viz`](ml4opf/viz): Visualization helpers for plots and tables.


## Installation

For development, the recommended installation method is using Conda environment files provided at [environment.yml](environment.yml) and [environment_cuda.yml](environment_cuda.yml):
```bash
cd ML4OPF                                  # cd into the repo
conda env create -f environment.yml        # create the environment
conda activate ml4opf                      # activate the environment
pip install -e ".[all]"                    # install ML4OPF
```

## Usage


### Training a `BasicNeuralNet` for ACOPF
```python
import torch

# load data
from ml4opf import ACProblem

data_path = ...

problem = ACProblem(data_path)

# make a basic neural network model
from ml4opf.models.basic_nn import ACBasicNeuralNet # requires pytorch-lightning

config = {
    "optimizer": "adam",
    "learning_rate": 1e-3,
    "loss": "mse",
    "hidden_sizes": [500,300,500],
    "activation": "sigmoid",
    "boundrepair": "none" # optionally clamp outputs to bounds (choices: "sigmoid", "relu", "clamp")
}

model = ACBasicNeuralNet(config, problem)

model.train(trainer_kwargs={'max_epochs': 100, 'accelerator': 'auto'}) # pass args to the PyTorch Lightning Trainer

evals = model.evaluate_model()

from ml4opf.viz import make_stats_df
print(make_stats_df(evals))

model.save_checkpoint("./basic_300bus") # creates a folder called "basic_300bus" with a file "trainer.ckpt" in it.
```

### Manually Loading Data
```python
import torch

from ml4opf import ACProblem

data_path = ...

# parse HDF5/JSON
problem = ACProblem(data_path)

# get train/test set:
train_data = problem.train_data
test_data = problem.test_data

train_data['input/pd'].shape # torch.Size([52863, 201])
test_data['input/pd'].shape # torch.Size([5000, 201])

# if needed, convert the HDF5 data to a tree dictionary instead of a flat dictionary:
from ml4opf.parsers import PGLearnParser
h5_tree = PGLearnParser.make_tree(train_data) # this tree structure should
                                              # exactly mimic the
                                              # structure of the HDF5 file.
h5_tree['input']['pd'].shape # torch.Size([52863, 201])
```
