# PGLearn
Instance generator for various OPF problems.

## Installation instructions


```bash
$ cd PGLearn
$ julia --project=.
julia> using PGLearn
```

If you are modifying the source code, it is recommened to use the package [`Revise.jl`](https://github.com/timholy/Revise.jl)
so that you can use the changes without having to start Julia.
Make sure you load `Revise` before loading `PGLearn` in your julia session.
```julia
using Revise
using PGLearn
```

### Using HSL solvers (ma27, ma57)

Please refer to [`Ipopt.jl` installation instructions](https://github.com/jump-dev/Ipopt.jl?tab=readme-ov-file#linear-solvers)
    for how to install non-default linear solvers, e.g., HSL, Pardiso, etc...
Note that a specific license may be required.

To use HSL linear solvers when solving OPF instances, set the parameter "linear_solver" to "ma27" or "ma57" in the config file.
The recommended solver for Ipopt is `ma27`.
```toml
solver.name = "Ipopt"
solver.attributes.linear_solver = "ma27"
```

## Quick start

### Building and solving an OPF problem

```julia
using PGLib, PowerModels
using PGLearn
using Ipopt

# Load data from PGLib case
pm_network = PowerModels.make_basic_network(pglib("14_ieee"))
data = PGLearn.OPFData(pm_network)

# Build OPF optimization model
# To switch formulations, replace ACOPF with, e.g., DCOPF or SOCOPF
opf = PGLearn.build_opf(PGLearn.ACOPF, data, Ipopt.Optimizer)

# Solve and extract result
PGLearn.solve!(opf)
res = PGLearn.extract_result(opf)
```


### Generating random OPF instances

```julia
using Random 
using PGLib, PowerModels
using PGLearn

# Load data from PGLib case
pm_network = PowerModels.make_basic_network(pglib("14_ieee"))
data = PGLearn.OPFData(pm_network)

# Load scaler using global scaling + uncorrelated LogNormal noise
config = Dict(
    "load" => Dict(
        "noise_type" => "ScaledUniform",
        "l" => 0.8,
        "u" => 1.2,
        "sigma" => 0.10,       
    )
)
opf_sampler = SimpleOPFSampler(data, config)

# Generate a new instance
rng = MersenneTwister(42)
new_data = rand(rng, opf_sampler)

data.pd[1] # old 
0.217

new_data.pd[1]  # new
0.21480423013573954  # ⚠️ exact value depends on Julia version)
```

To generate multiple instances, run the above code in a loop
```julia
dataset = [
    PGLearn.rand(rng, opf_sampler)
    for i in 1:100
]
```

## Generating datasets

A script for generating multiple ACOPF instances is given in [`exp/sampler.jl`](exp/sampler.jl).

It is called from the command-line as follows:
```bash
julia --project=. exp/sampler.jl <path/to/config.toml> <seed_min> <seed_max>
```
where
* `<path/to/config.toml>` is a path to a valid configuration file in TOML format (see [`exp/config.toml`](exp/config.toml) for an example)
* `<seed_min>` and `<seed_max>` are minimum and maximum values for the random seed. Must be integer values.
    The script will generate instances for values `smin, smin+1, ..., smax-1, smax`.
