from __future__ import annotations

from dataclasses import dataclass

from softmatcha.modules import SoftSet
from softmatcha.typing import Vector


@dataclass
class Pattern:
    """Pattern class stores the lexical and semantic pattern information.

    tokens (list[set[int]]): A sequence of token sets.
    embeddings (list[SoftSet]): D-dimensional base vectors of token sets.
    """

    tokens: list[set[int]]
    embeddings: list[SoftSet]

    def __len__(self) -> int:
        return len(self.tokens)

    @classmethod
    def build(cls, tokens: list[set[int]], embeddings: list[Vector]) -> Pattern:
        """Build a pattern class.

        Args:
            tokens (list[set[int]]): A sequence of token sets.
            embeddings (list[Vector]): D-dimensional embedding token vectors of each set.

        Returns:
            Pattern: This class.
        """
        return cls(tokens, [SoftSet.construct(embeds) for embeds in embeddings])
