#ifndef EMP_UTILS_FIELD_FP59_H__
#define EMP_UTILS_FIELD_FP59_H__

#include "emp-tool/emp-tool.h"
#include "src/utils/fields/utils.h"
#include <cstdint>

using namespace emp;

// finite field Fp, p = 2^59 - 2^28 + 1
class FP59 {
public:
  static uint64_t PR_bit_len;
  static uint64_t PR;
  static block DoublePR;
  static uint64_t Generator;
  static uint64_t Low59bMask;
  static block DoubleLow59bMask;

  uint64_t val;

  FP59() { val = 0; }

  FP59(uint64_t input, bool mod_it = true) {
    if (mod_it)
      val = mod(input);
    else
      val = input;
  }

  void operator=(const uint64_t rhs) { this->val = mod(rhs); }

  void operator=(const FP59 rhs) { this->val = rhs.val; }

  bool operator==(const uint64_t rhs) const {
    return (this->val == mod(rhs)) ? true : false;
  }

  bool operator==(const FP59 rhs) const {
    return (this->val == rhs.val) ? true : false;
  }

  bool operator!=(const uint64_t rhs) const {
    return (this->val == mod(rhs)) ? false : true;
  }

  bool operator!=(const FP59 rhs) const {
    return (this->val == rhs.val) ? false : true;
  }

  FP59 operator+(const FP59 rhs) const {
    FP59 res(*this);
    res.val = add_mod(res.val, rhs.val);
    return res;
  }

  FP59 operator-(const FP59 rhs) const {
    FP59 res(*this);
    res.val = add_mod(res.val, PR - rhs.val);
    return res;
  }

  FP59 operator*(const FP59 rhs) const {
    FP59 res(*this);
    res.val = mult_mod(res.val, rhs.val);
    return res;
  }

  FP59 negate() const {
    FP59 res(*this);
    if(res.val != 0)
      res.val = PR - res.val;
    return res;
  }

  FP59 inv() const {
    FP59 res(*this);
    res.val = mod_inv(res.val);
    return res;
  }

  uint64_t mod(const uint64_t x) const {
    uint64_t i = x >> 59;
    i = (i << 28) - i + (x & Low59bMask);
    return (i >= PR) ? i - PR : i;
  }

  // c = val + a mod PR
  uint64_t add_mod(const uint64_t a, const uint64_t b) const {
    uint64_t res = a + b;
    return (res >= PR) ? (res - PR) : res;
  }

  // c = a * b mod PR
  uint64_t mult_mod(const uint64_t a, const uint64_t b) const {
    uint64_t c = 0;	
    uint64_t e = mul64(a, b, (uint64_t*)&c);
    uint64_t i = (c<<5) + (e>>59);

    uint64_t j = i >> 31;
    j = (j << 28) - j + ((i << 28) & Low59bMask);
    j = (j >= PR) ? j-PR : j;

    i = j + (e & Low59bMask) + PR - i;
    i = (i >= PR) ? (i - PR) : i;
    return (i >= PR) ? (i - PR) : i;
  }

  // from SCL
  uint64_t mod_inv(const uint64_t a) const {
    if(a == 0)
      error("zero know invertible");

    int64_t k = 0;
    int64_t new_k = 1;
    int64_t r = (int64_t)PR;
    int64_t new_r = (int64_t)a;

    while(new_r != 0) {
      int64_t q = r / new_r;
      // assign(k, new_k, q)
      int64_t tmp = new_k;
      new_k = k - q * tmp;
      k = tmp;
      // assign(r, new_r, q)
      tmp = new_r;
      new_r = r - q * tmp;
      r = tmp;
    }

    if(k < 0) k = k + (int64_t)PR;
    return (uint64_t)k;
  }

  static std::size_t size() {
    return sizeof(uint64_t);
  }
};

uint64_t FP59::PR_bit_len = 59;
uint64_t FP59::PR = 576460752034988033;
block FP59::DoublePR = makeBlock(576460752034988033, 576460752034988033);
uint64_t FP59::Generator = 3;
uint64_t FP59::Low59bMask = (1ULL << 59) - 1;
block FP59::DoubleLow59bMask = makeBlock((1ULL << 59) - 1, (1ULL << 59) - 1);

#endif // EMP_UTILS_FIELD_FP59_H__
