import copy
import math
import pickle
import numpy as np
import torch
import torch.nn as nn
from .utils.layer import BasicBlock
from .motion_encoder import *


class WavEncoder(nn.Module):
    def __init__(self, out_dim, audio_in=1):
        super().__init__()
        self.out_dim = out_dim
        self.feat_extractor = nn.Sequential(
            BasicBlock(audio_in, out_dim // 4, 15, 5, first_dilation=1600, downsample=True),
            BasicBlock(out_dim // 4, out_dim // 4, 15, 6, first_dilation=0, downsample=True),
            BasicBlock(out_dim // 4, out_dim // 4, 15, 1, first_dilation=7, ),
            BasicBlock(out_dim // 4, out_dim // 2, 15, 6, first_dilation=0, downsample=True),
            BasicBlock(out_dim // 2, out_dim // 2, 15, 1, first_dilation=7),
            BasicBlock(out_dim // 2, out_dim, 15, 3, first_dilation=0, downsample=True),
        )

    def forward(self, wav_data):
        if wav_data.dim() == 2:
            wav_data = wav_data.unsqueeze(1)
        else:
            wav_data = wav_data.transpose(1, 2)
        out = self.feat_extractor(wav_data)
        return out.transpose(1, 2)


class MLP(nn.Module):
    def __init__(self, in_dim, hidden_size, out_dim):
        super().__init__()
        self.mlp = nn.Sequential(
            nn.Linear(in_dim, hidden_size),
            nn.LeakyReLU(0.2, True),
            nn.Linear(hidden_size, out_dim)
        )

    def forward(self, inputs):
        out = self.mlp(inputs)
        return out


class PeriodicPositionalEncoding(nn.Module):
    def __init__(self, d_model, dropout=0.1, period=15, max_seq_len=60):
        super(PeriodicPositionalEncoding, self).__init__()
        self.dropout = nn.Dropout(p=dropout)
        pe = torch.zeros(period, d_model)
        position = torch.arange(0, period, dtype=torch.float).unsqueeze(1)
        div_term = torch.exp(torch.arange(0, d_model, 2).float() * (-math.log(10000.0) / d_model))
        pe[:, 0::2] = torch.sin(position * div_term)
        pe[:, 1::2] = torch.cos(position * div_term)
        pe = pe.unsqueeze(0)  # (1, period, d_model)
        repeat_num = (max_seq_len // period) + 1
        pe = pe.repeat(1, repeat_num, 1)  # (1, repeat_num, period, d_model)
        self.register_buffer('pe', pe)

    def forward(self, x):
        # print(self.pe.shape, x.shape)
        x = x + self.pe[:, :x.size(1), :]
        return self.dropout(x)


class StyleCls(nn.Module):
    def __init__(self, args):
        super(StyleCls, self).__init__()
        self.args = args
        with open(f"{args.data_path}weights/vocab.pkl", 'rb') as f:
            self.lang_model = pickle.load(f)
            pre_trained_embedding = self.lang_model.word_embedding_weights
        self.text_pre_encoder_face = nn.Embedding.from_pretrained(torch.FloatTensor(pre_trained_embedding),
                                                                  freeze=args.t_fix_pre)
        self.text_encoder_face = nn.Linear(300, args.audio_f)
        self.text_encoder_face = nn.Linear(300, args.audio_f)
        self.text_pre_encoder_body = nn.Embedding.from_pretrained(torch.FloatTensor(pre_trained_embedding),
                                                                  freeze=args.t_fix_pre)
        self.text_encoder_body = nn.Linear(300, args.audio_f)
        self.text_encoder_body = nn.Linear(300, args.audio_f)

        self.audio_pre_encoder_face = WavEncoder(args.audio_f, audio_in=2)
        self.audio_pre_encoder_body = WavEncoder(args.audio_f, audio_in=2)

        self.at_attn_face = nn.Linear(args.audio_f * 2, args.audio_f * 2)
        self.at_attn_body = nn.Linear(args.audio_f * 2, args.audio_f * 2)

        args_top = copy.deepcopy(self.args)
        args_top.vae_layer = 3
        args_top.vae_length = args.motion_f
        args_top.vae_test_dim = args.pose_dims + 3 + 4
        self.motion_encoder = VQEncoderV6(args_top)  # masked motion to latent bs t 333 to bs t 256

        # face decoder
        self.feature2face = nn.Linear(args.audio_f * 2, args.hidden_size)
        self.face2latent = nn.Linear(args.hidden_size, args.vae_codebook_size)
        self.transformer_de_layer = nn.TransformerDecoderLayer(
            d_model=self.args.hidden_size,
            nhead=4,
            dim_feedforward=self.args.hidden_size * 2,
            batch_first=True
        )
        self.transformer_de_layer_style = nn.TransformerDecoderLayer(
            d_model=4*args.vae_codebook_size,
            nhead=4,
            dim_feedforward=args.vae_codebook_size * 8,
            batch_first=True
        )
        self.face_decoder = nn.TransformerDecoder(self.transformer_de_layer, num_layers=4)
        self.position_embeddings = PeriodicPositionalEncoding(self.args.hidden_size, period=self.args.pose_length,
                                                              max_seq_len=self.args.pose_length)

        # motion decoder
        self.transformer_en_layer = nn.TransformerEncoderLayer(
            d_model=self.args.hidden_size,
            nhead=4,
            dim_feedforward=self.args.hidden_size * 2,
            batch_first=True
        )
        self.motion_self_encoder = nn.TransformerEncoder(self.transformer_en_layer, num_layers=1)
        self.audio_feature2motion = nn.Linear(args.audio_f, args.hidden_size)
        self.feature2motion = nn.Linear(args.motion_f, args.hidden_size)

        self.bodyhints_face = MLP(args.motion_f, args.hidden_size, args.motion_f)
        self.bodyhints_body = MLP(args.motion_f, args.hidden_size, args.motion_f)
        self.motion2latent_upper = MLP(args.hidden_size, args.hidden_size, self.args.hidden_size)
        self.motion2latent_hands = MLP(args.hidden_size, args.hidden_size, self.args.hidden_size)
        self.motion2latent_lower = MLP(args.hidden_size, args.hidden_size, self.args.hidden_size)
        self.wordhints_decoder = nn.TransformerDecoder(self.transformer_de_layer, num_layers=8)

        self.upper_decoder = nn.TransformerDecoder(self.transformer_de_layer, num_layers=1)
        self.hands_decoder = nn.TransformerDecoder(self.transformer_de_layer, num_layers=1)
        self.lower_decoder = nn.TransformerDecoder(self.transformer_de_layer, num_layers=1)

        self.style_decoder = nn.TransformerDecoder(self.transformer_de_layer_style, num_layers=1)

        '''
        self.face_classifier = [MLP(self.args.vae_codebook_size, args.hidden_size, self.args.vae_codebook_size).cuda(),
                                MLP(self.args.vae_codebook_size, args.hidden_size, self.args.vae_codebook_size).cuda(),
                                MLP(self.args.vae_codebook_size, args.hidden_size, self.args.vae_codebook_size).cuda(),
                                MLP(self.args.vae_codebook_size, args.hidden_size, self.args.vae_codebook_size).cuda()]
        self.upper_classifier = [MLP(self.args.vae_codebook_size, args.hidden_size, self.args.vae_codebook_size).cuda(),
                                MLP(self.args.vae_codebook_size, args.hidden_size, self.args.vae_codebook_size).cuda(),
                                MLP(self.args.vae_codebook_size, args.hidden_size, self.args.vae_codebook_size).cuda(),
                                MLP(self.args.vae_codebook_size, args.hidden_size, self.args.vae_codebook_size).cuda()]
        self.hands_classifier = [MLP(self.args.vae_codebook_size, args.hidden_size, self.args.vae_codebook_size).cuda(),
                                MLP(self.args.vae_codebook_size, args.hidden_size, self.args.vae_codebook_size).cuda(),
                                MLP(self.args.vae_codebook_size, args.hidden_size, self.args.vae_codebook_size).cuda(),
                                MLP(self.args.vae_codebook_size, args.hidden_size, self.args.vae_codebook_size).cuda()]
        self.lower_classifier = [MLP(self.args.vae_codebook_size, args.hidden_size, self.args.vae_codebook_size).cuda(),
                                MLP(self.args.vae_codebook_size, args.hidden_size, self.args.vae_codebook_size).cuda(),
                                MLP(self.args.vae_codebook_size, args.hidden_size, self.args.vae_codebook_size).cuda(),
                                MLP(self.args.vae_codebook_size, args.hidden_size, self.args.vae_codebook_size).cuda()]
        '''

        self.face_classifier = MLP(self.args.vae_codebook_size, args.hidden_size, self.args.vae_codebook_size)
        self.upper_classifier = MLP(self.args.vae_codebook_size, args.hidden_size, self.args.vae_codebook_size)
        self.hands_classifier = MLP(self.args.vae_codebook_size, args.hidden_size, self.args.vae_codebook_size)
        self.lower_classifier = MLP(self.args.vae_codebook_size, args.hidden_size, self.args.vae_codebook_size)

        self.mask_embeddings = nn.Parameter(torch.zeros(1, 1, self.args.pose_dims + 3 + 4))
        self.motion_down_upper = nn.Linear(args.hidden_size, self.args.vae_codebook_size)
        self.motion_down_hands = nn.Linear(args.hidden_size, self.args.vae_codebook_size)
        self.motion_down_lower = nn.Linear(args.hidden_size, self.args.vae_codebook_size)
        self.motion_down_upper = nn.Linear(args.hidden_size, self.args.vae_codebook_size)
        self.motion_down_hands = nn.Linear(args.hidden_size, self.args.vae_codebook_size)
        self.motion_down_lower = nn.Linear(args.hidden_size, self.args.vae_codebook_size)

        self.scaler_face = [nn.Linear(self.args.vae_codebook_size, self.args.vae_codebook_size).cuda(),
                            nn.Linear(self.args.vae_codebook_size*2, self.args.vae_codebook_size).cuda(),
                            nn.Linear(self.args.vae_codebook_size*4, self.args.vae_codebook_size).cuda(),
                            nn.Linear(self.args.vae_codebook_size*8, self.args.vae_codebook_size).cuda()]
        self.scaler_hands = [nn.Linear(self.args.vae_codebook_size, self.args.vae_codebook_size).cuda(),
                            nn.Linear(self.args.vae_codebook_size*2, self.args.vae_codebook_size).cuda(),
                            nn.Linear(self.args.vae_codebook_size*4, self.args.vae_codebook_size).cuda(),
                            nn.Linear(self.args.vae_codebook_size*8, self.args.vae_codebook_size).cuda()]
        self.scaler_upper = [nn.Linear(self.args.vae_codebook_size, self.args.vae_codebook_size).cuda(),
                            nn.Linear(self.args.vae_codebook_size*2, self.args.vae_codebook_size).cuda(),
                            nn.Linear(self.args.vae_codebook_size*4, self.args.vae_codebook_size).cuda(),
                            nn.Linear(self.args.vae_codebook_size*8, self.args.vae_codebook_size).cuda()]
        self.scaler_lower = [nn.Linear(self.args.vae_codebook_size, self.args.vae_codebook_size).cuda(),
                            nn.Linear(self.args.vae_codebook_size*2, self.args.vae_codebook_size).cuda(),
                            nn.Linear(self.args.vae_codebook_size*4, self.args.vae_codebook_size).cuda(),
                            nn.Linear(self.args.vae_codebook_size*8, self.args.vae_codebook_size).cuda()]
        self.caption_encoder_index = nn.Linear(args.hidden_size, self.args.vae_codebook_size)
        self.caption_encoder_style = nn.Linear(args.hidden_size, self.args.vae_codebook_size)
        self.caption_encoder_styleloss = nn.Linear(4096, 4*self.args.vae_codebook_size)
        self.caption_encoder_body = nn.Linear(4096, args.hidden_size)
        self.caption_encoder_face = nn.Linear(4096, args.hidden_size)
        self.caption_decoder_style = nn.Linear(4*self.args.vae_codebook_size, 4096)
        self.leakyrelu = nn.LeakyReLU(0.2)

        self._reset_parameters()

        self.spearker_encoder_body = nn.Embedding(25, args.hidden_size)
        self.spearker_encoder_face = nn.Embedding(25, args.hidden_size)

    def _reset_parameters(self):
        nn.init.normal_(self.mask_embeddings, 0, self.args.hidden_size ** -0.5)

    def index_pred(self, cls, latent_list, caption_embedding):
        index_list = []

        prior_latent = latent_list[-1]
        prior_latent_re = prior_latent.repeat_interleave(2, dim=1)
        for i in range(len(latent_list)-1,-1,-1):
            if i == len(latent_list)-1:
                index_list.append(cls(prior_latent))
            else:
                prior_latent = prior_latent_re + latent_list[i]
                index_list.append(cls(prior_latent))
            prior_latent_re = prior_latent.repeat_interleave(2, dim=1)
        index_list.reverse()

        return index_list, prior_latent

    def forward(self, in_audio=None, in_word=None, in_caption=None, mask=None, is_test=None, in_motion=None, use_attentions=True,
                use_word=True, in_id=None):
        face_latent_list = []
        upper_latent_list = []
        lower_latent_list = []
        hands_latent_list = []
        cls_face = []
        cls_upper = []
        cls_lower = []
        cls_hands = []
        in_word_face = self.text_pre_encoder_face(in_word)
        in_word_face = self.text_encoder_face(in_word_face)
        in_word_body = self.text_pre_encoder_body(in_word)
        in_word_body = self.text_encoder_body(in_word_body)
        bs, t, c = in_word_face.shape
        in_audio_face = self.audio_pre_encoder_face(in_audio)
        in_audio_body = self.audio_pre_encoder_body(in_audio)
        if in_audio_face.shape[1] != in_motion.shape[1]:
            diff_length = in_motion.shape[1] - in_audio_face.shape[1]
            if diff_length < 0:
                in_audio_face = in_audio_face[:, :diff_length, :]
                in_audio_body = in_audio_body[:, :diff_length, :]
            else:
                in_audio_face = torch.cat((in_audio_face, in_audio_face[:, -diff_length:]), 1)
                in_audio_body = torch.cat((in_audio_body, in_audio_body[:, -diff_length:]), 1)

        if use_attentions:
            alpha_at_face = torch.cat([in_word_face, in_audio_face], dim=-1).reshape(bs, t, c * 2)
            alpha_at_face = self.at_attn_face(alpha_at_face).reshape(bs, t, c, 2)
            alpha_at_face = alpha_at_face.softmax(dim=-1)
            fusion_face = in_word_face * alpha_at_face[:, :, :, 1] + in_audio_face * alpha_at_face[:, :, :, 0]
            alpha_at_body = torch.cat([in_word_body, in_audio_body], dim=-1).reshape(bs, t, c * 2)
            alpha_at_body = self.at_attn_body(alpha_at_body).reshape(bs, t, c, 2)
            alpha_at_body = alpha_at_body.softmax(dim=-1)
            fusion_body = in_word_body * alpha_at_body[:, :, :, 1] + in_audio_body * alpha_at_body[:, :, :, 0]
        else:
            fusion_face = in_word_face + in_audio_face
            fusion_body = in_word_body + in_audio_body

        # use top 8 frames of in_motion ground truth as hint
        masked_embeddings = self.mask_embeddings.expand_as(in_motion)
        masked_motion = torch.where(mask == 1, masked_embeddings, in_motion)  # bs, t, 256
        body_hint = self.motion_encoder(masked_motion)  # bs t 256
        # speaker condition
        caption_style_embedding_face = self.caption_encoder_face(in_caption)#self.spearker_encoder_face(in_id).squeeze(2)
        caption_style_embedding_body = self.caption_encoder_body(in_caption)#self.spearker_encoder_body(in_id).squeeze(2)
        caption_style = self.caption_encoder_styleloss(in_caption)
        # decode face
        use_body_hints = True
        if use_body_hints:
            body_hint_face = self.bodyhints_face(body_hint)
            fusion_face = torch.cat([fusion_face, body_hint_face], dim=2)
        a2g_face = self.feature2face(fusion_face)
        face_embeddings = a2g_face
        face_embeddings = self.position_embeddings(face_embeddings)
        decoded_face = self.face_decoder(tgt=face_embeddings, memory=a2g_face)


        # motion spatial encoder
        body_hint_body = self.bodyhints_body(body_hint)
        motion_embeddings = self.feature2motion(body_hint_body)
        motion_embeddings = motion_embeddings
        motion_embeddings = self.position_embeddings(motion_embeddings)

        # bi-directional self-attention
        motion_refined_embeddings = self.motion_self_encoder(motion_embeddings)

        # audio to gesture cross-modal attention
        if use_word:
            a2g_motion = self.audio_feature2motion(fusion_body)
            motion_refined_embeddings_in = motion_refined_embeddings
            motion_refined_embeddings_in = self.position_embeddings(motion_refined_embeddings_in)
            word_hints = self.wordhints_decoder(tgt=motion_refined_embeddings_in, memory=a2g_motion)
            motion_refined_embeddings = motion_refined_embeddings + word_hints

        # feedforward
        upper_latent = self.motion2latent_upper(motion_refined_embeddings)
        hands_latent = self.motion2latent_hands(motion_refined_embeddings)
        lower_latent = self.motion2latent_lower(motion_refined_embeddings)

        upper_latent_in = upper_latent + caption_style_embedding_body
        upper_latent_in = self.position_embeddings(upper_latent_in)
        hands_latent_in = hands_latent + caption_style_embedding_body
        hands_latent_in = self.position_embeddings(hands_latent_in)
        lower_latent_in = lower_latent + caption_style_embedding_body
        lower_latent_in = self.position_embeddings(lower_latent_in)

        # transformer decoder
        motion_upper = self.upper_decoder(tgt=upper_latent_in, memory=hands_latent + lower_latent)
        motion_hands = self.hands_decoder(tgt=hands_latent_in, memory=upper_latent + lower_latent)
        motion_lower = self.lower_decoder(tgt=lower_latent_in, memory=upper_latent + hands_latent)


        face_latent_list.append(self.face2latent(decoded_face))
        upper_latent_list.append(self.motion_down_upper(motion_upper + upper_latent))
        hands_latent_list.append(self.motion_down_hands(motion_hands + hands_latent))
        lower_latent_list.append(self.motion_down_lower(motion_lower + lower_latent))

        face_latent_list =  [self.scaler_face[i](face_latent_list[0].reshape(face_latent_list[0].shape[0], -1, face_latent_list[0].shape[2]*(2**i))) for i in range(len(self.scaler_face))]
        upper_latent_list = [self.scaler_upper[i](upper_latent_list[0].reshape(upper_latent_list[0].shape[0], -1, upper_latent_list[0].shape[2]*(2**i))) for i in range(len(self.scaler_upper))]
        hands_latent_list = [self.scaler_hands[i](hands_latent_list[0].reshape(hands_latent_list[0].shape[0], -1, hands_latent_list[0].shape[2]*(2**i))) for i in range(len(self.scaler_hands))]
        lower_latent_list = [self.scaler_lower[i](lower_latent_list[0].reshape(lower_latent_list[0].shape[0], -1, lower_latent_list[0].shape[2]*(2**i))) for i in range(len(self.scaler_lower))]

        cls_face, face_latent =  self.index_pred(self.face_classifier, face_latent_list, caption_style_embedding_body)#[(self.face_classifier(face_latent_list[i])) for i in range(len(face_latent_list))] #.append(self.face_classifier(face_latent))
        cls_lower, lower_latent = self.index_pred(self.lower_classifier, lower_latent_list, caption_style_embedding_body)#[(self.lower_classifier(lower_latent_list[i])) for i in range(len(face_latent_list))]  # .append(self.lower_classifier(lower_latent))
        cls_upper, upper_latent = self.index_pred(self.upper_classifier, upper_latent_list, caption_style_embedding_body)#[(self.upper_classifier(upper_latent_list[i])) for i in range(len(face_latent_list))]#.append(self.upper_classifier(upper_latent))
        cls_hands, hands_latent = self.index_pred(self.hands_classifier, hands_latent_list, caption_style_embedding_body)#[(self.hands_classifier(hands_latent_list[i])) for i in range(len(face_latent_list))]#.append(self.hands_classifier(hands_latent))


        full_latent = torch.cat((face_latent, lower_latent, upper_latent, hands_latent), dim=2)
        style_latent = self.caption_encoder_styleloss(in_caption)
        full_style = self.style_decoder(tgt=style_latent[:,:1,:], memory=full_latent).repeat(1, style_latent.shape[1], 1)
        #full_style = self.caption_decoder_style(full_style).repeat(1, )
        #caption_style_embedding = self.caption_encoder_index(caption_style_embedding_body)
        #style_loss = nn.MSELoss(full_style, style_latent[:,:1,:])

        return {
            "rec_face": face_latent_list,
            "rec_upper": upper_latent_list,
            "rec_lower": lower_latent_list,
            "rec_hands": hands_latent_list,
            "cls_face": cls_face,
            "cls_upper": cls_upper,
            "cls_lower": cls_lower,
            "cls_hands": cls_hands,
            "full_style": full_style,
        }