# Comprehensive Code Documentation for Decentralized Causal Memory Networks (DCMN)

This document provides detailed explanations of each code file and function in the DCMN research implementation. The code combines neural networks, symbolic planning, and causal learning for advanced AI planning systems.

## System Overview

The DCMN system consists of several key components:
- **Neuro-Symbolic Planner**: Main planning engine combining neural networks and symbolic reasoning
- **Causal Memory Networks**: Learn and store cause-effect relationships from planning experiences
- **Execution Trace System**: Captures real execution data for causal learning
- **Multi-Agent Validation**: Consensus-based validation of causal knowledge

## Key Technologies Used

### Fast Downward Planner
**What it is**: Fast Downward is a state-of-the-art classical planning system developed at the University of Basel. It's considered one of the best symbolic planners in the AI planning community.

**Why we use it**: 
- **Optimality**: Finds provably correct action sequences that solve planning problems
- **Completeness**: Guarantees to find a solution if one exists
- **Efficiency**: Uses advanced search algorithms and heuristics for fast planning
- **Standard Format**: Works with PDDL (Planning Domain Definition Language), the standard format for planning problems
- **Research Validation**: Widely used in academic research, making our results comparable to other studies

**How it works**: Fast Downward takes two inputs:
1. Domain file (describes what actions are possible in general)
2. Problem file (describes the specific planning scenario)
It then uses search algorithms to find a sequence of actions that transforms the initial state into the goal state.

### PDDL (Planning Domain Definition Language)
**What it is**: PDDL is the standard language for describing planning problems in AI. It's like a formal specification language for planning.

**Why we use it**:
- **Standardization**: Universally accepted format in AI planning research
- **Precision**: Eliminates ambiguity in problem specification
- **Tool Compatibility**: Works with all major planning systems
- **Formal Verification**: Allows mathematical verification of plan correctness

**Structure**:
- **Domain**: Describes general rules (what actions exist, what they do)
- **Problem**: Describes specific instance (objects, initial state, goal)

### Graph Neural Networks (GNNs)
**What they are**: Neural networks designed to work with graph-structured data, where entities have relationships with each other.

**Why we use them**:
- **Relationship Modeling**: Perfect for representing cause-effect relationships
- **Flexible Structure**: Can handle varying numbers of entities and relationships
- **Pattern Recognition**: Learn to recognize important causal patterns
- **Scalability**: Efficiently process large knowledge graphs

### Large Language Models (LLMs)
**What they are**: Advanced AI models trained on massive text datasets to understand and generate human language.

**Why we use them**:
- **Natural Language Understanding**: Convert human instructions into formal planning problems
- **PDDL Generation**: Automatically create planning domain and problem files
- **Explanation Generation**: Provide human-readable explanations of plans
- **Semantic Reasoning**: Understanding context and intent in planning tasks

---

## 1. neuro_symbolic_planner.py

### Core Classes and Functions

#### `NeuralActionPredictor` (Neural Network)
**Purpose**: Predicts whether an action will succeed and how relevant it is to achieving the goal.

**Key Methods**:
- `__init__(input_dim=512, hidden_dim=256)`: Sets up neural network layers
  - Creates action encoder to convert actions into numbers the network understands
  - Creates success predictor to estimate probability of action success (0-1)
  - Creates relevance predictor to measure how important the action is (0-1)

- `forward(state_action_embedding)`: The main prediction function
  - Takes a combined representation of current state and proposed action
  - Returns two values: success probability and relevance score
  - Uses ReLU activations and dropout for regularization

#### `NeuralStateEvaluator` (Neural Network)
**Purpose**: Evaluates how good a planning state is and estimates distance to goal.

**Key Methods**:
- `__init__(input_dim=256, hidden_dim=256)`: Builds the evaluation network
  - Creates state encoder to understand the current world state
  - Creates goal distance head to estimate how far from the goal we are
  - Creates quality head to score how promising this state is

- `forward(state_embedding)`: Evaluates a state
  - Takes a numerical representation of the world state
  - Returns goal distance (how many steps left) and quality score (0-1)

#### `NeuralMemory`
**Purpose**: Stores successful planning experiences to help with future similar tasks.

**Key Methods**:
- `__init__(memory_size=1000)`: Creates memory storage with size limit
  - Uses a deque (double-ended queue) that automatically removes old memories

- `store(state_features, action_sequence, outcome, metadata)`: Saves a planning experience
  - Records what the world looked like, what actions were taken, and whether it worked
  - Includes timestamp for temporal analysis

- `retrieve_similar(query_state, k=3)`: Finds similar past experiences
  - Uses cosine similarity to compare current state with stored experiences
  - Returns the k most similar successful experiences
  - Helps the planner learn from past successes

#### `FastDownwardPlanner`
**Purpose**: Interface to the Fast Downward symbolic planner for finding optimal action sequences.

**What Fast Downward Does**: 
Fast Downward is a classical AI planner that finds optimal sequences of actions to achieve goals. Think of it like a GPS for AI planning - given a starting state and destination (goal), it finds the best route (action sequence) to get there.

**Why We Need It**:
- **Logical Correctness**: Ensures our plans actually work and don't have logical errors
- **Optimality**: Finds the shortest/best sequence of actions
- **Proven Technology**: Battle-tested in AI planning competitions and research
- **Domain Independence**: Works across different types of planning problems

**Key Methods**:
- `__init__(planner_dir="./fast-downward")`: Sets up the symbolic planner
  - Downloads and installs Fast Downward if not present (about 50MB download)
  - Configures the planner directory and build environment
  - This happens automatically when the system starts

- `_ensure_installed()`: Automatically installs Fast Downward
  - Clones the Fast Downward repository from GitHub (https://github.com/aibasel/downward.git)
  - Builds the planner using Python build scripts (compiles C++ code)
  - Handles installation failures gracefully with fallback options
  - Only runs once - subsequent starts use the already-installed version

- `plan(domain_file, problem_file, timeout=30)`: Runs symbolic planning
  - **Input**: Two PDDL files (domain defines rules, problem defines specific scenario)
  - **Process**: Tries multiple search strategies in order of sophistication:
    1. `lazy_greedy([ff(), cea()])` - Fast, greedy search with FF heuristic
    2. `astar(ff())` - Optimal A* search with FF heuristic  
    3. `lazy_greedy([hmax(), cea()])` - Alternative heuristic approach
    4. `eager_greedy([ff(), cea()])` - Eager evaluation variant
  - **Output Parsing**: Extracts action sequences from planner output text
  - **Fallback**: If all strategies fail, returns simple default plan
  - **Timeout Protection**: Stops trying after 30 seconds to prevent hanging

**How the Search Strategies Work**:
- **FF Heuristic**: Estimates how many actions needed to reach goal (Fast-Forward heuristic)
- **CEA**: Context-Enhanced Additive heuristic for better guidance
- **A***: Guarantees optimal solution but may be slower
- **Greedy**: Faster but may not find optimal solution

- `_extract_actions(domain_content)`: Parses PDDL domain files
  - Uses regex to find all action definitions
  - Returns set of available actions

- `_generate_move_plan(domain_content, problem_content)`: Creates fallback plans
  - Generates simple movement plans when symbolic planning fails
  - Extracts objects and locations from PDDL

#### `PDDLValidator`
**Purpose**: Validates that generated PDDL code is syntactically and semantically correct.

**Key Methods**:
- `validate_domain(domain_content)`: Checks PDDL domain syntax
  - Verifies domain definition is present
  - Checks parentheses are balanced
  - Ensures required sections (predicates, actions) exist

- `validate_problem(problem_content)`: Checks PDDL problem syntax
  - Verifies problem definition structure
  - Checks for required sections (domain, objects, init, goal)
  - Validates syntax integrity

#### `NeuroSymbolicPlanner` (Main Class)
**Purpose**: The central planner that combines neural networks, symbolic reasoning, and causal learning.

**Key Initialization**:
- Sets up Google Gemini AI for natural language understanding
- Initializes neural networks (action predictor, state evaluator)
- Creates symbolic planner and validator
- Sets up execution simulator and causal learning components

**Main Planning Method**:
- `plan_from_natural_language(task_description, max_iterations=3)`: 
  - **Step 1**: Assess task complexity and select planning strategy
  - **Step 2**: Generate PDDL domain and problem from natural language
  - **Step 3**: Use neural-guided symbolic planning
  - **Step 4**: Learn from execution results
  - **Step 5**: Return complete planning result with confidence scores

**Supporting Methods**:
- `_assess_task_complexity(task_description)`: Estimates how difficult a task is
- `_select_planning_strategy(confidence)`: Chooses planning approach based on difficulty
- `_llm_generate_pddl(task_description)`: Uses LLM to create PDDL from natural language
- `_neural_guided_planning()`: Combines neural predictions with symbolic planning

---

## 2. dcmn_causal_memory.py

### Understanding Causality and Causal Learning

#### What is Causality?
**Definition**: Causality is the relationship between cause and effect - understanding that one event (cause) leads to or influences another event (effect).

**In Planning Context**: In AI planning, causality means understanding:
- **Action Effects**: What happens when you perform an action (e.g., picking up a block makes your hand not empty)
- **Preconditions**: What must be true before an action can work (e.g., you need a clear path to move)
- **Dependencies**: How actions depend on each other (e.g., you must pick up a block before you can stack it)
- **State Changes**: How the world changes as a result of actions

**Example of Causal Relationships in Planning**:
- `pickup(block_a)` **PRODUCES** `holding(block_a)` (action causes effect)
- `pickup(block_a)` **REQUIRES** `clear(block_a)` (action needs precondition)
- `pickup(block_a)` **ENABLES** `stack(block_a, block_b)` (one action makes another possible)
- `pickup(block_a)` **PREVENTS** `pickup(block_b)` (can't hold two blocks at once)

#### What is Causal Learning?
**Traditional Approach**: Most AI planning systems use hand-coded rules like:
```
Rule: pickup(X) → holding(X) ∧ ¬hand-empty
```
These rules are written by humans and never change.

**Causal Learning Approach**: The system automatically discovers cause-effect relationships by:
1. **Observation**: Watching what happens when actions are performed
2. **Pattern Recognition**: Identifying recurring cause-effect patterns
3. **Relationship Extraction**: Learning that "when A happens, B usually follows"
4. **Confidence Assessment**: Understanding how reliable each causal relationship is

**Why Causal Learning Matters**:
- **Adaptability**: System adapts to new environments without reprogramming
- **Robustness**: Handles situations not anticipated by human programmers
- **Efficiency**: Learns from experience rather than exhaustive manual specification
- **Generalization**: Applies learned patterns to new but similar situations

#### How Causality is Applied in DCMN Architecture

##### 1. **Causal Relationship Types**
The system recognizes five fundamental types of causality in planning:

```python
class CausalRelationType(Enum):
    ENABLES = "enables"    # Action A makes action B possible
    REQUIRES = "requires"  # Action A needs condition C to work  
    PRODUCES = "produces"  # Action A creates effect E
    PREVENTS = "prevents"  # Action A stops action B from happening
    MODIFIES = "modifies"  # Action A changes state S
```

**Real-World Examples**:
- **ENABLES**: `open(door)` enables `enter(room)` - you must open a door before entering
- **REQUIRES**: `stack(A,B)` requires `holding(A)` - you must be holding block A to stack it
- **PRODUCES**: `pickup(A)` produces `holding(A)` - picking up creates the holding state
- **PREVENTS**: `pickup(A)` prevents `pickup(B)` - can't hold two objects simultaneously
- **MODIFIES**: `move(robot, kitchen)` modifies `location(robot)` - changes robot's location

##### 2. **Automatic Causal Discovery Process**

**Step 1: Execution Simulation**
```python
# System simulates plan execution and records state changes
execution_step = {
    'action': 'pickup(block_a)',
    'pre_state': {'clear(block_a)': True, 'hand-empty': True},
    'post_state': {'holding(block_a)': True, 'hand-empty': False},
    'success': True
}
```

**Step 2: Causal Relationship Extraction**
```python
# System automatically infers:
causal_triple_1 = CausalTriple(
    subject="pickup(block_a)",
    predicate=CausalRelationType.REQUIRES,
    object="clear(block_a)",  # This was needed before the action
    confidence=0.9
)

causal_triple_2 = CausalTriple(
    subject="pickup(block_a)", 
    predicate=CausalRelationType.PRODUCES,
    object="holding(block_a)",  # This was created by the action
    confidence=0.95
)
```

**Step 3: Learning Confidence Over Time**
- Initial observations have lower confidence (0.7)
- Repeated successful patterns increase confidence (up to 0.99)
- Failed attempts decrease confidence
- Conflicting evidence triggers re-evaluation

##### 3. **Causal Knowledge Representation**

**CausalKnowledgeAsset**: Packages related causal relationships together
```python
asset = CausalKnowledgeAsset(
    causal_triples=[
        # Multiple related cause-effect relationships learned together
        CausalTriple("pickup(X)", REQUIRES, "clear(X)"),
        CausalTriple("pickup(X)", PRODUCES, "holding(X)"),
        CausalTriple("pickup(X)", ENABLES, "stack(X,Y)")
    ],
    plan_trace=["pickup(block_a)", "stack(block_a, block_b)"],  # The plan that taught us this
    success_rate=0.92,  # How often this knowledge leads to success
    domain="blocks"  # What type of planning problem this applies to
)
```

##### 4. **Graph-Based Causal Reasoning**

**Think of it Like a Family Tree, But for Actions**

Imagine you're drawing a family tree that shows how people are related. In our system, we draw a similar tree that shows how actions and states are related to each other through cause-and-effect.

**What the Graph Looks Like**:

```
     pickup(A)  ----PRODUCES---→  holding(A)
                                      |
                                  ENABLES
                                      ↓
     stack(A,B)  ----PRODUCES---→   on(A,B)
```

**Breaking it Down Simply**:

1. **Nodes (the Circles)**: These are the "people" in our family tree
   - **Actions**: Things you can do (like `pickup(A)`, `stack(A,B)`)
   - **States**: Conditions in the world (like `holding(A)`, `on(A,B)`)

2. **Edges (the Arrows)**: These show relationships between the "people"
   - **PRODUCES**: "Action X creates state Y"
   - **ENABLES**: "State X makes action Y possible" 
   - **REQUIRES**: "Action X needs state Y to work"

**Real-World Example**:
```
Turn key ----PRODUCES---→ Engine running
    |                          |
    |                      ENABLES
    ↓                          ↓
Requires car                Drive car ----PRODUCES---→ Car moving
```

**How the Computer Processes This**:

**Step 1: Build the Knowledge Graph**
```python
# Like creating a contact list with relationships
nodes = [
    "pickup(A)",     # Action: pick up block A
    "holding(A)",    # State: I'm holding block A  
    "stack(A,B)",    # Action: put A on top of B
    "on(A,B)"        # State: A is sitting on B
]

edges = [
    # From → To, with relationship type
    ("pickup(A)", "holding(A)", "PRODUCES"),      # Picking up creates holding
    ("holding(A)", "stack(A,B)", "ENABLES"),      # Holding allows stacking
    ("stack(A,B)", "on(A,B)", "PRODUCES")         # Stacking creates the on-top state
]
```

**Step 2: Neural Network Makes Predictions**
The Graph Neural Network is like a smart friend who studies your family tree and can guess missing relationships:

```python
# You ask: "What's the relationship between pickup(A) and stack(A,B)?"
predicted_relation = gnn.predict_causal_link("pickup(A)", "stack(A,B)")
# Smart friend answers: ("ENABLES", confidence=87%)
```

**Why This Is Powerful**:

1. **Fills in Missing Links**: Even if we never directly observed "pickup enables stack", the network can figure it out by looking at the chain: pickup → holding → stack

2. **Learns Patterns**: After seeing many examples, it learns general rules like "holding something usually enables actions with that thing"

3. **Confidence Scoring**: It doesn't just guess - it tells you how sure it is (87% confident in this case)

**Visual Example of What the Network "Sees"**:
```
Input: "What connects pickup(A) to stack(A,B)?"

Network thinks:
- pickup(A) PRODUCES holding(A) ✓ (seen this many times)
- holding(A) ENABLES stack(A,B) ✓ (seen this too)
- So pickup(A) must ENABLE stack(A,B) through the chain!

Output: "ENABLES with 87% confidence"
```

This is like having a detective that can solve puzzles about cause-and-effect by studying patterns in the evidence!

#### Impact of Causal Learning in DCMN

##### 1. **Automatic Domain Adaptation**
**Traditional Systems**: Require manual programming for each new domain
```
// Manual rules for logistics domain
if truck_at(location) and package_at(location) then can_load(package, truck)
if package_in(truck) and truck_at(destination) then can_unload(package)
```

**DCMN System**: Learns domain rules automatically
```python
# System observes logistics planning and automatically learns:
CausalTriple("drive(truck, location)", ENABLES, "load(package, truck)")
CausalTriple("load(package, truck)", ENABLES, "unload(package, destination)")
CausalTriple("unload(package, destination)", PRODUCES, "package_at(destination)")
```

##### 2. **Cross-Domain Knowledge Transfer**
**Example**: Knowledge from cooking domain helps with chemistry lab tasks
```python
# Learned in cooking domain:
CausalTriple("heat(water)", PRODUCES, "hot(water)")
CausalTriple("hot(water)", ENABLES, "dissolve(salt)")

# Applied to chemistry domain:
# System recognizes that heating solutions enables dissolution reactions
```

##### 3. **Failure Recovery and Learning**
**When Plans Fail**: System learns what went wrong
```python
# Failed execution teaches negative causality:
CausalTriple("pickup(block_a)", REQUIRES, "clear(block_a)")  # High confidence
CausalTriple("pickup(block_a)", PREVENTS, "pickup(block_b)")  # Learned from failure
```

##### 4. **Confidence-Based Planning**

**Think of it Like Driving in Different Weather Conditions**

Imagine you're deciding how to drive to work:
- **Sunny day** → Drive normally, take shortcuts, go at normal speed
- **Heavy fog** → Drive slowly, take main roads, be extra careful

The AI planner does something similar - it adjusts its planning strategy based on how confident it is about cause-and-effect relationships.

**How Confidence Works**:

**High Confidence (80%+ sure)**: "I've seen this situation many times and know what works"
- The system has learned reliable causal relationships
- Example: In blocks world, it knows that pickup → holding → stack works 95% of the time
- Like driving a route you take every day

**Low Confidence (Below 80%)**: "This is new or uncertain territory"
- The system hasn't seen this situation much, or has conflicting information
- Example: First time seeing a cooking task when only trained on blocks
- Like driving in an unfamiliar neighborhood

**The Two Planning Strategies**:

**1. Fast Greedy Planning (High Confidence)**
```python
if causal_confidence > 0.8:
    use_fast_greedy_planning()  # "I know what I'm doing, let's go fast!"
```

**What this means**:
- **Speed**: Makes decisions quickly, doesn't second-guess itself
- **Efficiency**: Uses shortcuts and heuristics to find plans faster
- **Risk**: Willing to take calculated risks because it trusts its knowledge
- **Like**: An experienced chef who can cook without looking at recipes

**Example scenario**: 
```python
Task: "Stack block A on block B"
Confidence: 0.92 (very high - seen this thousands of times)
Strategy: Fast greedy
Reasoning: "I know exactly what to do: pickup(A) then stack(A,B). Done in 2 seconds."
```

**2. Careful Search Planning (Low Confidence)**
```python
else:
    use_careful_search_planning()  # "I'm not sure, better be safe"
```

**What this means**:
- **Thorough**: Considers many different options and possibilities
- **Safe**: Avoids risky actions that might fail
- **Slower**: Takes more time to ensure the plan will work
- **Validated**: Double-checks each step before committing
- **Like**: A student cook who carefully follows recipes and measures everything

**Example scenario**:
```python
Task: "Prepare a molecular gastronomy dish"
Confidence: 0.45 (low - never seen this before)
Strategy: Careful search
Reasoning: "This is new. Let me carefully consider all options, check preconditions, 
          and use the most reliable actions even if it takes longer."
```

**Detailed Example of How This Works**:

**Scenario**: AI needs to plan "Make coffee in an unfamiliar kitchen"

**Step 1: Check Confidence**
```python
# AI looks at its causal knowledge about coffee making
coffee_knowledge = [
    ("turn_on(coffee_maker)", "PRODUCES", "hot_coffee"),  # Confidence: 0.6 (seen only twice)
    ("add_water(coffee_maker)", "ENABLES", "turn_on(coffee_maker)"),  # Confidence: 0.4 (unsure)
    ("find(coffee_beans)", "ENABLES", "make_coffee"),  # Confidence: 0.3 (very unsure)
]

overall_confidence = average([0.6, 0.4, 0.3]) = 0.43  # Below 0.8 threshold
```

**Step 2: Choose Strategy**
```python
if 0.43 > 0.8:  # False
    use_fast_greedy_planning()
else:
    use_careful_search_planning()  # This gets executed
```

**Step 3: Execute Careful Strategy**
```python
# Careful approach:
1. First, explore the kitchen thoroughly to understand available equipment
2. Check all preconditions before each action
3. Consider multiple backup plans
4. Prefer well-known, reliable actions over risky shortcuts
5. Validate each step before proceeding

Plan generated:
[
    "explore_kitchen()",           # Safe: gather information first
    "locate_coffee_maker()",       # Safe: make sure it exists  
    "check_water_level()",         # Safe: verify preconditions
    "add_water_if_needed()",       # Safe: handle contingency
    "locate_coffee_supplies()",    # Safe: ensure we have ingredients
    "follow_standard_procedure()", # Safe: use tried-and-true method
]
```

**Contrast with High-Confidence Scenario**:

**Scenario**: AI needs to plan "Stack 3 blocks" (familiar task)

```python
# High confidence from lots of experience
stacking_confidence = 0.94

if 0.94 > 0.8:  # True
    use_fast_greedy_planning()  # Execute this
    
# Fast approach:
Plan generated in 0.5 seconds:
[
    "pickup(block_a)",
    "stack(block_a, block_b)", 
    "pickup(block_c)",
    "stack(block_c, block_a)"
]
# Done! No hesitation, no extra checking needed.
```

**Why This Matters**:

1. **Efficiency**: Don't waste time being overly careful when you're sure of what to do
2. **Safety**: Don't rush when you're in unfamiliar territory
3. **Adaptability**: Automatically adjusts to new situations and domains
4. **Human-like**: Mimics how humans adjust their approach based on experience
5. **Resource Management**: Allocates computational resources wisely

**Real-World Benefits**:
- **Emergency situations**: Quick decisions when confidence is high
- **Novel problems**: Careful exploration when confidence is low
- **Learning**: System gets more confident (and faster) as it gains experience
- **Reliability**: Reduces failures by matching strategy to knowledge level

This is like having an AI that knows when to trust its gut and when to double-check everything!

#### Uniqueness of DCMN's Causal Learning

##### 1. **Dynamic vs. Static Knowledge**
**Traditional**: Fixed rules that never change
**DCMN**: Continuously learning and updating causal relationships

##### 2. **Multi-Agent Validation of Causality**
**Traditional**: Single system learns causality (prone to errors)
**DCMN**: Multiple agents validate each causal relationship through consensus

##### 3. **Cryptographic Verification**
**Traditional**: No way to verify integrity of learned knowledge
**DCMN**: Each causal knowledge asset is cryptographically hashed and traceable

##### 4. **Graph Neural Network Integration**
**Traditional**: Simple if-then rules
**DCMN**: Sophisticated graph neural networks understand complex causal patterns

##### 5. **Domain-Agnostic Representation**
**Traditional**: Domain-specific rules that don't transfer
**DCMN**: Abstract causal representations that work across domains

##### 6. **Explainable Causality**
**Traditional**: Black box decision making
**DCMN**: Can explain why it believes certain causal relationships exist

This causal learning approach transforms AI planning from rigid, hand-coded systems to adaptive, learning systems that improve through experience and can handle novel situations they've never explicitly been programmed for.

---

### Core Classes and Functions

#### `CausalTriple`
**Purpose**: Represents a single cause-effect relationship in planning.

**Structure**:
- `subject`: The action or condition that causes something (e.g., "pickup(block)")
- `predicate`: The type of relationship (ENABLES, REQUIRES, PRODUCES, PREVENTS, MODIFIES)
- `object`: What is affected by the subject (e.g., "holding(block)")
- `confidence`: How sure we are about this relationship (0.0 to 1.0)

**Key Methods**:
- `to_rdf()`: Converts the relationship to RDF format for knowledge graphs

**What is RDF Format?**
RDF stands for **Resource Description Framework** - it's like a universal language for describing information that different computer systems can understand and share.

**Think of RDF Like Sentence Structure**:
Every piece of information in RDF follows the same pattern: **Subject - Predicate - Object**
- Just like English sentences: "John likes pizza" 
- **Subject**: John (who/what we're talking about)
- **Predicate**: likes (the relationship/action)  
- **Object**: pizza (what it relates to)

**In Our Causal System**:
```python
# Our causal relationship:
CausalTriple("pickup(block_a)", PRODUCES, "holding(block_a)")


**Why Use RDF?**
1. **Universal Standard**: Like having a common language that all AI systems can speak
2. **Interoperability**: Our causal knowledge can be shared with other AI systems
3. **Web Compatibility**: Works with the semantic web and linked data
4. **Database Integration**: Can be stored in graph databases and queried easily
5. **Future-Proof**: Standard format that will work with future AI technologies

**Real-World Example**:
Instead of storing "pickup causes holding" in our own custom format, RDF lets us store it in a way that other planning systems, knowledge bases, and AI research tools can understand and use.
- `__hash__()`: Allows relationships to be stored in sets and dictionaries

#### `CausalKnowledgeAsset`
**Purpose**: A cryptographically-verified collection of causal relationships learned from planning.

**Structure**:
- Contains multiple causal triples learned from one planning experience
- Includes the plan that taught us these relationships
- Has success rate and validation scores from multiple agents
- Cryptographically hashed for integrity verification

**Key Methods**:
- `generate_hash()`: Creates unique identifier for the knowledge
- `add_validation(validator_id, score)`: Records validation from an expert agent
- `get_consensus_score()`: Calculates weighted agreement across validators
- `to_rdf_graph()`: Exports knowledge to RDF format for sharing


#### CausalGraphMemory (Neural Network + Graph Storage)
**Purpose**: Neural network-based memory that stores and retrieves causal knowledge using graph neural networks.

**What It Does**: 
Think of this as the "brain" of the system that remembers what causes what in planning. It's like having a smart notebook that not only stores your experiences but also understands the relationships between them and can suggest relevant past experiences when you face a new problem.

**Why Graph Neural Networks**:
- **Relationship-Aware**: Unlike regular neural networks, GNNs understand that entities are connected
- **Variable Structure**: Can handle graphs of any size (some problems have 3 objects, others have 30)
- **Attention Mechanism**: Focuses on the most important relationships when making decisions
- **Learning Capability**: Gets better at understanding causal patterns over time

**Architecture Details**:
- **Graph Attention Networks (GAT)**: Uses attention mechanisms to focus on important causal relationships
- **Dual Storage**: Stores knowledge in both neural format (for learning) and symbolic format (for reasoning)
- **Embedding Dimension**: 256-dimensional vectors represent actions and states
- **Multi-Head Attention**: 4 attention heads capture different types of relationships

**Key Methods**:
- `store_asset(asset)`: Saves causal knowledge in the graph
  - **RDF Storage**: Adds to formal knowledge graph (like a database of facts)
  - **Neural Indexing**: Updates neural network indices for fast similarity search
  - **Graph Structure**: Creates nodes for actions/states and edges for relationships
  - **Retrieval Optimization**: Preprocesses data for fast future retrieval

- `retrieve_by_task(task_description, task_embedding, k=5)`: Finds relevant knowledge
  - **Graph Conversion**: Converts knowledge graph to PyTorch Geometric format
  - **Node Features**: Creates vector representations for each action/state
  - **Edge Processing**: Processes causal relationships as graph edges
  - **GAT Processing**: Runs Graph Attention Networks to compute embeddings
  - **Similarity Scoring**: Uses cosine similarity between task and stored experiences
  - **Consensus Weighting**: Weights results by how many experts agreed on them

- `find_similar_tasks(current_task, similarity_threshold=0.7)`: Cross-task learning
  - **Text Similarity**: Uses Jaccard similarity (word overlap) for task comparison
  - **Success Weighting**: Prioritizes experiences that worked well in the past
  - **Consensus Filtering**: Only considers knowledge validated by multiple agents
  - **Transfer Learning**: Enables using kitchen experience for household tasks, etc.

- `discover_causal_links(action1, action2)`: Predicts new relationships
  - **Neural Prediction**: Uses trained networks to predict if action1 affects action2
  - **Relationship Classification**: Predicts type (ENABLES, REQUIRES, PRODUCES, etc.)
  - **Confidence Scoring**: Provides confidence level for the prediction
  - **Novel Discovery**: Can discover relationships never seen before in training

#### `MultiAgentValidator`
**Purpose**: Implements consensus-based validation where multiple AI agents verify causal knowledge.

**What This Solves**: 
In traditional AI systems, there's no way to verify if learned knowledge is actually correct. This system uses multiple independent AI agents to validate each piece of causal knowledge, similar to peer review in scientific research.

**Why Multi-Agent Validation**:
- **Quality Control**: Prevents bad knowledge from polluting the system
- **Robustness**: Single agent mistakes don't affect the system
- **Confidence Scoring**: Provides reliability estimates for each piece of knowledge
- **Self-Improving**: Agent reputations improve the validation over time

**How Consensus Works**:
- **Minimum Validators**: Requires at least 3 agents to validate each piece of knowledge
- **Weighted Voting**: Agents with better track records have more influence
- **Consensus Threshold**: Knowledge needs 70% agreement to be accepted
- **Dynamic Thresholds**: Can adjust requirements based on knowledge importance

**Key Methods**:
- `validate_asset(asset, validators)`: Multi-agent consensus validation
  - **Input**: Causal knowledge asset + list of (agent_id, validation_score) pairs
  - **Reputation Weighting**: Multiplies each score by the agent's reputation (0.1 to 2.0)
  - **Consensus Calculation**: Computes weighted average of all validation scores
  - **Threshold Check**: Accepts knowledge if consensus ≥ 0.7 (70% agreement)
  - **Asset Update**: Stores all validation scores in the knowledge asset
  - **History Tracking**: Records validation event with timestamp

- `_update_reputations(validators, consensus)`: Agent reputation management
  - **Alignment Scoring**: Measures how close each agent's score was to final consensus
  - **Reputation Delta**: Agents closer to consensus get +0.1 boost, others get -0.05 penalty
  - **Bounds Checking**: Keeps reputation between 0.1 (minimum) and 2.0 (maximum)
  - **Decay Protection**: Prevents reputation from dropping too quickly
  - **Quality Incentive**: Rewards agents who consistently provide good validations

#### `DomainParanet`
**Purpose**: Self-organizing network of AI agents that specialize in specific planning domains (e.g., blocks world, logistics).

**Key Methods**:
- `add_member(agent_id, initial_expertise)`: Adds expert to domain network
  - Only admits agents with sufficient expertise
  - Tracks performance over time

- `share_knowledge(asset, contributor_id)`: Knowledge sharing within domain
  - Stores validated knowledge accessible to all domain members
  - Rewards contributors with expertise boosts

- `update_expertise(agent_id, performance)`: Dynamic expertise tracking
  - Updates agent expertise based on recent performance
  - Removes agents whose expertise drops below threshold

- `collective_validation(asset)`: Domain expert validation
  - Gets validation scores from top domain experts
  - Leverages specialized knowledge for better validation

#### `GraphNeuralRetriever`
**Purpose**: Advanced retrieval system using Graph Neural Networks for finding relevant causal knowledge.

**Key Methods**:
- `retrieve(query, k=5)`: GNN-based knowledge retrieval
  - Encodes queries using neural networks
  - Uses graph structure to find most relevant knowledge
  - Returns ranked list of causal assets

- `find_causal_paths(start_state, goal_state)`: Causal path discovery
  - Uses graph traversal (BFS) to find causal chains
  - Connects starting conditions to goal states
  - Returns multiple possible causal pathways

#### Integration Functions
- `integrate_dcmn_with_planner(planner)`: Adds DCMN capabilities to existing planner
  - Replaces simple memory with causal graph memory
  - Adds multi-agent validation
  - Creates domain-specific expert networks
  - Enables causal learning from planning experiences

---

## 3. execution_trace_system.py

### Core Classes and Functions

#### `ExecutionStep`
**Purpose**: Records what happened during a single action in plan execution.

**Why This Matters**: 
Traditional planning systems never actually "execute" their plans - they just generate them. This system simulates execution to learn what really happens when actions are performed, capturing the cause-effect relationships that occur.

**What It Captures**:
- **Before and After**: Complete state of the world before and after each action
- **Success/Failure**: Whether the action actually worked as intended
- **Timing**: How long actions take (useful for real-world applications)
- **Error Details**: Specific reasons why actions might fail

**Structure**:
- `action`: The action that was performed (e.g., "pickup(block_a)")
- `pre_state`: World state before the action (dict of condition → true/false)
- `post_state`: World state after the action (dict showing what changed)
- `success`: Whether the action succeeded (boolean)
- `duration`: How long the action took (float, in seconds)
- `error_message`: Any error that occurred (string, None if successful)
- `timestamp`: When this action was executed (for temporal analysis)

#### `ExecutionTrace`
**Purpose**: Complete record of an entire plan execution from start to finish.

**Structure**:
- Contains sequence of execution steps
- Records initial and final world states
- Tracks overall success and total execution time

#### `PDDLExecutionSimulator`
**Purpose**: Simulates plan execution to generate realistic execution traces for causal learning.

**Key Methods**:
- `simulate_plan_execution(domain_pddl, problem_pddl, plan, task_description)`: Main simulation
  - Parses PDDL to understand world and actions
  - Simulates each action step by step
  - Records state changes and success/failure
  - Returns complete execution trace

- `_parse_initial_state(problem_pddl)`: Extracts starting world state
  - Uses regex to find initial state predicates
  - Converts to true/false state representation

- `_parse_action_effects(domain_pddl)`: Understands what actions do
  - Extracts preconditions (what actions need)
  - Extracts effects (what actions change)
  - Creates action definition database

- `_simulate_action(action, current_state, action_definitions)`: Simulates single action
  - Checks if preconditions are met
  - Applies action effects to state
  - Returns success/failure and new state

- `_simulate_unknown_action(action, current_state)`: Handles unknown actions
  - Makes reasonable assumptions about action effects
  - Uses action name to infer likely effects
  - Adds realistic failure probability

#### `CausalLearningIntegrator`
**Purpose**: Connects execution traces with causal learning to automatically discover cause-effect relationships.

**Key Methods**:
- `execute_and_learn(domain_pddl, problem_pddl, plan, task_description, causal_memory)`: 
  - Simulates plan execution
  - Extracts causal relationships from execution trace
  - Stores learned knowledge in causal memory
  - Updates system's understanding of cause-effect relationships

- `get_execution_statistics()`: Performance tracking
  - Calculates success rates across multiple executions
  - Tracks average plan lengths and execution times
  - Provides insights into system performance

---

## 4. dcmn_integration_example.py

### Demonstration Functions

#### `demonstrate_dcmn_integration()`
**Purpose**: Shows how all DCMN components work together with practical examples.

**Examples Demonstrated**:
1. **Storing Causal Knowledge**: How planning experiences become causal assets
2. **Multi-Agent Validation**: How multiple AI agents verify knowledge quality
3. **Knowledge Retrieval**: How to find relevant past experiences
4. **Causal Path Discovery**: How to find cause-effect chains
5. **Plan Augmentation**: How causal knowledge improves planning
6. **Domain Expertise**: How agents specialize in specific areas
7. **Knowledge Export**: How to save and share causal knowledge

#### `demonstrate_advanced_features(planner)`
**Purpose**: Shows advanced DCMN capabilities.

**Advanced Features**:
1. **Cross-Domain Transfer**: Using knowledge from one domain in another
2. **Causal Link Prediction**: Predicting new relationships using neural networks
3. **Consensus Evolution**: How validation consensus changes over time
4. **RDF Export**: Making knowledge compatible with other AI systems

---

## 5. Test and Debug Files

### test_basic_functionality.py
**Purpose**: Verifies that the DCMN system works correctly with simple tasks.
- Tests system initialization
- Runs simple planning tasks
- Reports success/failure rates

### debug_pddl_generation.py
**Purpose**: Debugging tool for PDDL generation and symbolic planning.
- Shows generated PDDL code
- Tests Fast Downward planner directly
- Helps identify planning failures

---

## Key System Innovations

### 1. True Neuro-Symbolic Integration
**Traditional Approach**: Most "neuro-symbolic" systems just use neural networks as heuristics for symbolic planners - they don't truly integrate.

**Our Innovation**: Neural components actively guide every aspect of symbolic planning:
- **Neural Action Selection**: Neural networks predict which actions are most likely to succeed
- **State Quality Assessment**: Neural evaluation of how promising each planning state is
- **Experience-Guided Planning**: Neural memory retrieves similar successful experiences
- **Confidence-Based Strategy**: Neural confidence scores determine which planning strategy to use

**Why This Matters**: Creates a truly integrated system where neural and symbolic components work together, rather than neural components just providing hints.

### 2. Causal Learning from Execution
**Traditional Approach**: Most systems use hand-coded domain knowledge or pre-programmed rules about what causes what.

**Our Innovation**: Learns cause-effect relationships automatically by:
- **Execution Simulation**: Actually "runs" the plans to see what happens
- **State Change Tracking**: Records exactly what changes when each action is performed
- **Pattern Discovery**: Identifies recurring cause-effect patterns across many executions
- **Dynamic Learning**: Continuously improves understanding as more plans are executed

**Why This Matters**: Eliminates need for manual domain engineering and adapts to new domains automatically.

### 3. Decentralized Knowledge Validation
**Traditional Approach**: Knowledge is either hard-coded by humans or learned by a single system without validation.

**Our Innovation**: Multi-agent consensus system ensures knowledge quality:
- **Peer Review Model**: Multiple independent AI agents validate each piece of learned knowledge
- **Cryptographic Integrity**: Knowledge assets are cryptographically hashed to prevent tampering
- **Reputation Systems**: Agents build reputations based on validation accuracy
- **Domain Expertise**: Specialists in each domain (blocks, logistics, cooking) provide expert validation

**Why This Matters**: Prevents bad knowledge from corrupting the system and provides confidence estimates.

### 4. Graph Neural Network Memory
**Traditional Approach**: Memory systems use simple vector similarity or keyword matching.

**Our Innovation**: Advanced graph-based memory that understands relationships:
- **Relationship Awareness**: Graph Neural Networks understand that actions and states are connected
- **Attention Mechanisms**: Focuses on the most important causal relationships for each query
- **Structural Learning**: Learns patterns in the structure of causal relationships
- **Scalable Retrieval**: Efficiently searches large knowledge graphs

**Why This Matters**: Enables sophisticated reasoning about cause-effect relationships rather than just pattern matching.

### 5. Cross-Domain Learning
**Traditional Approach**: Planning systems are domain-specific and can't transfer knowledge between different types of problems.

**Our Innovation**: Enables knowledge transfer across domains:
- **Abstract Causal Patterns**: Represents causality in domain-independent ways
- **Similarity Matching**: Finds similar causal patterns across different domains
- **Federated Knowledge**: Shares knowledge across domain-specific expert networks
- **Transfer Learning**: Uses cooking knowledge to help with household tasks, etc.

**Why This Matters**: Dramatically reduces the amount of training needed for new domains.

### 6. Explainable AI Planning
**Traditional Approach**: Planning systems provide action sequences but no explanation of why those actions were chosen.

**Our Innovation**: Natural language explanations based on causal reasoning:
- **Causal Explanation**: Explains why each action was chosen based on learned cause-effect relationships
- **Confidence Communication**: Tells users how confident the system is in each decision
- **Alternative Discussion**: Explains why other actions were not chosen
- **Learning Transparency**: Shows what the system learned from each planning experience

**Why This Matters**: Makes AI planning decisions understandable and trustworthy for human users.

This architecture represents a significant advance in AI planning by combining the strengths of neural learning, symbolic reasoning, and decentralized validation in a unified system that learns and improves over time while remaining explainable and trustworthy.