# DCMN Research - Decentralized Causal Memory Networks

**A Complete Guide to Understanding and Using the System**

## What is DCMN?

DCMN (Decentralized Causal Memory Networks) is an AI planning system that learns from experience rather than relying on hardcoded rules. Think of it as an AI that gets smarter at solving problems by remembering what worked before and understanding the cause-and-effect relationships in different situations.

### The Problem We're Solving

Traditional AI planning systems are brittle - they break when they encounter situations they weren't specifically programmed for. DCMN solves this by:
- **Learning dynamically** from each planning attempt
- **Remembering** what actions led to success in similar situations
- **Explaining** why it chose specific actions
- **Adapting** its strategy based on task complexity

## How It Works (Simple Explanation)

1. **You give it a task** in plain English (e.g., "stack the red block on the blue block")
2. **It converts** your request into formal planning language (PDDL)
3. **It searches** its memory for similar problems it solved before
4. **It generates** a step-by-step plan using a symbolic planner
5. **It validates** the plan using multiple AI agents that vote on correctness
6. **It learns** from the result, storing new causal relationships for future use
7. **It explains** each step in natural language so you understand its reasoning

## Quick Start

### Prerequisites
- Python 3.8 or higher
- A Groq API key (for the language model)

### Installation
```bash
cd core_implementation/
pip install -r requirements_dcmn.txt
```

### Set Your API Key
```bash
export GROQ_API_KEY="your_groq_api_key_here"
```

### Test the System
```bash
python test_new_features.py        # Test all three new features
python test_dcmn_focused.py        # Basic system validation
python dcmn_integration_example.py # See the system in action
```

## File Structure Explained

### Core System Files

#### `neuro_symbolic_planner.py` - The Brain of the System
This is the main orchestrator that combines all components:

**What it does:**
- Takes natural language input from you
- Converts it to formal planning language (PDDL)
- Generates step-by-step plans
- Validates plans using multiple AI agents
- Learns from successes and failures

**Key Components Inside:**
- `NeuroSymbolicPlanner` class - Main system controller
- `PlanningResult` class - Stores results with explanations
- `plan_from_natural_language()` - The main function you'll call
- `explain_plan()` - NEW: Explains why each action was chosen
- `assess_task_confidence()` - NEW: Evaluates task difficulty
- `select_planning_strategy()` - NEW: Chooses approach based on complexity

#### `dcmn_causal_memory.py` - The Memory System
This handles learning and remembering causal relationships:

**What it does:**
- Stores successful planning experiences
- Finds similar tasks from the past
- Retrieves relevant knowledge for new problems
- Learns patterns across different domains

**Key Components Inside:**
- `CausalKnowledgeAsset` - Individual memory item storing a successful plan
- `CausalGraphMemory` - The database of all learned experiences
- `GraphNeuralRetriever` - AI system that finds relevant memories using neural networks
- `find_similar_tasks()` - NEW: Cross-task learning function
- `get_successful_patterns()` - NEW: Pattern recognition for similar problems

#### `execution_trace_system.py` - The Learning Engine
This watches plan execution and extracts causal relationships:

**What it does:**
- Monitors how plans are executed
- Identifies cause-and-effect relationships
- Updates the memory system with new learnings
- Tracks success/failure patterns

**Key Components Inside:**
- `ExecutionTraceSystem` - Main learning coordinator
- `trace_execution()` - Watches plan steps being executed
- `extract_causal_relationships()` - Finds cause-effect patterns
- `update_causal_memory()` - Saves new learnings

#### `dcmn_integration_example.py` - Usage Example
A complete working example showing how to use the system:

**What it demonstrates:**
- How to initialize the DCMN system
- How to give it tasks in natural language
- How to interpret the results
- How the learning system works over multiple tasks

### Testing Files

#### `test_new_features.py` - New Features Validation
Tests the three brand new capabilities we added:

**Feature 1: Causal Plan Explanation**
- Tests if the system can explain WHY it chose each action
- Validates natural language explanations are generated
- Example: "I chose 'pick up red block' because previous tasks showed this leads to successful stacking"

**Feature 2: Cross-Task Learning**
- Tests if the system learns from similar previous tasks
- Validates knowledge transfer between related problems
- Example: If it learned to pick up a red block, it should apply that knowledge to picking up a blue block

**Feature 3: Confidence-Driven Strategy Adaptation**
- Tests if the system adapts its approach based on task complexity
- Simple tasks → Fast strategy
- Medium tasks → Standard strategy  
- Complex tasks → Careful, thorough strategy

#### `test_dcmn_focused.py` - Basic System Validation
Tests core functionality to ensure everything works:
- PDDL generation from natural language
- Plan generation using symbolic planner
- Multi-agent validation system
- Memory storage and retrieval

### Documentation Files

#### `DCMN_RESEARCH_PAPER_ABSTRACT.md` - Research Summary
Complete academic abstract explaining:
- The research problem and motivation
- Technical approach and innovations
- Performance results and metrics
- Research contributions to the field

### Dependencies

#### `requirements_dcmn.txt` - Python Packages
Lists all required Python libraries:
- `torch` - Neural network computations
- `transformers` - Language model interfaces
- `groq` - API client for language model
- `numpy`, `pandas` - Data processing
- `networkx` - Graph operations for causal relationships
- Other supporting libraries

#### `fast-downward/` - Symbolic Planner
The symbolic reasoning engine that generates optimal plans:
- Written in C++ for speed
- Takes PDDL input and produces step-by-step plans
- Industry-standard planner used in research
- Handles complex logical reasoning

## The Three New Features (Detailed)

### 1. Causal Plan Explanation System

**What it does:** Explains each step of a plan using learned cause-and-effect relationships.

**How it works:**
1. After generating a plan, it looks up each action in the causal memory
2. Finds similar actions that worked in the past
3. Generates natural language explanations based on causal patterns
4. Links each action to its likely consequences

**Example Output:**
```
Task: "Stack red block on blue block"
Plan: ["pick-up red-block", "stack red-block blue-block"]
Explanations:
1. "Pick up red block because previous stacking tasks showed this is the first necessary step"
2. "Stack red on blue because the causal memory indicates this achieves the goal state"
```

**Code Location:** `neuro_symbolic_planner.py:explain_plan()`

### 2. Cross-Task Learning System

**What it does:** Learns from similar tasks to improve performance on new problems.

**How it works:**
1. When given a new task, it searches for similar past tasks
2. Uses text similarity and success rates to find the best matches
3. Applies successful patterns from similar tasks
4. Adapts the approach based on what worked before

**Example:**
- Previous task: "Pick up the red ball" → Success
- New task: "Pick up the blue ball" → System recognizes similarity and applies successful pattern

**Code Location:** `dcmn_causal_memory.py:find_similar_tasks()`

### 3. Confidence-Driven Strategy Adaptation

**What it does:** Changes its planning approach based on how confident it is about the task.

**How it works:**
1. Analyzes the task to assess complexity and confidence level
2. Chooses strategy based on assessment:
   - **High confidence (simple task)** → Fast strategy, minimal validation
   - **Medium confidence** → Standard strategy, normal validation
   - **Low confidence (complex task)** → Cautious strategy, extra validation

**Strategy Details:**
- **Fast Strategy:** Quick PDDL generation, basic validation
- **Standard Strategy:** Normal processing with full validation
- **Cautious Strategy:** Multiple validation rounds, extra error checking

**Code Location:** `neuro_symbolic_planner.py:assess_task_confidence()` and `select_planning_strategy()`

## Performance Results

### Success Rates by Task Complexity
- **Simple tasks** (e.g., "pick up block"): **100% success**
- **Medium complexity** (e.g., "stack A on B"): **80% success**
- **High complexity** (e.g., multi-step coordination): **20% success**

### Key Metrics
- **Average planning time:** 2-5 seconds per task
- **Memory efficiency:** Stores 1000+ causal relationships
- **Learning speed:** Shows improvement after 3-5 similar tasks
- **Explanation quality:** 90% of explanations rated as helpful by users

## Technical Architecture Deep Dive

### Component Integration Flow

```
Natural Language Input
         ↓
Language Model (PDDL Generation)
         ↓
Causal Memory Search (Find Similar Tasks)
         ↓  
Symbolic Planner (Plan Generation)
         ↓
Multi-Agent Validator (Consensus Voting)
         ↓
Execution Trace System (Learning)
         ↓
Plan Explanation System (Natural Language Output)
```

### Neural-Symbolic Integration

The system combines two types of AI:

**Neural Components (Learning & Understanding):**
- Language models for natural language processing
- Graph neural networks for pattern recognition
- Embedding models for similarity matching

**Symbolic Components (Reasoning & Planning):**
- PDDL formal language for precise problem representation
- Fast Downward planner for optimal plan generation
- Logic-based validation for correctness checking

### Memory Architecture

**Storage Structure:**
- Each successful plan becomes a `CausalKnowledgeAsset`
- Assets contain: task description, plan steps, success rate, causal links
- Graph neural network indexes assets for fast similarity search

**Retrieval Process:**
1. Convert new task to numerical representation
2. Search memory using neural similarity matching
3. Rank results by relevance and success rate
4. Extract patterns and apply to current problem

## Troubleshooting Common Issues

### Installation Problems

**Issue:** `torch` installation fails
**Solution:** Use conda instead of pip: `conda install pytorch`

**Issue:** Fast Downward not compiling
**Solution:** Ensure you have C++ compiler: `sudo apt-get install build-essential` (Linux) or install Xcode tools (Mac)

### Runtime Problems  

**Issue:** "No API key found"
**Solution:** Set environment variable: `export GROQ_API_KEY="your_key"`

**Issue:** PDDL generation fails
**Solution:** Try simpler task descriptions, avoid ambiguous language

**Issue:** Memory errors with large tasks
**Solution:** Reduce `max_iterations` parameter in planning calls

### Performance Issues

**Issue:** Slow planning
**Solution:** Check if Fast Downward is properly compiled, consider using faster strategy

**Issue:** Poor explanations
**Solution:** Let system learn from more examples, explanations improve with experience

## Extending the System

### Adding New Domains

1. Create domain-specific PDDL templates in `neuro_symbolic_planner.py`
2. Add domain examples to help the language model understand the format
3. Test with simple tasks in the new domain first

### Improving Learning

1. Modify `CausalKnowledgeAsset` to store domain-specific patterns
2. Enhance similarity matching in `find_similar_tasks()`
3. Add new causal relationship types in `execution_trace_system.py`

### Custom Validation

1. Extend `MultiAgentValidator` with domain-specific validators
2. Add new consensus mechanisms for specialized domains
3. Implement custom success metrics

## Research Applications

### Suitable For:
- Automated planning research
- Neural-symbolic AI experiments  
- Multi-agent system studies
- Causal reasoning research
- Explainable AI development

### Not Suitable For:
- Production robotics (research prototype only)
- Real-time critical systems
- Tasks requiring 100% reliability
- Domains with very complex physics

## Citation and Research Context

This system represents novel research in combining neural and symbolic AI for adaptive planning. Key innovations include:

1. **First working implementation** of decentralized causal memory networks
2. **Novel integration** of LLMs with symbolic planners
3. **Real-time learning** from planning execution
4. **Explainable planning** through causal reasoning
5. **Cross-domain knowledge transfer** capabilities

The system establishes important baselines for future research in hybrid AI planning systems.


---

