#!/usr/bin/env python3
"""
Debug script to examine what PDDL is being generated and why Fast Downward fails
"""

import os
import sys
from pathlib import Path

# Add parent directory to path
sys.path.append(str(Path(__file__).parent))

from neuro_symbolic_planner import NeuroSymbolicPlanner
from dcmn_causal_memory import integrate_dcmn_with_planner

def debug_pddl_generation():
    """Debug PDDL generation for a simple task"""
    
    print("=" * 70)
    print("🔍 DEBUGGING PDDL GENERATION")
    print("=" * 70)
    
    # Get API key
    api_key = os.getenv("GROQ_API_KEY", "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")
    
    try:
        print("\n🔧 Initializing DCMN System...")
        planner = NeuroSymbolicPlanner(api_key)
        planner = integrate_dcmn_with_planner(planner)
        print("   ✓ DCMN system initialized")
        
        # Test very simple task
        simple_task = "Pick up the red block"
        print(f"\n📋 Task: {simple_task}")
        
        # Generate PDDL directly
        print("\n🧠 Generating PDDL...")
        domain_pddl, problem_pddl = planner._llm_generate_pddl(simple_task)
        
        print("\n" + "="*50)
        print("📄 GENERATED DOMAIN PDDL:")
        print("="*50)
        print(domain_pddl)
        
        print("\n" + "="*50) 
        print("📄 GENERATED PROBLEM PDDL:")
        print("="*50)
        print(problem_pddl)
        
        # Try to run Fast Downward manually
        print("\n" + "="*50)
        print("🚀 TESTING FAST DOWNWARD:")
        print("="*50)
        
        # Save to temp files
        import tempfile
        with tempfile.NamedTemporaryFile(mode='w', suffix='.pddl', delete=False) as f:
            f.write(domain_pddl)
            domain_file = f.name
            
        with tempfile.NamedTemporaryFile(mode='w', suffix='.pddl', delete=False) as f:
            f.write(problem_pddl)
            problem_file = f.name
        
        print(f"Domain file: {domain_file}")
        print(f"Problem file: {problem_file}")
        
        # Test with Fast Downward
        success, plan = planner.symbolic_planner.plan(domain_file, problem_file)
        
        print(f"\nPlanning Result:")
        print(f"  Success: {success}")
        print(f"  Plan: {plan}")
        
        # Cleanup
        os.unlink(domain_file)
        os.unlink(problem_file)
        
    except Exception as e:
        print(f"❌ Error: {e}")
        import traceback
        traceback.print_exc()

if __name__ == "__main__":
    debug_pddl_generation()