#!/usr/bin/env python3
"""
Test with a known working PDDL to make sure Fast Downward works
"""

import tempfile
import os
import sys
from pathlib import Path

# Add parent directory to path
sys.path.append(str(Path(__file__).parent))

from neuro_symbolic_planner import FastDownwardPlanner

def test_known_working_pddl():
    """Test with a known working PDDL"""
    
    print("🧪 Testing with known working PDDL...")
    
    # Simple working domain
    domain_pddl = """(define (domain simple-blocks)
  (:requirements :strips)
  (:predicates
    (on-table ?x)
    (holding ?x)
    (clear ?x)
  )
  (:action pick-up
    :parameters (?x)
    :precondition (and (clear ?x) (on-table ?x))
    :effect (and (holding ?x) (not (clear ?x)) (not (on-table ?x)))
  )
  (:action put-down
    :parameters (?x)
    :precondition (holding ?x)
    :effect (and (clear ?x) (on-table ?x) (not (holding ?x)))
  )
)"""

    # Simple working problem
    problem_pddl = """(define (problem simple-pickup)
  (:domain simple-blocks)
  (:objects a)
  (:init
    (clear a)
    (on-table a)
  )
  (:goal (holding a))
)"""

    print("Domain PDDL:")
    print(domain_pddl)
    print("\nProblem PDDL:")
    print(problem_pddl)
    
    # Test with Fast Downward
    planner = FastDownwardPlanner()
    
    # Save to temp files
    with tempfile.NamedTemporaryFile(mode='w', suffix='.pddl', delete=False) as f:
        f.write(domain_pddl)
        domain_file = f.name
        
    with tempfile.NamedTemporaryFile(mode='w', suffix='.pddl', delete=False) as f:
        f.write(problem_pddl)
        problem_file = f.name
    
    print(f"\nTesting with Fast Downward...")
    success, plan = planner.plan(domain_file, problem_file)
    
    print(f"Result:")
    print(f"  Success: {success}")
    print(f"  Plan: {plan}")
    
    # Cleanup
    os.unlink(domain_file)
    os.unlink(problem_file)
    
    return success, plan

if __name__ == "__main__":
    test_known_working_pddl()