#!/bin/bash

# List of templates to use for generating prompts
templates=("instruction")

# Sample size for generating prompts and responses; set to -1 to use the full dataset
sample_size=1000

# Output directory to store results
output_dir="output_sao"

# List of model paths to use for synthesis and scoring
model_paths=("google/gemma-2-9b-it")

# Create output directory if it doesn't exist
mkdir -p $output_dir

# Loop through each model and template combination
for model_path in "${model_paths[@]}"; do
  for template in "${templates[@]}"; do
    echo "Processing template: $template with model: $model_path"
    
    # Prepare model-specific output directory
    model_name=$(basename $model_path)
    model_output_dir="${output_dir}/${model_name}"
    mkdir -p $model_output_dir

    # Generate prompts and include sample size in the filename
    prompt_output="${model_output_dir}/vllm_${template}_synthesis_prompt_${sample_size}.jsonl"
    if [ -f "$prompt_output" ]; then
      echo "Prompt file already exists: $prompt_output. Skipping prompt generation."
    else
      python code/vllm_synthesize_prompt.py --model_path $model_path --template $template --sample_size $sample_size --output_path $prompt_output
    fi

    # Synthesize responses and include sample size in the filename
    response_output="${model_output_dir}/vllm_${template}_synthesis_response_${sample_size}.jsonl"
    if [ -f "$response_output" ]; then
      echo "Response file already exists: $response_output. Skipping response synthesis."
    else
      python code/vllm_synthesize_response.py --model_id $model_path --input_file $prompt_output --output_file $response_output --num_responses 2
    fi

    # Score responses and include sample size in the filename
    score_output="${model_output_dir}/vllm_${template}_synthesis_score_${sample_size}.jsonl"
    if [ -f "$score_output" ]; then
      echo "Score file already exists: $score_output. Skipping response scoring."
    else
      python code/vllm_synthesize_score.py --model_id $model_path --input_file_path $response_output --output_file "$score_output" --prompt_template_path "code/llm_as_a_judge_pairwise_prompt.txt"
    fi

    echo "Completed processing for template: $template with model: $model_path"
  done
done

echo "All tasks completed!"
