import numpy as np
import torch
import torch.nn as nn
from diff_models import diff_CSDI
import ot as pot
import math
from scipy import integrate
from torchdiffeq import odeint


class CSDI_base(nn.Module):
    def __init__(self, target_dim, config, device):
        super().__init__()
        self.device = device
        self.target_dim = target_dim

        self.emb_time_dim = config["model"]["timeemb"]
        self.emb_feature_dim = config["model"]["featureemb"]
        self.is_unconditional = config["model"]["is_unconditional"]
        self.target_strategy = config["model"]["target_strategy"]

        self.emb_total_dim = self.emb_time_dim + self.emb_feature_dim
        if self.is_unconditional == False:
            self.emb_total_dim += 1  # for conditional mask
        self.embed_layer = nn.Embedding(
            num_embeddings=self.target_dim, embedding_dim=self.emb_feature_dim
        )

        config_diff = config["diffusion"]
        config_diff["side_dim"] = self.emb_total_dim
        """
        modified to rectified
        """
        input_dim = 1 if self.is_unconditional == True else 2
        #input_dim = 1 
        
        self.diffmodel = diff_CSDI(config_diff, input_dim)
        #self.diffmodel_score = diff_CSDI(config_diff,1)
        self.diffmodel_score = diff_CSDI(config_diff, input_dim)
        
        
        # parameters for diffusion models
        self.num_steps = config_diff["num_steps"]
        if config_diff["schedule"] == "quad":
            self.beta = np.linspace(
                config_diff["beta_start"] ** 0.5, config_diff["beta_end"] ** 0.5, self.num_steps
            ) ** 2
        elif config_diff["schedule"] == "linear":
            self.beta = np.linspace(
                config_diff["beta_start"], config_diff["beta_end"], self.num_steps
            )
        
        self.sigma = 1e-3#1e-4
        
        #CSDI
        self.beta = np.linspace(0.0001, 0.5, self.num_steps)
        self.alpha_hat = 1 - self.beta
        self.alpha = np.cumprod(self.alpha_hat)
        self.alpha_torch = torch.tensor(self.alpha).float().to(self.device).unsqueeze(1).unsqueeze(1)
        
        
    def time_embedding(self, pos, d_model=128):
        pe = torch.zeros(pos.shape[0], pos.shape[1], d_model).to(self.device)
        position = pos.unsqueeze(2)
        div_term = 1 / torch.pow(
            10000.0, torch.arange(0, d_model, 2).to(self.device) / d_model
        )
        pe[:, :, 0::2] = torch.sin(position * div_term)
        pe[:, :, 1::2] = torch.cos(position * div_term)
        return pe

    def get_randmask(self, observed_mask):
        rand_for_mask = torch.rand_like(observed_mask) * observed_mask
        rand_for_mask = rand_for_mask.reshape(len(rand_for_mask), -1)
        for i in range(len(observed_mask)):
            sample_ratio = np.random.rand()  # missing ratio
            num_observed = observed_mask[i].sum().item()
            num_masked = round(num_observed * sample_ratio)
            rand_for_mask[i][rand_for_mask[i].topk(num_masked).indices] = -1
        cond_mask = (rand_for_mask > 0).reshape(observed_mask.shape).float()
        return cond_mask

    def get_hist_mask(self, observed_mask, for_pattern_mask=None):
        if for_pattern_mask is None:
            for_pattern_mask = observed_mask
        if self.target_strategy == "mix":
            rand_mask = self.get_randmask(observed_mask)

        cond_mask = observed_mask.clone()
        for i in range(len(cond_mask)):
            mask_choice = np.random.rand()
            if self.target_strategy == "mix" and mask_choice > 0.5:
                cond_mask[i] = rand_mask[i]
            else:  # draw another sample for histmask (i-1 corresponds to another sample)
                cond_mask[i] = cond_mask[i] * for_pattern_mask[i - 1] 
        return cond_mask

    def get_side_info(self, observed_tp, cond_mask):
        B, K, L = cond_mask.shape

        time_embed = self.time_embedding(observed_tp, self.emb_time_dim)  # (B,L,emb)
        time_embed = time_embed.unsqueeze(2).expand(-1, -1, K, -1)
        feature_embed = self.embed_layer(
            torch.arange(self.target_dim).to(self.device)
        )  # (K,emb)
        feature_embed = feature_embed.unsqueeze(0).unsqueeze(0).expand(B, L, -1, -1)

        side_info = torch.cat([time_embed, feature_embed], dim=-1)  # (B,L,K,*)
        side_info = side_info.permute(0, 3, 2, 1)  # (B,*,K,L)

        if self.is_unconditional == False:
            side_mask = cond_mask.unsqueeze(1)  # (B,1,K,L)
            side_info = torch.cat([side_info, side_mask], dim=1)
        return side_info

    
    def calc_loss_valid(
        self, observed_data, cond_mask, observed_mask, side_info, is_train
    ):
        loss_sum = 0
        for t in range(self.num_steps):  # calculate loss for all t
            loss = self.calc_loss(
                observed_data, cond_mask, observed_mask, side_info, is_train, set_t=t
            )
            loss_sum += loss.detach()
        return loss_sum / self.num_steps
 

    def compute_sigma_t(self, t):
        return self.sigma * torch.sqrt(t * (1 - t))


    def compute_lambda(self,t):
        sigma_t = self.compute_sigma_t(t)
        return 2 * sigma_t / (self.sigma**2 + 1e-8)
    
        
    def compute_conditional_flow(self, x0, x1, t, xt):
        t = pad_t_like_x(t, x0)
        mu_t = self.compute_mu_t(x0, x1, t)
        sigma_t_prime_over_sigma_t = (1 - 2 * t) / (2 * t * (1 - t) + 1e-8)
        ut = sigma_t_prime_over_sigma_t * (xt - mu_t) + x1 - x0
        return ut      
    
 
    def get_gamma(self,t):
        gamma = lambda t: torch.sqrt(t*(1-t))
        gamma_dot = lambda t: (1/(2*torch.sqrt(t*(1-t)))) * (1 -2*t)
        gg_dot = lambda t: (1/2)*(1-2*t)
        return gamma, gamma_dot, gg_dot
    
    def get_grad_h(self, kl, z_t):
        grad_h = torch.autograd.grad(kl.sum(), z_t, allow_unused=True,retain_graph=True)[0]
        return grad_h
   
    
    
    def calc_loss(
        self, observed_data, cond_mask, observed_mask, side_info, is_train, set_t=-1
    ):
        B, K, L = observed_data.shape
        target_mask = observed_mask - cond_mask
        
        z0 = 1e-1*(torch.rand(observed_data.shape).float().to(self.device))
        noise = 1e-2*(torch.rand(observed_data.shape).float().to(self.device))
        
        z1 = observed_data + 1e-3*(torch.rand(observed_data.shape).float().to(self.device))
        #t_rec = (1-1.e-3)*torch.rand([B,1]).to(self.device).requires_grad_(True)
        t_rec = (1-1.e-2)*torch.rand([B,1]).to(self.device).requires_grad_(True)
        t_rec_expand = t_rec.view(-1, 1, 1).repeat(1, K, L).float().to(self.device)
                        
        ### EMD 
        a, b = pot.unif(B), pot.unif(B) 
        M = torch.cdist((z0).reshape(B,K*L), (z1).reshape(B,K*L))** 2
        M = (M / M.max())
        pi = pot.emd(a, b, M.detach().cpu().numpy())
        #pi = pot.sinkhorn(a, b, M.detach().cpu().numpy(),0.01)
        # Sample random interpolations on pi
        p = pi.flatten()
        p = p / p.sum()
        choices = np.random.choice(pi.shape[0] * pi.shape[1], p=p, size=B)
        i, j = np.divmod(choices, pi.shape[1])
        z0 = z0[i]
        z1 = z1[j] 
        

        z_t = (t_rec_expand * z1)*(1-cond_mask) + (1.-t_rec_expand)*z0 #+ torch.sqrt(t_rec_expand*(1.-t_rec_expand))*noise
        
        
        cond_obs = (cond_mask * observed_data).unsqueeze(1)        
              
        
        target = (observed_data-noise)*(observed_mask-cond_mask)       
        total_input = torch.cat([cond_obs, z_t.unsqueeze(1)], dim=1)
  
        
        predicted = self.diffmodel(total_input, side_info, t_rec)

        
        num_eval = target_mask.sum()   
        residual = (target - predicted)*target_mask
        loss = residual**2
  

        loss = (loss.sum())/(num_eval if num_eval > 0 else 1)
        
        return loss

    
    def set_input_to_diffmodel(self, noisy_data, observed_data, cond_mask):
        if self.is_unconditional == True:
            total_input = noisy_data.unsqueeze(1)  # (B,1,K,L)
        else:
            cond_obs = (cond_mask * observed_data).unsqueeze(1)
            noisy_target = ((1 - cond_mask) * noisy_data).unsqueeze(1)
            total_input = torch.cat([cond_obs, noisy_target], dim=1)  # (B,2,K,L)

        return total_input

    
    def impute(self, observed_data, cond_mask, side_info, n_samples):
        B, K, L = observed_data.shape

        # CSDI: input: cat[noise, obs_data] 
        imputed_samples = torch.zeros(B, n_samples, K, L).to(self.device)      
        cond_obs = (cond_mask * observed_data).unsqueeze(1)
        dt = torch.Tensor([1./self.num_steps]).float().to(self.device)
        eps = torch.Tensor([1e-2]).float().to(self.device)
        
        
        for i in range(n_samples):
            x_0 = 1e-1*torch.rand(observed_data.shape).float().to(self.device).requires_grad_(True)#.detach().clone()  
            z_t = x_0.unsqueeze(1)            
            for j in range(self.num_steps):        
                t_diff = (torch.ones([B,1])*j / self.num_steps).float().to(self.device).requires_grad_(True)
                          
                total_input = torch.cat([cond_obs, z_t], dim=1)
                pred = self.diffmodel(total_input, side_info, t_diff)
                           
                pred = pred.unsqueeze(1)    
                
                """Euler"""
                z_t = z_t.detach().clone() + pred*dt

		"""potential function """
 		#_, _, _, z_recon = vae.forward(z_t)
                #recon_loss = (torch.exp((z_recon.unsqueeze(1)-z_t)**2)*(target_mask))
                #z_t = z_t+1e-2*(z_recon.unsqueeze(1)-z_t)*(1-cond_mask).unsqueeze(1)*dt

                #"""Euler-Maruyama"""
                # dW = torch.sqrt(dt)*torch.randn(size=pred.shape).to(self.device).detach().clone()  
                #z_t = z_t.detach().clone() + pred * dt + torch.sqrt(2*eps)*dW
         
            imputed_samples[:, i] = z_t.squeeze(1).detach()
        return imputed_samples    
    



    def forward(self, batch, is_train=1):
        (
            observed_data,
            observed_mask,
            observed_tp,
            gt_mask,
            for_pattern_mask,
            _,
        ) = self.process_data(batch)
        if is_train == 0:
            cond_mask = gt_mask
        elif self.target_strategy != "random":
            cond_mask = self.get_hist_mask(
                observed_mask, for_pattern_mask=for_pattern_mask
            )
        else:
            cond_mask = self.get_randmask(observed_mask)

        side_info = self.get_side_info(observed_tp, cond_mask)

        loss_func = self.calc_loss if is_train == 1 else self.calc_loss_valid

        return loss_func(observed_data, cond_mask, observed_mask, side_info, is_train)

    def evaluate(self, batch, n_samples):
        (
            observed_data,
            observed_mask,
            observed_tp,
            gt_mask,
            _,
            cut_length,
        ) = self.process_data(batch)

        #with torch.no_grad():
        with torch.enable_grad():
            cond_mask = gt_mask
            target_mask = observed_mask - cond_mask

            side_info = self.get_side_info(observed_tp, cond_mask)
            #samples = self.impute_rk45(observed_data, cond_mask, side_info, n_samples)
            samples = self.impute(observed_data, cond_mask, side_info, n_samples)

            for i in range(len(cut_length)):  # to avoid double evaluation
                target_mask[i, ..., 0 : cut_length[i].item()] = 0
        return samples, observed_data, target_mask, observed_mask, observed_tp, cond_mask


class CSDI_PM25(CSDI_base):
    def __init__(self, config, device, target_dim=36):
        super(CSDI_PM25, self).__init__(target_dim, config, device)

    def process_data(self, batch):
        observed_data = batch["observed_data"].to(self.device).float()
        observed_mask = batch["observed_mask"].to(self.device).float()
        observed_tp = batch["timepoints"].to(self.device).float()
        gt_mask = batch["gt_mask"].to(self.device).float()
        cut_length = batch["cut_length"].to(self.device).long()
        for_pattern_mask = batch["hist_mask"].to(self.device).float()

        observed_data = observed_data.permute(0, 2, 1)
        observed_mask = observed_mask.permute(0, 2, 1)
        gt_mask = gt_mask.permute(0, 2, 1)
        for_pattern_mask = for_pattern_mask.permute(0, 2, 1)

        return (
            observed_data,
            observed_mask,
            observed_tp,
            gt_mask,
            for_pattern_mask,
            cut_length,
        )


class CSDI_Physio(CSDI_base):
    def __init__(self, config, device, target_dim=35):
        super(CSDI_Physio, self).__init__(target_dim, config, device)

    def process_data(self, batch):
        observed_data = batch["observed_data"].to(self.device).float()
        observed_mask = batch["observed_mask"].to(self.device).float()
        observed_tp = batch["timepoints"].to(self.device).float()
        gt_mask = batch["gt_mask"].to(self.device).float()

        observed_data = observed_data.permute(0, 2, 1)
        observed_mask = observed_mask.permute(0, 2, 1)
        gt_mask = gt_mask.permute(0, 2, 1)

        cut_length = torch.zeros(len(observed_data)).long().to(self.device)
        for_pattern_mask = observed_mask

        return (
            observed_data,
            observed_mask,
            observed_tp,
            gt_mask,
            for_pattern_mask,
            cut_length,
        )
