import torch.nn as nn
import torch.nn.functional as F
import numpy as np
import torch
from custom_modules import *

__all__ = ['resnet18_fp', 'resnet50_fp', 'resnet18_quant_allQ',
           'resnet50_quant_allQ', 'resnet18_quant', 'resnet50_quant']

class LambdaLayer(nn.Module):
    def __init__(self, lambd):
        super(LambdaLayer, self).__init__()
        self.lambd = lambd

    def forward(self, x):
        return self.lambd(x)


import torch
import torch.nn as nn
import torch.nn.functional as F


def _weights_init(m):
    classname = m.__class__.__name__
    if isinstance(m, nn.Linear) or isinstance(m, nn.Conv2d) or isinstance(m, QConv_frozen):
        nn.init.kaiming_normal_(m.weight)

class BasicBlock(nn.Module):
    expansion = 1

    def __init__(self, in_planes, planes, args, stride=1):
        super(BasicBlock, self).__init__()
        self.conv1 = nn.Conv2d(
            in_planes, planes, kernel_size=3, stride=stride, padding=1, bias=False)
        self.bn1 = nn.BatchNorm2d(planes)
        self.conv2 = nn.Conv2d(planes, planes, kernel_size=3,
                               stride=1, padding=1, bias=False)
        self.bn2 = nn.BatchNorm2d(planes)

        self.downsample = nn.Sequential()
        if stride != 1 or in_planes != self.expansion*planes:
            self.downsample = nn.Sequential(
                nn.Conv2d(in_planes, self.expansion*planes,
                          kernel_size=1, stride=stride, bias=False),
                nn.BatchNorm2d(self.expansion*planes)
            )

    def forward(self, x):
        out = F.relu(self.bn1(self.conv1(x)))
        out = self.bn2(self.conv2(out))
        out += self.downsample(x)
        out = F.relu(out)
        return out

class BasicBlock_quant_frozen(nn.Module):
    expansion = 1

    def __init__(self, in_planes, planes, args, stride=1):
        super(BasicBlock_quant_frozen, self).__init__()
        self.conv1 = QConv_frozen(in_planes, planes, kernel_size=3, stride=stride, padding=1, bias=False, args=args)
        # self.conv1 = nn.Conv2d(
        #     in_planes, planes, kernel_size=3, stride=stride, padding=1, bias=False)
        self.bn1 = nn.BatchNorm2d(planes)
        self.conv2 = QConv_frozen(planes, planes, kernel_size=3, stride=1, padding=1, bias=False, args=args)
        # self.conv2 = nn.Conv2d(planes, planes, kernel_size=3,
        #                        stride=1, padding=1, bias=False)
        self.bn2 = nn.BatchNorm2d(planes)

        self.downsample = nn.Sequential()
        if stride != 1 or in_planes != self.expansion*planes:
            self.downsample = nn.Sequential(
                # nn.Conv2d(in_planes, self.expansion*planes,
                #           kernel_size=1, stride=stride, bias=False),
                QConv_frozen(in_planes, self.expansion * planes, kernel_size=1, stride=stride, bias=False, args=args),
                nn.BatchNorm2d(self.expansion*planes)
            )

    def forward(self, x):
        out = F.relu(self.bn1(self.conv1(x)))
        out = self.bn2(self.conv2(out))
        out += self.downsample(x)
        out = F.relu(out)
        return out

class Bottleneck(nn.Module):
    expansion = 4

    def __init__(self, in_planes, planes, args, stride=1):
        super(Bottleneck, self).__init__()
        self.conv1 = nn.Conv2d(in_planes, planes, kernel_size=1, bias=False)
        self.bn1 = nn.BatchNorm2d(planes)
        self.conv2 = nn.Conv2d(planes, planes, kernel_size=3,
                               stride=stride, padding=1, bias=False)
        self.bn2 = nn.BatchNorm2d(planes)
        self.conv3 = nn.Conv2d(planes, self.expansion *
                               planes, kernel_size=1, bias=False)
        self.bn3 = nn.BatchNorm2d(self.expansion*planes)

        self.downsample = nn.Sequential()
        if stride != 1 or in_planes != self.expansion*planes:
            self.downsample = nn.Sequential(
                nn.Conv2d(in_planes, self.expansion*planes,
                          kernel_size=1, stride=stride, bias=False),
                nn.BatchNorm2d(self.expansion*planes)
            )

    def forward(self, x):
        out = F.relu(self.bn1(self.conv1(x)))
        out = F.relu(self.bn2(self.conv2(out)))
        out = self.bn3(self.conv3(out))
        out += self.downsample(x)
        out = F.relu(out)
        return out

class Bottleneck_quant_frozen(nn.Module):
    expansion = 4

    def __init__(self, in_planes, planes, args, stride=1):
        super(Bottleneck_quant_frozen, self).__init__()
        self.conv1 = QConv_frozen(in_planes, planes, kernel_size=1, bias=False, args=args)
        # self.conv1 = nn.Conv2d(in_planes, planes, kernel_size=1, bias=False)
        self.bn1 = nn.BatchNorm2d(planes)
        self.conv2 = QConv_frozen(planes, planes, kernel_size=3,
                               stride=stride, padding=1, bias=False, args=args)
        # self.conv2 = nn.Conv2d(planes, planes, kernel_size=3,
        #                        stride=stride, padding=1, bias=False)
        self.bn2 = nn.BatchNorm2d(planes)
        self.conv3 = QConv_frozen(planes, self.expansion *
                               planes, kernel_size=1, bias=False, args=args)
        # self.conv3 = nn.Conv2d(planes, self.expansion *
        #                        planes, kernel_size=1, bias=False)
        self.bn3 = nn.BatchNorm2d(self.expansion*planes)

        self.downsample = nn.Sequential()
        if stride != 1 or in_planes != self.expansion*planes:
            self.downsample = nn.Sequential(
                QConv_frozen(in_planes, self.expansion*planes,
                          kernel_size=1, stride=stride, bias=False, args=args),
                # nn.Conv2d(in_planes, self.expansion*planes,
                #           kernel_size=1, stride=stride, bias=False),
                nn.BatchNorm2d(self.expansion*planes)
            )

    def forward(self, x):
        out = F.relu(self.bn1(self.conv1(x)))
        out = F.relu(self.bn2(self.conv2(out)))
        out = self.bn3(self.conv3(out))
        out += self.downsample(x)
        out = F.relu(out)
        return out

class ResNet(nn.Module):
    def __init__(self, block, num_blocks, args=None, num_classes=1000):
        super(ResNet, self).__init__()
        self.in_planes = 64
        self.args = args

        self.conv1 = nn.Conv2d(3, 64, kernel_size=7, stride=2, padding=3, bias=False)
        self.maxpool = nn.MaxPool2d(kernel_size=3, stride=2, padding=1)
        self.bn1 = nn.BatchNorm2d(64)
        self.layer1 = self._make_layer(block, 64, num_blocks[0], stride=1)
        self.layer2 = self._make_layer(block, 128, num_blocks[1], stride=2)
        self.layer3 = self._make_layer(block, 256, num_blocks[2], stride=2)
        self.layer4 = self._make_layer(block, 512, num_blocks[3], stride=2)
        self.bn2 = nn.BatchNorm1d(512*block.expansion)
        self.linear = nn.Linear(512*block.expansion, num_classes)

        self.apply(_weights_init)

    def _make_layer(self, block, planes, num_blocks, stride):
        strides = [stride] + [1]*(num_blocks-1)
        layers = []
        for stride in strides:
            layers.append(block(self.in_planes, planes, self.args, stride))
            self.in_planes = planes * block.expansion
        return nn.Sequential(*layers)

    def forward(self, x):
        out = self.maxpool(F.relu(self.bn1(self.conv1(x))))
        out = self.layer1(out)
        out = self.layer2(out)
        out = self.layer3(out)
        out = self.layer4(out)
        out = F.avg_pool2d(out, 7)
        out = out.view(out.size(0), -1)
        out = self.bn2(out)
        out = self.linear(out)
        return out

class ResNet_allQ(nn.Module):
    def __init__(self, block, num_blocks, args=None, num_classes=1000):
        super(ResNet_allQ, self).__init__()
        self.in_planes = 64
        self.args = args

        self.conv1 = QConv_frozen_first(3, 64, kernel_size=7, stride=2, padding=3, bias=False, args=args)
        self.maxpool = nn.MaxPool2d(kernel_size=3, stride=2, padding=1)
        self.bn1 = nn.BatchNorm2d(64)
        self.layer1 = self._make_layer(block, 64, num_blocks[0], stride=1)
        self.layer2 = self._make_layer(block, 128, num_blocks[1], stride=2)
        self.layer3 = self._make_layer(block, 256, num_blocks[2], stride=2)
        self.layer4 = self._make_layer(block, 512, num_blocks[3], stride=2)
        self.bn2 = nn.BatchNorm1d(512*block.expansion)
        # self.linear = nn.Linear(512*block.expansion, num_classes)
        self.linear = QLinear(512*block.expansion, num_classes, args, bias=True)

        self.apply(_weights_init)

    def _make_layer(self, block, planes, num_blocks, stride):
        strides = [stride] + [1]*(num_blocks-1)
        layers = []
        for stride in strides:
            layers.append(block(self.in_planes, planes, self.args, stride))
            self.in_planes = planes * block.expansion
        return nn.Sequential(*layers)

    def forward(self, x):
        out = self.maxpool(F.relu(self.bn1(self.conv1(x))))
        out = self.layer1(out)
        out = self.layer2(out)
        out = self.layer3(out)
        out = self.layer4(out)
        out = F.avg_pool2d(out, 7)
        out = out.view(out.size(0), -1)
        out = self.bn2(out)
        out = self.linear(out)
        return out

def resnet18_fp(args):
    return ResNet(BasicBlock, [2, 2, 2, 2], args=args)

def resnet50_fp(args):
    return ResNet(Bottleneck, [3, 4, 6, 3], args=args)

def resnet18_quant_allQ(args):
    return ResNet_allQ(BasicBlock_quant_frozen, [2, 2, 2, 2], args=args)

def resnet50_quant_allQ(args):
    return ResNet_allQ(Bottleneck_quant_frozen, [3, 4, 6, 3], args=args)

def resnet18_quant(args):
    return ResNet(BasicBlock_quant_frozen, [2, 2, 2, 2], args=args)

def resnet50_quant(args):
    return ResNet(Bottleneck_quant_frozen, [3, 4, 6, 3], args=args)

