#lang racket

(define (minimum-time n edges disappearing-nodes)
  (define adj (make-vector n '()))
  (for ([edge edges])
    (match-define (list u v t) edge)
    (vector-set! adj u (cons (cons v t) (vector-ref adj u)))
    (vector-set! adj v (cons (cons u t) (vector-ref adj v))))

  (define dist (make-vector n +inf.0))
  (vector-set! dist 0 0)

  (define heap (make-heap (lambda (a b) (< (car a) (car b)))))
  (heap-add! heap (cons 0 0))

  (define (dijkstra)
    (let loop ()
      (unless (heap-empty? heap)
        (match-define (cons time u) (heap-min heap))
        (heap-remove-min! heap)
        (when (<= time (vector-ref disappearing-nodes u))
          (for ([neighbor (vector-ref adj u)])
            (match-define (cons v t) neighbor)
            (define new-time (+ time t))
            (when (and (< new-time (vector-ref dist v))
                     (<= new-time (vector-ref disappearing-nodes v)))
              (vector-set! dist v new-time)
              (heap-add! heap (cons new-time v)))))
        (loop))))

  (dijkstra)

  (define result (make-vector n -1))
  (for ([i (in-range n)])
    (when (< (vector-ref dist i) +inf.0)
      (vector-set! result i (vector-ref dist i))))
  (vector->list result))

(define-struct heap (cmp vec size))

(define (make-heap cmp)
  (heap cmp (make-vector 16 #f) 0))

(define (heap-add! heap item)
  (define vec (heap-vec heap))
  (define size (heap-size heap))
  (when (>= size (vector-length vec))
    (define new-vec (make-vector (* 2 (vector-length vec)) #f))
    (vector-copy! new-vec 0 vec)
    (set-heap-vec! heap new-vec))
  (vector-set! vec size item)
  (set-heap-size! heap (add1 size))
  (sift-up heap (sub1 (heap-size heap))))

(define (sift-up heap idx)
  (define vec (heap-vec heap))
  (define cmp (heap-cmp heap))
  (let loop ([i idx])
    (when (> i 0)
      (define parent (quotient (sub1 i) 2))
      (when (cmp (vector-ref vec i) (vector-ref vec parent))
        (define tmp (vector-ref vec i))
        (vector-set! vec i (vector-ref vec parent))
        (vector-set! vec parent tmp)
        (loop parent)))))

(define (heap-min heap)
  (vector-ref (heap-vec heap) 0))

(define (heap-remove-min! heap)
  (define vec (heap-vec heap))
  (define size (heap-size heap))
  (vector-set! vec 0 (vector-ref vec (sub1 size)))
  (set-heap-size! heap (sub1 size))
  (sift-down heap 0))

(define (sift-down heap idx)
  (define vec (heap-vec heap))
  (define cmp (heap-cmp heap))
  (define size (heap-size heap))
  (let loop ([i idx])
    (define left (+ (* 2 i) 1))
    (define right (+ (* 2 i) 2))
    (define smallest i)
    (when (and (< left size) (cmp (vector-ref vec left) (vector-ref vec smallest)))
      (set! smallest left))
    (when (and (< right size) (cmp (vector-ref vec right) (vector-ref vec smallest)))
      (set! smallest right))
    (unless (= smallest i)
      (define tmp (vector-ref vec i))
      (vector-set! vec i (vector-ref vec smallest))
      (vector-set! vec smallest tmp)
      (loop smallest)))))

(define (heap-empty? heap)
  (zero? (heap-size heap)))