import os
from tqdm import tqdm


def format_interval(t):
    """
    Format seconds into H:MM:SS or M:SS (same as tqdm).
    Example: 3661.7 -> '1:01:01'
    """
    mins, s = divmod(int(t), 60)
    h, m = divmod(mins, 60)
    if h:
        return f"{h}:{m:02d}:{s:02d}"
    else:
        return f"{m}:{s:02d}"


class SilentTqdm:
    def __init__(self, total, desc=None, initial=0):
        self._bar = tqdm(
            total=total,
            initial=initial,
            desc=desc,
            file=open(os.devnull, "w"),  # 屏蔽输出
            dynamic_ncols=False,
        )

    def update(self, n=1):
        self._bar.update(n)

    @property
    def n(self):
        return self._bar.format_dict.get("n", 0)

    @property
    def total(self):
        return self._bar.total

    @property
    def elapsed(self):
        return self._bar.format_dict.get("elapsed", 0.0)

    @property
    def rate(self):
        return self._bar.format_dict.get("rate", 0.0)

    @property
    def eta(self):
        if self.rate > 0 and self.n is not None:
            return (self.total - self.n) / self.rate
        return float("inf")

    def status(self, fmt="{n}/{total} | Elapsed: {elapsed} | ETA: {eta} | Rate: {rate:.2f} it/s"):
        return fmt.format(
            n=self.n,
            total=self.total,
            elapsed=format_interval(self.elapsed),
            eta=format_interval(self.eta),
            rate=self.rate,
        )

    def close(self):
        self._bar.close()

    def __del__(self):
        self.close()
