import os
import random
import time
from dataclasses import dataclass, field
import gymnasium as gym
import numpy as np
import torch
import torch.nn as nn
import torch.nn.functional as F
import tyro
from stable_baselines3.common.buffers import ReplayBuffer
from torch.utils.tensorboard import SummaryWriter
import metaworld
from torch.optim import Optimizer
from typing import List, Union, Tuple
from torch import Tensor
import math

@dataclass
class Args:
    exp_name: str = os.path.basename(__file__)[: -len(".py")]
    """the name of this experiment"""
    seed: int = 1
    """seed of the experiment"""
    torch_deterministic: bool = True
    """if toggled, `torch.backends.cudnn.deterministic=False`"""
    cuda: bool = True
    """if toggled, cuda will be enabled by default"""

    # Algorithm specific arguments
    env_id: str = "mt4"
    """the environment id of the task"""
    env_names: list = field(default_factory=lambda: ['window-open-v2', 'window-close-v2', 'drawer-open-v2', 'drawer-close-v2'])
    """the environment id of the task"""
    total_timesteps: int = 500000
    """total timesteps of the experiments"""
    test_timesteps: int = 100000
    """test timesteps of the experiments"""
    buffer_size: int = int(1e6)
    """the replay memory buffer size"""
    gamma: float = 0.99
    """the discount factor gamma"""
    tau: float = 0.005
    """target smoothing coefficient (default: 0.005)"""
    batch_size: int = 256 # 1280
    """the batch size of sample from the reply memory"""
    learning_starts: int = 1e4
    """timestep to start learning"""
    policy_lr: float = 3e-4
    """the learning rate of the policy network optimizer"""
    q_lr: float = 1e-3
    """the learning rate of the Q network network optimizer"""
    policy_frequency: int = 2
    """the frequency of training policy (delayed)"""
    target_network_frequency: int = 1
    """the frequency of updates for the target nerworks"""
    noise_clip: float = 0.5
    """noise clip parameter of the Target Policy Smoothing Regularization"""
    alpha: float = 0.2
    """Entropy regularization coefficient."""
    autotune: bool = True
    """automatic tuning of the entropy coefficient"""
    average_num: int = 100
    """the number of episode to compute average success rate"""
    threshold: float = 0.9
    """the threshold representing finishing training"""
    frozen_rate: list = field(default_factory=lambda: [0.2, 0.2, 0.2, 0.2])
    """the number of RL skill neurons"""
    score_coef: float = 0.1
    """Score coefficient."""
    replay_interval: int = 10
    """the interval of replay the previous experience"""
    cycle_num: int = 2
    """the number of cycling"""

def make_env(env_name, gamma):
    def thunk():
        mt1 = metaworld.MT1(env_name) # Construct the benchmark, sampling tasks
        env = mt1.train_classes[env_name]()  # Create an environment with task `pick_place`
        task = mt1.train_tasks[0]
        env.set_task(task)  # Set task
        env = gym.wrappers.RecordEpisodeStatistics(env)
        return env
    return thunk


# ALGO LOGIC: initialize agent here:
class SoftQNetwork(nn.Module):
    def __init__(self, env):
        super().__init__()
        self.fc1 = nn.Linear(np.array(env.single_observation_space.shape).prod() + np.prod(env.single_action_space.shape), 768)
        self.fc2 = nn.Linear(768, 768)
        self.fc3 = nn.Linear(768, 768)
        self.fc4 = nn.Linear(768, 1)
        self.layers = [self.fc1, self.fc2, self.fc3]

    def forward(self, x, a):
        x = torch.cat([x, a], 1)
        x = F.relu(self.fc1(x))
        x = F.relu(self.fc2(x))
        x = F.relu(self.fc3(x))
        x = self.fc4(x)
        return x

    def forward_with_feature(self, x, a):
        x = torch.cat([x, a], 1)
        x1 = F.relu(self.fc1(x))
        x2 = F.relu(self.fc2(x1))
        x3 = F.relu(self.fc3(x2))
        x4 = self.fc4(x3)
        return x4, [x1, x2, x3]

LOG_STD_MAX = 10
LOG_STD_MIN = -10

class Actor(nn.Module):
    def __init__(self, env):
        super().__init__()
        self.fc1 = nn.Linear(np.array(env.single_observation_space.shape).prod(), 768)
        self.fc2 = nn.Linear(768, 768)
        self.fc3 = nn.Linear(768, 768)
        self.fc_mean = nn.Linear(768, np.prod(env.single_action_space.shape))
        self.fc_logstd = nn.Linear(768, np.prod(env.single_action_space.shape))
        # action rescaling
        self.register_buffer(
            "action_scale", torch.tensor((env.action_space.high - env.action_space.low) / 2.0, dtype=torch.float32)
        )
        self.register_buffer(
            "action_bias", torch.tensor((env.action_space.high + env.action_space.low) / 2.0, dtype=torch.float32)
        )
        self.layers = [self.fc1, self.fc2, self.fc_mean, self.fc_logstd]

    def forward(self, x):
        x1 = F.relu(self.fc1(x))
        x2 = F.relu(self.fc2(x1))
        x3 = F.relu(self.fc3(x2))
        mean = self.fc_mean(x3)
        log_std = self.fc_logstd(x3)
        log_std = torch.tanh(log_std)
        log_std = LOG_STD_MIN + 0.5 * (LOG_STD_MAX - LOG_STD_MIN) * (log_std + 1)  # From SpinUp / Denis Yarats

        return mean, log_std, [x1, x2, x3]

    def get_action(self, x):
        mean, log_std, _ = self(x)
        std = log_std.exp()
        normal = torch.distributions.Normal(mean, std)
        x_t = normal.rsample()  # for reparameterization trick (mean + std * N(0,1))
        y_t = torch.tanh(x_t)
        action = y_t * self.action_scale + self.action_bias
        log_prob = normal.log_prob(x_t)
        # Enforcing Action Bound
        log_prob -= torch.log(self.action_scale * (1 - y_t.pow(2)) + 1e-6)
        log_prob = log_prob.sum(1, keepdim=True)
        mean = torch.tanh(mean) * self.action_scale + self.action_bias
        return action, log_prob, mean
    
    def get_action_with_feature(self, x):
        mean, log_std, feature = self(x)
        std = log_std.exp()
        normal = torch.distributions.Normal(mean, std)
        x_t = normal.rsample()  # for reparameterization trick (mean + std * N(0,1))
        y_t = torch.tanh(x_t)
        action = y_t * self.action_scale + self.action_bias
        log_prob = normal.log_prob(x_t)
        # Enforcing Action Bound
        log_prob -= torch.log(self.action_scale * (1 - y_t.pow(2)) + 1e-6)
        log_prob = log_prob.sum(1, keepdim=True)
        mean = torch.tanh(mean) * self.action_scale + self.action_bias
        return action, log_prob, mean, feature

def adam_step(params: List[Tensor],
         grads: List[Tensor],
         exp_avgs: List[Tensor],
         exp_avg_sqs: List[Tensor],
         neuron_lrs: List[Tensor],
         state_steps: List[Tensor],
         beta1: float,
         beta2: float,
         lr: Union[float, Tensor],
         weight_decay: float,
         eps: float,
         ):
    for i, param in enumerate(params):
        exp_avg = exp_avgs[i]
        exp_avg_sq = exp_avg_sqs[i]
        step_t = state_steps[i]
        neuron_lr = neuron_lrs[i]
        grad = grads[i] * neuron_lr

        # update step
        step_t += 1

        if weight_decay != 0:
            grad = grad.add(param, alpha=weight_decay)
        
        # Decay the first and second moment running average coefficient
        exp_avg.lerp_(grad, 1 - beta1)
        exp_avg_sq.mul_(beta2).addcmul_(grad, grad.conj(), value=1 - beta2)
        
        step = step_t.item()

        bias_correction1 = 1 - beta1 ** step
        bias_correction2 = 1 - beta2 ** step

        step_size = lr / bias_correction1

        bias_correction2_sqrt = math.sqrt(bias_correction2)

        denom = (exp_avg_sq.sqrt() / bias_correction2_sqrt).add_(eps)

        param.addcdiv_(exp_avg, denom, value=-step_size)
         

class Adam(Optimizer):
    def __init__(self,
                 params,
                 lr: Union[float, Tensor] = 1e-3,
                 betas: Tuple[float, float] = (0.99, 0.99),
                 eps: float = 1e-8,
                 weight_decay: float = 0,
                 ):

        defaults = dict(lr=lr, betas=betas, eps=eps,
                        weight_decay=weight_decay)
        super().__init__(params, defaults)
    
    def _init_group(
        self,
        group,
        params_with_grad,
        grads,
        exp_avgs,
        exp_avg_sqs,
        state_steps,
        neuron_lrs
    ):
        for p in group['params']:
            if p.grad is not None:
                params_with_grad.append(p)
                grads.append(p.grad)

                state = self.state[p]
                # Lazy state initialization
                if len(state) == 0:
                    state['step'] = torch.tensor(0.)
                    # Exponential moving average of gradient values
                    state['exp_avg'] = torch.zeros_like(p, memory_format=torch.preserve_format)
                    # Exponential moving average of squared gradient values
                    state['exp_avg_sq'] = torch.zeros_like(p, memory_format=torch.preserve_format)
                    state['exp_avg_sq'] = torch.zeros_like(p, memory_format=torch.preserve_format)
                    state['lr'] = torch.ones_like(p, memory_format=torch.preserve_format)

                exp_avgs.append(state['exp_avg'])
                exp_avg_sqs.append(state['exp_avg_sq'])
                neuron_lrs.append(state['lr'])
                state_steps.append(state['step'])
  
    def set_lr(self, skill_neurons, skill_scores, task_index, num_task, score_coef):
        for group in self.param_groups:
            index = 0
            for p in group['params']:
                if p.grad is not None and index < 6:
                    state = self.state[p]
                    if index % 2 == 0:
                        state['lr'][:,:] = 1.0
                        length = torch.ones_like(p.grad)
                        for task in range(num_task):
                            if task != task_index and skill_neurons[task] != []:
                                for i in range(len(skill_neurons[task][index//2])): 
                                    if state['lr'][skill_neurons[task][index//2][i],0] == 1.0:
                                        state['lr'][skill_neurons[task][index//2][i],:] = (score_coef * (1 - skill_scores[task][index//2][i]))
                                    else:
                                        state['lr'][skill_neurons[task][index//2][i],:] += (score_coef * (1 - skill_scores[task][index//2][i]))
                                    length[skill_neurons[task][index//2][i],:] += 1
                        state['lr'] /= length
                    else:
                        state['lr'][:] = 1.0
                        length = torch.ones_like(p.grad)
                        for task in range(num_task):
                            if task != task_index and skill_neurons[task] != []:
                                for i in range(len(skill_neurons[task][index//2])):
                                    if state['lr'][skill_neurons[task][index//2][i]] == 1.0:
                                        state['lr'][skill_neurons[task][index//2][i]] = (score_coef * (1 - skill_scores[task][index//2][i]))
                                    else:
                                        state['lr'][skill_neurons[task][index//2][i]] += (score_coef * (1 - skill_scores[task][index//2][i]))
                                    length[skill_neurons[task][index//2][i]] += 1
                        state['lr'] /= length
                index += 1
                
    @torch.no_grad()       
    def step(self, closure=None):
        for group in self.param_groups:
            params_with_grad = []
            grads = []
            exp_avgs = []
            exp_avg_sqs = []
            state_steps = []
            neuron_lrs = []
            beta1, beta2 = group['betas']

            self._init_group(
                group,
                params_with_grad,
                grads,
                exp_avgs,
                exp_avg_sqs,
                state_steps,
                neuron_lrs)

            adam_step(
                params_with_grad,
                grads,
                exp_avgs,
                exp_avg_sqs,
                neuron_lrs,
                state_steps,
                beta1=beta1,
                beta2=beta2,
                lr=group['lr'],
                weight_decay=group['weight_decay'],
                eps=group['eps'],
            )
     
    
def update_top(top, add_top):
    for i in range(len(top)):
        top[i] = list(set(top[i]).union(add_top[i]))
    return top


def merge_buffers(rb1, rb2):
    if rb1.full:
        for idx in range(max(0, rb1.buffer_size - 100000), rb1.buffer_size):
            rb2.add(
            rb1.observations[idx],
            rb1.next_observations[idx],
            rb1.actions[idx],
            rb1.rewards[idx],
            rb1.dones[idx],
            []
        )
    else:
        for idx in range(max(0, rb1.pos - 100000), rb1.pos):
            rb2.add(
                rb1.observations[idx],
                rb1.next_observations[idx],
                rb1.actions[idx],
                rb1.rewards[idx],
                rb1.dones[idx],
                []
            )
    return rb2

def rank_top(masks, eval_episode, k):
    masks_sum = torch.cat(masks, dim=0)
    masks_sum /= eval_episode

    neg_masks = torch.ones_like(masks_sum) - masks_sum
    scores = torch.maximum(neg_masks, masks_sum)
    top_result = torch.topk(scores, k, dim=0, largest=True)
    top_value = top_result[0]
    top_sum = top_result[1]

    # Prepare for dynamically handling mask splits based on shape
    top = [[] for _ in range(len(masks))]
    score = [[] for _ in range(len(masks))]
    split_points = [0] + list(torch.cumsum(torch.tensor([mask.shape[0] for mask in masks]), 0))

    # Distribute the top indices into the appropriate group
    for j in range(len(top_sum)):
        index = top_sum[j]
        value = top_value[j]

        for i in range(1, len(split_points)):
            if split_points[i - 1] <= index < split_points[i]:
                top[i - 1].append(index - split_points[i - 1])
                score[i-1].append(value)
                break
    return top, score

def eval_neuron_critic(qf1_features, qf1_values, qf2_features, qf2_values, rb, num, eval_episode, k):
    qf1_feature_ave = []
    qf2_feature_ave = []
    qf1_masks = []
    qf2_masks = []

    for feature in qf1_features:
        qf1_feature_ave.append(feature / num)
        qf1_masks.append(torch.zeros_like(feature))
    for feature in qf2_features:
        qf2_feature_ave.append(feature / num)
        qf2_masks.append(torch.zeros_like(feature))

    qf1_values_ave = qf1_values * 1.0 / num
    qf2_values_ave = qf2_values * 1.0 / num

    for i in range(eval_episode):
        data = rb.sample(1)

        qf1_a_values, qf1_feature = qf1.forward_with_feature(data.observations.to(torch.float32),
                                                             data.actions.to(torch.float32))
        qf2_a_values, qf2_feature = qf2.forward_with_feature(data.observations.to(torch.float32),
                                                             data.actions.to(torch.float32))

        for i in range(len(qf1_feature)):
            mask = ((qf1_feature[i][0].cpu().detach() > qf1_feature_ave[i]).int() == (
                        qf1_a_values[0].cpu().detach() > qf1_values_ave)).float()
            qf1_masks[i] += mask

            mask = ((qf2_feature[i][0].cpu().detach() > qf2_feature_ave[i]).int() == (
                        qf2_a_values[0].cpu().detach() > qf2_values_ave)).float()
            qf2_masks[i] += mask

    qf1_top, qf1_score = rank_top(qf1_masks, eval_episode, k)
    qf2_top, qf2_score = rank_top(qf2_masks, eval_episode, k)
    return qf1_top, qf1_score, qf2_top, qf2_score

def eval_neuron_actor(base_features, envs, num, eval_episode, device, k):
    feature_ave = []
    features = []
    masks = []
    for feature in base_features:
        feature_ave.append(feature / num)
        features.append(torch.zeros_like(feature))
        masks.append(torch.zeros_like(feature))

    for i in range(eval_episode):
        obs, _ = envs.reset(seed=args.seed)
        done = 0
        while done == 0:
            with torch.no_grad():
                actions, _, _, feature = actor.get_action_with_feature(torch.Tensor(obs).to(device))
                actions = actions.detach().cpu().numpy()

            for i in range(len(feature)):
                features[i] += feature[i].to("cpu")

            # execute the task and log data.
            next_obs, rewards, terminations, truncations, infos = envs.step(actions)

            # record rewards for plotting purposes
            if "final_info" in infos:
                for info in infos["final_info"]:
                    done = 1
                    for i in range(len(features)):
                        features[i] /= info["episode"]["l"]
                        mask = ((features[i] > feature_ave[i]).int() == torch.Tensor(
                            [info['success'] == 1])).float()
                        masks[i] += mask

            obs = next_obs

    top, score = rank_top(masks, eval_episode, k)

    return top, score

def train_env(envs, device, args, env_name, task_index, rb_pre, frozen_rate, checkpoint):
    if checkpoint is None:
        # Automatic entropy tuning
        if args.autotune:
            target_entropy = -torch.prod(torch.Tensor(envs.single_action_space.shape).to(device)).item()
            log_alpha = torch.zeros(1, requires_grad=True, device=device)
            alpha = log_alpha.exp().item()
            a_optimizer = Adam([log_alpha], lr=args.q_lr)
        else:
            alpha = args.alpha
        envs.single_observation_space.dtype = np.float32
        rb = ReplayBuffer(
            args.buffer_size,
            envs.single_observation_space,
            envs.single_action_space,
            device,
            handle_timeout_termination=False,
        )
        start_time = time.time()

        obs, _ = envs.reset(seed=args.seed)
        success_total = 0
        episode_total = 0
        end = False
        for global_step in range(args.total_timesteps):
            if end:
                break
            # ALGO LOGIC: put action logic here
            if global_step < args.learning_starts:
                actions = np.array([envs.single_action_space.sample() for _ in range(envs.num_envs)])
            else:
                actions, _, _ = actor.get_action(torch.Tensor(obs).to(device))
                actions = actions.detach().cpu().numpy()

            # execute the game and log data.
            next_obs, rewards, terminations, truncations, infos = envs.step(actions)

            # record rewards for plotting purposes
            if "final_info" in infos:
                for info in infos["final_info"]:
                    if info['success'] == 1:
                        success_total += 1
                    episode_total += 1
                    rate = success_total / episode_total
                    if episode_total % args.average_num == 0:
                        if rate > args.threshold:
                            end = True
                        episode_total = success_total = 0
                    print(
                        f"{env_name}: global_step={global_step}, episodic_return={info['episode']['r']}, success_rate={rate}")
                    writer.add_scalar(f"{env_name}/{task_index}/episodic_return", info["episode"]["r"], global_step)
                    writer.add_scalar(f"{env_name}/{task_index}/episodic_length", info["episode"]["l"], global_step)
                    writer.add_scalar(f"{env_name}/{task_index}/success_rate", rate, global_step)

            # save data to reply buffer; handle `final_observation`
            real_next_obs = next_obs.copy()
            for idx, trunc in enumerate(truncations):
                if trunc:
                    real_next_obs[idx] = infos["final_observation"][idx]
            rb.add(obs, real_next_obs, actions, rewards, terminations, infos)

            # CRUCIAL step easy to overlook
            obs = next_obs

            # ALGO LOGIC: training.
            if global_step > args.learning_starts:
                if task_index > 0 and global_step % args.replay_interval == 0:
                    data = rb_pre.sample(args.batch_size)
                else:
                    data = rb.sample(args.batch_size)
                with torch.no_grad():
                    next_state_actions, next_state_log_pi, _ = actor.get_action(data.next_observations)
                    qf1_next_target = qf1_target(data.next_observations, next_state_actions)
                    qf2_next_target = qf2_target(data.next_observations, next_state_actions)
                    min_qf_next_target = torch.min(qf1_next_target, qf2_next_target) - alpha * next_state_log_pi
                    next_q_value = data.rewards.flatten() + (1 - data.dones.flatten()) * args.gamma * (
                        min_qf_next_target).view(-1)

                qf1_a_values = qf1(data.observations.to(torch.float32), data.actions.to(torch.float32)).view(-1)
                qf2_a_values = qf2(data.observations.to(torch.float32), data.actions.to(torch.float32)).view(-1)
                qf1_loss = F.mse_loss(qf1_a_values, next_q_value)
                qf2_loss = F.mse_loss(qf2_a_values, next_q_value)
                qf_loss = qf1_loss + qf2_loss

                # optimize the model
                qf1_optimizer.zero_grad()
                qf2_optimizer.zero_grad()
                qf_loss.backward()
                qf1_optimizer.step()
                qf2_optimizer.step()

                if global_step % args.policy_frequency == 0:
                    for _ in range(
                            args.policy_frequency
                    ):  # compensate for the delay by doing 'actor_update_interval' instead of 1
                        pi, log_pi, _ = actor.get_action(data.observations)
                        qf1_pi = qf1(data.observations, pi)
                        qf2_pi = qf2(data.observations, pi)
                        min_qf_pi = torch.min(qf1_pi, qf2_pi)
                        actor_loss = ((alpha * log_pi) - min_qf_pi).mean()

                        actor_optimizer.zero_grad()
                        actor_loss.backward()
                        actor_optimizer.step()

                        if args.autotune:
                            with torch.no_grad():
                                _, log_pi, _ = actor.get_action(data.observations)
                            alpha_loss = (-log_alpha.exp() * (log_pi + target_entropy)).mean()

                            a_optimizer.zero_grad()
                            alpha_loss.backward()
                            a_optimizer.step()
                            alpha = log_alpha.exp().item()

                # update the target networks
                if global_step % args.target_network_frequency == 0:
                    for param, target_param in zip(qf1.parameters(), qf1_target.parameters()):
                        target_param.data.copy_(args.tau * param.data + (1 - args.tau) * target_param.data)
                    for param, target_param in zip(qf2.parameters(), qf2_target.parameters()):
                        target_param.data.copy_(args.tau * param.data + (1 - args.tau) * target_param.data)

                if global_step % 100 == 0:
                    writer.add_scalar(f"{env_name}/{task_index}/qf1_values", qf1_a_values.mean().item(), global_step)
                    writer.add_scalar(f"{env_name}/{task_index}/qf2_values", qf2_a_values.mean().item(), global_step)
                    writer.add_scalar(f"{env_name}/{task_index}/qf1_loss", qf1_loss.item(), global_step)
                    writer.add_scalar(f"{env_name}/{task_index}/qf2_loss", qf2_loss.item(), global_step)
                    writer.add_scalar(f"{env_name}/{task_index}/qf_loss", qf_loss.item() / 2.0, global_step)
                    writer.add_scalar(f"{env_name}/{task_index}/actor_loss", actor_loss.item(), global_step)
                    writer.add_scalar(f"{env_name}/{task_index}/alpha", alpha, global_step)
                    print("SPS:", int(global_step / (time.time() - start_time)))
                    writer.add_scalar(f"{env_name}/{task_index}/SPS", int(global_step / (time.time() - start_time)),
                                      global_step)
                    if args.autotune:
                        writer.add_scalar(f"{env_name}/{task_index}/alpha_loss", alpha_loss.item(), global_step)

        actor_state_dict = actor.state_dict()
        torch.save(actor_state_dict, f"runs/{run_name}/actor_state_dict.pth")

        qf1_state_dict = qf1.state_dict()
        torch.save(qf1_state_dict, f"runs/{run_name}/qf1_state_dict.pth")

        qf2_state_dict = qf2.state_dict()
        torch.save(qf2_state_dict, f"runs/{run_name}/qf2_state_dict.pth")

        torch.save(rb, f"runs/{run_name}/replay_buffer.pth")
    else:
        actor_state_dict = torch.load(checkpoint + "/actor_state_dict.pth")
        actor.load_state_dict(actor_state_dict)

        qf1_state_dict = torch.load(checkpoint + "/qf1_state_dict.pth")
        qf1.load_state_dict(qf1_state_dict)

        qf2_state_dict = torch.load(checkpoint + "/qf2_state_dict.pth")
        qf2.load_state_dict(qf2_state_dict)

        qf1_target.load_state_dict(qf1.state_dict())
        qf2_target.load_state_dict(qf2.state_dict())

        rb = torch.load(checkpoint + "/replay_buffer.pth")

    def get_ave_critic():
        obs, _ = envs.reset(seed=args.seed)

        qf1_values = 0
        qf2_values = 0
        qf1_features = []
        qf2_features = []
        length = 0

        for step in range(100):
            data = rb.sample(args.batch_size)

            qf1_a_values, qf1_feature = qf1.forward_with_feature(data.observations.to(torch.float32),
                                                                 data.actions.to(torch.float32))
            qf1_a_values = qf1_a_values.view(-1)
            qf2_a_values, qf2_feature = qf2.forward_with_feature(data.observations.to(torch.float32),
                                                                 data.actions.to(torch.float32))
            qf2_a_values = qf2_a_values.view(-1)

            if len(qf1_features) == 0:
                for feature_layer in qf1_feature:
                    feature_layer = torch.sum(feature_layer, 0)
                    qf1_features.append(feature_layer.cpu().detach())

                for feature_layer in qf2_feature:
                    feature_layer = torch.sum(feature_layer, 0)
                    qf2_features.append(feature_layer.cpu().detach())
            else:
                for i in range(len(qf1_feature)):
                    feature_layer = torch.sum(qf1_feature[i], 0)
                    qf1_features[i] += feature_layer.cpu().detach()

                    feature_layer = torch.sum(qf2_feature[i], 0)
                    qf2_features[i] += feature_layer.cpu().detach()

            length += args.batch_size
            qf1_values += torch.sum(qf1_a_values).cpu().detach()
            qf2_values += torch.sum(qf2_a_values).cpu().detach()

        return qf1_features, qf1_values, qf2_features, qf2_values, length

    def get_ave_actor():
        obs, _ = envs.reset(seed=args.seed)
        feature_aves = []
        episode_num = 0
        features = []
        for global_step in range(50000):
            actions, _, _, feature = actor.get_action_with_feature(torch.Tensor(obs).to(device))
            actions = actions.detach().cpu().numpy()

            if len(features) == 0:
                for feature_layer in feature:
                    features.append(feature_layer.cpu().detach())
            else:
                for i in range(len(feature)):
                    features[i] += feature[i].cpu().detach()

            # execute the task and log data.
            next_obs, rewards, terminations, truncations, infos = envs.step(actions)

            # record rewards for plotting purposes
            if "final_info" in infos:
                for info in infos["final_info"]:
                    episode_num += 1
                    if len(feature_aves) == 0:
                        for feature in features:
                            feature_aves.append(feature / info["episode"]["l"])
                    else:
                        for i in range(len(features)):
                            feature_aves[i] += features[i] / info["episode"]["l"]
                    features = []

            # CRUCIAL step easy to overlook
            obs = next_obs
        return feature_aves, episode_num

    qf1_features, qf1_values, qf2_features, qf2_values, length = get_ave_critic()
    feature_aves, episode_num = get_ave_actor()
    qf1_top, qf1_score, qf2_top, qf2_score = eval_neuron_critic(qf1_features, qf1_values, qf2_features, qf2_values, rb, length, 50000,
                                          int(frozen_rate*2304))
    actor_top, actor_score = eval_neuron_actor(feature_aves, envs, episode_num, 100, device, int(frozen_rate*2304))
    envs.close()
    return qf1_top, qf1_score, qf2_top, qf1_score, actor_top, actor_score, rb

def test_env(envs, device, args, env_name, task_index):
    obs, _ = envs.reset(seed=args.seed)
    success_total = 0
    episode_total = 0
    for global_step in range(args.test_timesteps):
        actions, _, _ = actor.get_action(torch.Tensor(obs).to(device))
        actions = actions.detach().cpu().numpy()

        # TRY NOT TO MODIFY: execute the game and log data.
        next_obs, rewards, terminations, truncations, infos = envs.step(actions)

        # TRY NOT TO MODIFY: record rewards for plotting purposes
        if "final_info" in infos:
            for info in infos["final_info"]:
                if info['success'] == 1:
                    success_total += 1
                episode_total += 1
                rate = success_total / episode_total
                if episode_total % args.average_num == 0:
                    episode_total = success_total = 0
                print(
                    f"test_{env_name}: global_step={global_step}, episodic_return={info['episode']['r']}, success_rate={rate}")
                writer.add_scalar(f"test_{env_name}/{task_index}/episodic_return", info["episode"]["r"], global_step)
                writer.add_scalar(f"test_{env_name}/{task_index}/episodic_length", info["episode"]["l"], global_step)
                writer.add_scalar(f"test_{env_name}/{task_index}/success_rate", rate, global_step)

        # TRY NOT TO MODIFY: CRUCIAL step easy to overlook
        obs = next_obs
    envs.close()

if __name__ == "__main__":
    import stable_baselines3 as sb3

    if sb3.__version__ < "2.0":
        raise ValueError(
            """Ongoing migration: run the following command to install the new dependencies:
poetry run pip install "stable_baselines3==2.0.0a1"
"""
        )

    args = tyro.cli(Args)
    run_name = f"{args.env_id}__{args.exp_name}__{args.seed}__{int(time.time())}"
    writer = SummaryWriter(f"runs/{run_name}")
    writer.add_text(
        "hyperparameters",
        "|param|value|\n|-|-|\n%s" % ("\n".join([f"|{key}|{value}|" for key, value in vars(args).items()])),
    )

    # TRY NOT TO MODIFY: seeding
    random.seed(args.seed)
    np.random.seed(args.seed)
    torch.manual_seed(args.seed)
    torch.backends.cudnn.deterministic = args.torch_deterministic

    device = torch.device("cuda" if torch.cuda.is_available() and args.cuda else "cpu")

    # env setup
    envs = [gym.vector.SyncVectorEnv([make_env(env_name, args.gamma)])
            for env_name in args.env_names]

    max_action = float(envs[0].single_action_space.high[0])

    actor = Actor(envs[0]).to(device)
    qf1 = SoftQNetwork(envs[0]).to(device)
    qf2 = SoftQNetwork(envs[0]).to(device)
    qf1_target = SoftQNetwork(envs[0]).to(device)
    qf2_target = SoftQNetwork(envs[0]).to(device)
    qf1_target.load_state_dict(qf1.state_dict())
    qf2_target.load_state_dict(qf2.state_dict())
    qf1_optimizer = Adam(list(qf1.parameters()), lr=args.q_lr)
    qf2_optimizer = Adam(list(qf2.parameters()), lr=args.q_lr)
    actor_optimizer = Adam(list(actor.parameters()), lr=args.policy_lr)

    env_num = len(envs)
    sum_rb = ReplayBuffer(
        args.buffer_size * env_num,
        envs[0].single_observation_space,
        envs[0].single_action_space,
        device,
        handle_timeout_termination=False,
    )
    qf1_skill_neurons = [[] for _ in range(env_num)]
    qf2_skill_neurons = [[] for _ in range(env_num)]
    actor_skill_neurons = [[] for _ in range(env_num)]
    qf1_skill_scores = [[] for _ in range(env_num)]
    qf2_skill_scores = [[] for _ in range(env_num)]
    actor_skill_scores = [[] for _ in range(env_num)]
    for i in range(args.cycly_num):
        for j in range(len(envs)):
            actor_optimizer.set_lr(actor_skill_neurons, actor_skill_scores, j, env_num, args.score_coef)
            qf1_optimizer.set_lr(qf1_skill_neurons, qf1_skill_scores, j, env_num, args.score_coef)
            qf2_optimizer.set_lr(qf2_skill_neurons, qf2_skill_scores, j, env_num, args.score_coef)

            qf1_skill_neuron, qf1_skill_score, qf2_skill_neuron, qf2_skill_score, actor_skill_neuron, actor_skill_score, rb = train_env(envs[j], device, args, args.env_names[j], i*env_num+j, sum_rb, args.frozen_size[j], None)
            qf1_skill_neurons[j] = qf1_skill_neuron
            qf2_skill_neurons[j] = qf2_skill_neuron
            actor_skill_neurons[j] = actor_skill_neuron
            qf1_skill_scores[j] = qf1_skill_score
            qf2_skill_scores[j] = qf2_skill_score
            actor_skill_scores[j] = actor_skill_score
               
            if i == 0:
                sum_rb = merge_buffers(rb, sum_rb)
            for k in range(env_num):
                test_env(envs[k], device, args, args.env_names[k], i*env_num+j)
            
    writer.close()
