import os
import random
import time
from dataclasses import dataclass, field
import gymnasium as gym
import numpy as np
import torch
import torch.nn as nn
import torch.nn.functional as F
import torch.optim as optim
import tyro
from stable_baselines3.common.buffers import ReplayBuffer
from torch.utils.tensorboard import SummaryWriter
import metaworld


@dataclass
class Args:
    exp_name: str = os.path.basename(__file__)[: -len(".py")]
    """the name of this experiment"""
    seed: int = 1
    """seed of the experiment"""
    torch_deterministic: bool = True
    """if toggled, `torch.backends.cudnn.deterministic=False`"""
    cuda: bool = True
    """if toggled, cuda will be enabled by default"""

    # Algorithm specific arguments
    env_id: str = "mt4"
    """the environment id of the task"""
    env_names: list = field(
        default_factory=lambda: ['window-open-v2', 'window-close-v2', 'drawer-open-v2', 'drawer-close-v2'])
    """the environment id of the task"""
    total_timesteps: int = 500000
    """total timesteps of the experiments"""
    test_timesteps: int = 100000
    """test timesteps of the experiments"""
    buffer_size: int = int(1e6)
    """the replay memory buffer size"""
    gamma: float = 0.99
    """the discount factor gamma"""
    tau: float = 0.005
    """target smoothing coefficient (default: 0.005)"""
    batch_size: int = 256  # 1280
    """the batch size of sample from the reply memory"""
    learning_starts: int = 1e4
    """timestep to start learning"""
    policy_lr: float = 3e-4
    """the learning rate of the policy network optimizer"""
    q_lr: float = 1e-3
    """the learning rate of the Q network network optimizer"""
    policy_frequency: int = 2
    """the frequency of training policy (delayed)"""
    target_network_frequency: int = 1  # Denis Yarats' implementation delays this by 2.
    """the frequency of updates for the target nerworks"""
    noise_clip: float = 0.5
    """noise clip parameter of the Target Policy Smoothing Regularization"""
    alpha: float = 0.2
    """Entropy regularization coefficient."""
    autotune: bool = True
    """automatic tuning of the entropy coefficient"""
    average_num: int = 100
    """the number of episode to compute average success rate"""
    threshold: float = 0.9
    """the threshold representing finishing training"""
    cycle_num: int = 2
    """the number of cycling"""


def make_env(env_name, gamma):
    def thunk():
        mt1 = metaworld.MT1(env_name)  # Construct the benchmark, sampling tasks
        env = mt1.train_classes[env_name]()  # Create an environment with task `pick_place`
        task = mt1.train_tasks[0]
        env.set_task(task)  # Set task
        env = gym.wrappers.RecordEpisodeStatistics(env)
        return env

    return thunk

def layer_init(layer, bias_const=0.0):
    nn.init.kaiming_normal_(layer.weight)
    torch.nn.init.constant_(layer.bias, bias_const)
    return layer

# ALGO LOGIC: initialize agent here:
class SoftQNetwork(nn.Module):
    def __init__(self, env, delta = 0.9, alpha = 0.05, beta = 0.5, eta_max = 0.1):
        super().__init__()
        self.fc1 = nn.Linear(np.array(env.single_observation_space.shape).prod() + np.prod(env.single_action_space.shape), 768)
        self.fc2 = nn.Linear(768, 768)
        self.fc3 = nn.Linear(768, 768)
        self.fc4 = nn.Linear(768, 1)
        self.layers = [self.fc1, self.fc2]
        self.activations = {}
        self.importance_weights = {}
        self.updated_lrs = {}
        self.delta = delta
        self.beta = beta
        self.eta_max = torch.Tensor([eta_max]).to('cuda')
        self.alpha = alpha
        self.register_hooks()

    def forward(self, x, a):
        x = torch.cat([x, a], 1)
        x = F.relu(self.fc1(x))
        x = F.relu(self.fc2(x))
        x = F.relu(self.fc3(x))
        x = self.fc4(x)
        return x

    def forward_hook(self, module, input, output):
        self.activations[module] = F.relu(output)

    def backward_hook(self, module, grad_input, grad_output):
        eps = 1e-8
        if module in self.activations:
            grad_output = grad_output[0]
            activation = self.activations[module]
            neuron_importance = torch.mean(torch.abs(activation * grad_output), dim=0)

            norm_importance = neuron_importance / (torch.sum(neuron_importance) / len(neuron_importance) + eps)
            if module not in self.importance_weights:
                self.importance_weights[module] = importance_weights = (1 - self.delta) * norm_importance
            else:
                self.importance_weights[module] = importance_weights = self.delta * self.importance_weights[module] + (1 - self.delta) * norm_importance
            
            self.updated_lrs[module] = torch.where(
                importance_weights <= self.beta,
                torch.min(self.eta_max, self.alpha * torch.sqrt(self.beta / importance_weights - 1)),
                torch.zeros_like(importance_weights)
            )
            
    def register_hooks(self):
        for module in self.modules():
            if isinstance(module, nn.Linear):
                module.register_forward_hook(self.forward_hook)
                module.register_full_backward_hook(self.backward_hook)


LOG_STD_MAX = 10
LOG_STD_MIN = -10


class Actor(nn.Module):
    def __init__(self, env, delta = 0.9, alpha = 0.05, beta = 0.5, eta_max = 0.1):
        super().__init__()
        self.fc1 = nn.Linear(np.array(env.single_observation_space.shape).prod(), 768)
        self.fc2 = nn.Linear(768, 768)
        self.fc3 = nn.Linear(768, 768)
        self.fc_mean = nn.Linear(768, np.prod(env.single_action_space.shape))
        self.fc_logstd = nn.Linear(768, np.prod(env.single_action_space.shape))
        # action rescaling
        self.register_buffer(
            "action_scale", torch.tensor((env.action_space.high - env.action_space.low) / 2.0, dtype=torch.float32)
        )
        self.register_buffer(
            "action_bias", torch.tensor((env.action_space.high + env.action_space.low) / 2.0, dtype=torch.float32)
        )
        self.layers = [self.fc1, self.fc2, self.fc_mean, self.fc_logstd]
        self.activations = {}
        self.gradients = {}
        self.importance_weights = {}
        self.updated_lrs = {}
        self.delta = delta
        self.beta = beta
        self.eta_max = torch.Tensor([eta_max]).to('cuda')
        self.alpha = alpha
        self.register_hooks()

    def forward(self, x):
        x1 = F.relu(self.fc1(x))
        x2 = F.relu(self.fc2(x1))
        x3 = F.relu(self.fc3(x2))
        mean = self.fc_mean(x3)
        log_std = self.fc_logstd(x3)
        log_std = torch.tanh(log_std)
        log_std = LOG_STD_MIN + 0.5 * (LOG_STD_MAX - LOG_STD_MIN) * (log_std + 1)  # From SpinUp / Denis Yarats

        return mean, log_std

    def get_action(self, x):
        mean, log_std = self(x)
        std = log_std.exp()
        normal = torch.distributions.Normal(mean, std)
        x_t = normal.rsample()  # for reparameterization trick (mean + std * N(0,1))
        y_t = torch.tanh(x_t)
        action = y_t * self.action_scale + self.action_bias
        log_prob = normal.log_prob(x_t)
        # Enforcing Action Bound
        log_prob -= torch.log(self.action_scale * (1 - y_t.pow(2)) + 1e-6)
        log_prob = log_prob.sum(1, keepdim=True)
        mean = torch.tanh(mean) * self.action_scale + self.action_bias
        return action, log_prob, mean

    def forward_hook(self, module, input, output):
        self.activations[module] = F.relu(output)

    def backward_hook(self, module, grad_input, grad_output):
        eps = 1e-8
        if module in self.activations:
            grad_output = grad_output[0]
            activation = self.activations[module]
            neuron_importance = torch.mean(torch.abs(activation * grad_output), dim=0)

            norm_importance = neuron_importance / (torch.sum(neuron_importance) / len(neuron_importance) + eps)

            if module not in self.importance_weights:
                self.importance_weights[module] = importance_weights =  (1 - self.delta) * norm_importance
            else:
                self.importance_weights[module] = importance_weights = self.delta * self.importance_weights[module] + (1 - self.delta) * norm_importance

            self.updated_lrs[module] = torch.where(
                importance_weights <= self.beta,
                torch.min(self.eta_max, self.alpha * torch.sqrt(self.beta / importance_weights - 1)),
                torch.zeros_like(importance_weights)
            )
            
    def register_hooks(self):
        for module in self.modules():
            if isinstance(module, nn.Linear):
                module.register_forward_hook(self.forward_hook)
                module.register_full_backward_hook(self.backward_hook)


def apply_updated_learning_rates(optimizer, model):
    with torch.no_grad():
        index = 2 * len(model.updated_lrs)-1
        for group in optimizer.param_groups:
            for p in group['params']:
                lrs = model.updated_lrs[list(model.updated_lrs.keys())[index//2]]
                if p.grad.ndim != lrs.ndim:
                    lrs = lrs.unsqueeze(1)
                p.grad *= lrs
                index -= 1

def train_env(envs, device, args, env_name, task_index):
    # Automatic entropy tuning
    if args.autotune:
        target_entropy = -torch.prod(torch.Tensor(envs.single_action_space.shape).to(device)).item()
        log_alpha = torch.zeros(1, requires_grad=True, device=device)
        alpha = log_alpha.exp().item()
        a_optimizer = optim.Adam([log_alpha], lr=args.q_lr)
    else:
        alpha = args.alpha
    envs.single_observation_space.dtype = np.float32
    rb = ReplayBuffer(
        args.buffer_size,
        envs.single_observation_space,
        envs.single_action_space,
        device,
        handle_timeout_termination=False,
    )
    start_time = time.time()

    # TRY NOT TO MODIFY: start the game
    obs, _ = envs.reset(seed=args.seed)
    success_total = 0
    episode_total = 0
    end = False
    for global_step in range(args.total_timesteps):
        if end == True:
            break
        # ALGO LOGIC: put action logic here
        if global_step < args.learning_starts:
            actions = np.array([envs.single_action_space.sample() for _ in range(envs.num_envs)])
        else:
            actions, _, _ = actor.get_action(torch.Tensor(obs).to(device))
            actions = actions.detach().cpu().numpy()

        # TRY NOT TO MODIFY: execute the game and log data.
        next_obs, rewards, terminations, truncations, infos = envs.step(actions)

        # TRY NOT TO MODIFY: record rewards for plotting purposes
        if "final_info" in infos:
            for info in infos["final_info"]:
                if info['success'] == 1:
                    success_total += 1
                episode_total += 1
                rate = success_total / episode_total
                if episode_total % args.average_num == 0:
                    if rate > args.threshold:
                        end = True
                    episode_total = success_total = 0
                print(
                    f"{env_name}: global_step={global_step}, episodic_return={info['episode']['r']}, success_rate={rate}")
                writer.add_scalar(f"{env_name}/{task_index}/episodic_return", info["episode"]["r"], global_step)
                writer.add_scalar(f"{env_name}/{task_index}/episodic_length", info["episode"]["l"], global_step)
                writer.add_scalar(f"{env_name}/{task_index}/success_rate", rate, global_step)

        # TRY NOT TO MODIFY: save data to reply buffer; handle `final_observation`
        real_next_obs = next_obs.copy()
        for idx, trunc in enumerate(truncations):
            if trunc:
                real_next_obs[idx] = infos["final_observation"][idx]
        rb.add(obs, real_next_obs, actions, rewards, terminations, infos)

        # TRY NOT TO MODIFY: CRUCIAL step easy to overlook
        obs = next_obs

        # ALGO LOGIC: training.
        if global_step > args.learning_starts:
            data = rb.sample(args.batch_size)
            with torch.no_grad():
                next_state_actions, next_state_log_pi, _ = actor.get_action(data.next_observations.float())
                qf1_next_target = qf1_target(data.next_observations.float(), next_state_actions)
                qf2_next_target = qf2_target(data.next_observations.float(), next_state_actions)
                min_qf_next_target = torch.min(qf1_next_target, qf2_next_target) - alpha * next_state_log_pi
                next_q_value = data.rewards.flatten() + (1 - data.dones.flatten()) * args.gamma * (
                    min_qf_next_target).view(-1)

            qf1_a_values = qf1(data.observations.to(torch.float32), data.actions.to(torch.float32)).view(-1)
            qf2_a_values = qf2(data.observations.to(torch.float32), data.actions.to(torch.float32)).view(-1)
            qf1_loss = F.mse_loss(qf1_a_values, next_q_value)
            qf2_loss = F.mse_loss(qf2_a_values, next_q_value)
            qf_loss = qf1_loss + qf2_loss

            # optimize the model
            qf1_optimizer.zero_grad()
            qf2_optimizer.zero_grad()
            qf_loss.backward()
            if task_index > 0:
                apply_updated_learning_rates(qf1_optimizer, qf1)
                apply_updated_learning_rates(qf2_optimizer, qf2)
            qf1_optimizer.step()
            qf2_optimizer.step()

            if global_step % args.policy_frequency == 0:  # TD 3 Delayed update support
                for _ in range(
                        args.policy_frequency
                ):  # compensate for the delay by doing 'actor_update_interval' instead of 1
                    pi, log_pi, _ = actor.get_action(data.observations.float())
                    qf1_pi = qf1(data.observations.float(), pi)
                    qf2_pi = qf2(data.observations.float(), pi)
                    min_qf_pi = torch.min(qf1_pi, qf2_pi)
                    actor_loss = ((alpha * log_pi) - min_qf_pi).mean()

                    actor_optimizer.zero_grad()
                    actor_loss.backward()
                    if task_index > 0:
                        apply_updated_learning_rates(actor_optimizer, actor)
                    actor_optimizer.step()

                    if args.autotune:
                        with torch.no_grad():
                            _, log_pi, _ = actor.get_action(data.observations.float())
                        alpha_loss = (-log_alpha.exp() * (log_pi + target_entropy)).mean()

                        a_optimizer.zero_grad()
                        alpha_loss.backward()
                        a_optimizer.step()
                        alpha = log_alpha.exp().item()

            # update the target networks
            if global_step % args.target_network_frequency == 0:
                for param, target_param in zip(qf1.parameters(), qf1_target.parameters()):
                    target_param.data.copy_(args.tau * param.data + (1 - args.tau) * target_param.data)
                for param, target_param in zip(qf2.parameters(), qf2_target.parameters()):
                    target_param.data.copy_(args.tau * param.data + (1 - args.tau) * target_param.data)

            if global_step % 100 == 0:
                writer.add_scalar(f"{env_name}/{type}/{task_index}/qf1_values", qf1_a_values.mean().item(),
                                  global_step)
                writer.add_scalar(f"{env_name}/{type}/{task_index}/qf2_values", qf2_a_values.mean().item(),
                                  global_step)
                writer.add_scalar(f"{env_name}/{type}/{task_index}/qf1_loss", qf1_loss.item(), global_step)
                writer.add_scalar(f"{env_name}/{type}/{task_index}/qf2_loss", qf2_loss.item(), global_step)
                writer.add_scalar(f"{env_name}/{type}/{task_index}/qf_loss", qf_loss.item() / 2.0, global_step)
                writer.add_scalar(f"{env_name}/{type}/{task_index}/actor_loss", actor_loss.item(), global_step)
                writer.add_scalar(f"{env_name}/{type}/{task_index}/alpha", alpha, global_step)
                print("SPS:", int(global_step / (time.time() - start_time)))
                writer.add_scalar(f"{env_name}/{type}/{task_index}/SPS",
                                  int(global_step / (time.time() - start_time)),
                                  global_step)
                if args.autotune:
                    writer.add_scalar(f"{env_name}/{type}/losses/alpha_loss", alpha_loss.item(), global_step)
    envs.close()

def test_env(envs, device, args, env_name, task_index):
    obs, _ = envs.reset(seed=args.seed)
    success_total = 0
    episode_total = 0
    for global_step in range(args.test_timesteps):

        actions, _, _ = actor.get_action(torch.Tensor(obs).to(device))
        actions = actions.detach().cpu().numpy()

        # TRY NOT TO MODIFY: execute the game and log data.
        next_obs, rewards, terminations, truncations, infos = envs.step(actions)

        # TRY NOT TO MODIFY: record rewards for plotting purposes
        if "final_info" in infos:
            for info in infos["final_info"]:
                if info['success'] == 1:
                    success_total += 1
                episode_total += 1
                rate = success_total / episode_total
                if episode_total % args.average_num == 0:
                    episode_total = success_total = 0
                print(
                    f"test_{env_name}: global_step={global_step}, episodic_return={info['episode']['r']}, success_rate={rate}")
                writer.add_scalar(f"test_{env_name}/{task_index}/episodic_return", info["episode"]["r"],
                                  global_step)
                writer.add_scalar(f"test_{env_name}/{task_index}/episodic_length", info["episode"]["l"],
                                  global_step)
                writer.add_scalar(f"test_{env_name}/{task_index}/success_rate", rate, global_step)
        # TRY NOT TO MODIFY: CRUCIAL step easy to overlook
        obs = next_obs
    envs.close()

if __name__ == "__main__":
    import stable_baselines3 as sb3

    if sb3.__version__ < "2.0":
        raise ValueError(
            """Ongoing migration: run the following command to install the new dependencies:
poetry run pip install "stable_baselines3==2.0.0a1"
"""
        )

    args = tyro.cli(Args)
    run_name = f"{args.env_id}__{args.exp_name}__{args.seed}__{int(time.time())}"
    writer = SummaryWriter(f"runs/{run_name}")
    writer.add_text(
        "hyperparameters",
        "|param|value|\n|-|-|\n%s" % ("\n".join([f"|{key}|{value}|" for key, value in vars(args).items()])),
    )

    # TRY NOT TO MODIFY: seeding
    random.seed(args.seed)
    np.random.seed(args.seed)
    torch.manual_seed(args.seed)
    torch.backends.cudnn.deterministic = args.torch_deterministic

    device = torch.device("cuda" if torch.cuda.is_available() and args.cuda else "cpu")

    # env setup
    envs = [gym.vector.SyncVectorEnv([make_env(env_name, args.gamma)])
            for env_name in args.env_names]

    max_action = float(envs[0].single_action_space.high[0])

    actor = Actor(envs[0]).to(device)
    qf1 = SoftQNetwork(envs[0]).to(device)
    qf2 = SoftQNetwork(envs[0]).to(device)
    qf1_target = SoftQNetwork(envs[0]).to(device)
    qf2_target = SoftQNetwork(envs[0]).to(device)
    qf1_target.load_state_dict(qf1.state_dict())
    qf2_target.load_state_dict(qf2.state_dict())
    qf1_optimizer = optim.Adam(list(qf1.parameters()), lr=args.q_lr)
    qf2_optimizer = optim.Adam(list(qf2.parameters()), lr=args.q_lr)
    actor_optimizer = optim.Adam( 
        list(actor.parameters()), lr=args.policy_lr
    )

    env_num = len(envs)
    for i in range(args.cycle_num):
        for j in range(len(envs)):
            
            train_env(envs[j], device, args, args.env_names[j], i*env_num+j)
        
            for k in range(env_num):
                test_env(envs[k], device, args, args.env_names[k], i*env_num+j)
                
    writer.close()
