from typing import List, Tuple, Union
import numpy as np


def normalize_scores(scores: Union[List[float], np.ndarray]) -> np.ndarray:
    """
    Normalizes a list or array of scores to the [0, 1] range using min-max scaling.

    :param scores: A numpy array or list of scores.
    :return: A numpy array of normalized scores.
    """
    scores_arr = np.asarray(scores)
    min_score, max_score = np.min(scores_arr), np.max(scores_arr)
    score_range = max_score - min_score

    # Avoid division by zero if all scores are the same
    if score_range == 0:
        # Return an array of 0s if all scores are the same, or handle as needed.
        # Here we assume they are all equivalent, so a mid-point or 0 is reasonable.
        return np.zeros_like(scores_arr)

    return (scores_arr - min_score) / score_range


def sort_similarity_scores(
    features: List[str], scores: Union[List[float], np.ndarray]
) -> Tuple[List[str], List[float]]:
    """
    Sorts a list of features based on a corresponding list of scores in ascending order.
    """
    scores_arr = np.asarray(scores)
    features_arr = np.asarray(features)

    # Get the indices that would sort the scores array
    sorted_indices = scores_arr.argsort()

    # Use these indices to sort both the features and scores
    sorted_features = features_arr[sorted_indices].tolist()
    sorted_scores = scores_arr[sorted_indices].tolist()

    return sorted_features, sorted_scores


def reverse_scores(scores: Union[List[float], np.ndarray]) -> np.ndarray:
    """Reverses scores by calculating 1 - score for each element."""
    return 1 - np.asarray(scores)