import json
import sys


def sort_permutations_by_loss(json_file_path):
    """
    Reads a JSON file containing permutation losses, sorts them by loss in ascending order,
    and prints the sorted permutation IDs.

    Args:
        json_file_path (str): Path to the JSON file.
    """
    try:
        with open(json_file_path, "r") as f:
            data = json.load(f)
    except FileNotFoundError:
        print(f"Error: File not found at {json_file_path}", file=sys.stderr)
        sys.exit(1)
    except json.JSONDecodeError:
        print(f"Error: Could not decode JSON from {json_file_path}", file=sys.stderr)
        sys.exit(1)

    if not isinstance(data, dict):
        print(f"Error: Expected a JSON object (dict) in {json_file_path}, but got {type(data)}", file=sys.stderr)
        sys.exit(1)

    # Convert string keys to int for proper sorting if they represent numbers,
    # and ensure values are numeric for loss.
    try:
        # Items are (permutation_id_str, loss_value)
        # We sort by loss_value.
        sorted_permutations = sorted(data.items(), key=lambda item: item[1])
    except TypeError:
        print(
            f"Error: Could not sort. Ensure permutation IDs are keys and losses are numeric values in {json_file_path}.",
            file=sys.stderr,
        )
        sys.exit(1)

    for perm_id, loss in sorted_permutations:
        print(perm_id)


if __name__ == "__main__":
    if len(sys.argv) != 2:
        print("Usage: python sort_permutations.py <path_to_json_file>", file=sys.stderr)
        sys.exit(1)

    json_file_path = sys.argv[1]
    sort_permutations_by_loss(json_file_path)
