import itertools
from typing import List


def split_permutation(permutation: List[int], num_blocks: int) -> List[List[int]]:
    """
    Splits a permutation into a specified number of blocks.
    Handles cases where the permutation length is not evenly divisible by the number of blocks.
    """
    if num_blocks <= 0:
        raise ValueError("Number of blocks must be positive.")

    n = len(permutation)
    base_block_size = n // num_blocks
    remainder = n % num_blocks

    blocks = []
    current_pos = 0
    for i in range(num_blocks):
        size = base_block_size + (1 if i < remainder else 0)
        blocks.append(permutation[current_pos : current_pos + size])
        current_pos += size

    return blocks


def combine_blocks(blocks: List[List[int]]) -> List[int]:
    """
    Combines a list of blocks back into a single permutation list.
    """
    return [item for block in blocks for item in block]


def generate_intra_block_candidates(base_perm: List[int], block_idx: int, num_blocks: int) -> List[List[int]]:
    """
    Generates candidate permutations by permuting elements ONLY within a specific block.
    Other blocks remain fixed.
    """
    blocks = split_permutation(base_perm, num_blocks)

    if block_idx < 0 or block_idx >= len(blocks):
        raise IndexError("block_idx is out of range.")

    target_block_content = blocks[block_idx]

    if len(target_block_content) <= 1:
        return [base_perm]

    candidate_perms = []
    for p_content in itertools.permutations(target_block_content):
        new_blocks = blocks[:block_idx] + [list(p_content)] + blocks[block_idx + 1 :]
        candidate_perms.append(combine_blocks(new_blocks))

    return candidate_perms


def generate_inter_block_candidates(base_perm: List[int], num_blocks: int) -> List[List[int]]:
    """
    Generates candidate permutations by permuting the order of the blocks themselves.
    The internal order of elements within each block is preserved.
    """
    blocks = split_permutation(base_perm, num_blocks)

    if len(blocks) <= 1:
        return [base_perm]

    candidate_perms = []
    for p_blocks in itertools.permutations(blocks):
        candidate_perms.append(combine_blocks(list(p_blocks)))

    return candidate_perms
