from __future__ import annotations

import inspect
import json
import os
import re
import csv
from datetime import datetime, timedelta
from pathlib import Path
from typing import Any, Callable, Dict, List
from tqdm import tqdm
import pandas as pd
from openai import OpenAI, RateLimitError
from src.data_generation.data_generation_utils import HARDCODED_CURRENT_TIME



# ---------------------------------------------------------------------------
# Configuration helpers
# ---------------------------------------------------------------------------

# These domain objects are assumed to be the same as in the original repo
from src.tools import (
    calendar,
    email,
    analytics,
    project_management,
    customer_relationship_manager,
    company_directory,
)

DOMAINS = [
    calendar,
    email,
    analytics,
    project_management,
    customer_relationship_manager,
    company_directory,
]

# ---------------------------------------------------------------------------
# Utility: convert python callables → OpenAI tool schemas
# ---------------------------------------------------------------------------

def _annotation_to_json_type(annotation: Any) -> str:
    # crude mapping; expand as needed
    mapping = {int: "number", float: "number", bool: "boolean", str: "string"}
    return mapping.get(annotation, "string")


def function_to_tool(fn):
    """
    Build an OpenAI function schema from a LangChain-style tool object.
    Expects `fn` to have:
      - `name`: the tool identifier (e.g. "email.delete_email")
      - `args_schema`: a Pydantic BaseModel class describing its parameters
      - `description`: a docstring or human-readable description
    Returns None if required attributes are missing.
    """
    # Ensure the tool has the necessary metadata
    if not hasattr(fn, 'name'):
        return None

    # The args_schema should be a Pydantic model class
    schema_model = fn.args_schema


    # Generate the JSON Schema from the Pydantic model
    model_schema = schema_model.model_json_schema()
    properties = model_schema.get("properties", {})
    required = model_schema.get("required", [])

    return {
        "type": "function",
        "function": {
            "name": fn.name,
            "description": (fn.description or "No description provided.").strip(),
            "parameters": {
                "type": "object",
                "properties": properties,
                "required": required,
            },
        },
    }


def build_tools(selected_toolkits: List[str]) -> tuple[list[dict[str, Any]], dict[str, Callable]]:
    tools_json: list[dict[str, Any]] = []
    lookup: dict[str, Callable] = {}

    module_alias = {
        'crm': 'customer_relationship_manager',
    }

    for module_name in selected_toolkits:
        actual_module_name = module_alias.get(module_name, module_name)
        module = globals()[actual_module_name]          # calendar / email / analytics …
        for attr in dir(module):
            obj = getattr(module, attr)

            if not (hasattr(obj, "name") and hasattr(obj, "args_schema")):
                continue

            schema = function_to_tool(obj)
            if schema:
                tools_json.append(schema)

            impl = getattr(obj, "func", obj)
            lookup[obj.name] = impl

    return tools_json, lookup





# ---------------------------------------------------------------------------
# Core ReAct‑style loop using OpenAI function‑calling
# ---------------------------------------------------------------------------

def _fmt_call(name: str, args: Dict[str, Any]) -> str:
    arg_list = [f'{k}="{v}"' for k, v in args.items()]
    return f"{name}.func(" + ", ".join(arg_list) + ")"


def run_agent(
    client,
    model: str,
    query: str,
    tools: List[Dict[str, Any]],
    fn_lookup: Dict[str, Callable],
    max_iter: int = 20,
    max_seconds: int = 120,
    temperature: float = 0,
):
    # ============================
    for dom in DOMAINS:
        if hasattr(dom, "reset_state"):
            dom.reset_state()
    # ============================

    system_prompt = (
        f"You are a helpful assistant and your task is to complete the user's query using the given tools."
        f"Today's date is {HARDCODED_CURRENT_TIME.strftime('%A, %B %d, %Y')} "
        f"and the current time is {HARDCODED_CURRENT_TIME.strftime('%H:%M')}."
        "Remember the current date and time when answering queries. Meetings must not start before 9am or end after 6pm."
        "Please keep in mind that I am giving you all the information you need, if you need additional information please find it yourself from the API, don't look to me for any other additional information. Respond to the human as helpfully and accurately as possible."
    )
    messages = [
        {"role": "system", "content": system_prompt},
        {"role": "user", "content": query},
    ]

    start_time = datetime.now()
    function_calls: List[str] = []

    for _ in range(max_iter):
        if (datetime.now() - start_time).total_seconds() > max_seconds:
            return {
                "output": "Agent stopped due to iteration limit or time limit.",
                "function_calls": function_calls,
            }

        resp = client.chat.completions.create(
            model=model,
            messages=messages,
            tools=tools,
            tool_choice="auto",
            temperature=temperature
        )
        choice = resp.choices[0]
        msg = choice.message

        if choice.finish_reason == "tool_calls":
            for call in msg.tool_calls:
                fn_name = call.function.name
                raw_args = call.function.arguments or "{}"
                try:
                    args = json.loads(raw_args)
                except json.JSONDecodeError:
                    args = {}

                result = fn_lookup[fn_name](**args)

                function_calls.append(_fmt_call(fn_name, args))

                messages.append(
                    {"role": "assistant", "content": None, "tool_calls": [call]}
                )
                messages.append(
                    {
                        "role": "tool",
                        "tool_call_id": call.id,
                        "name": fn_name,
                        "content": str(result),
                    }
                )
            continue

        messages.append({"role": "assistant", "content": msg.content})
        return {"output": msg.content, "function_calls": function_calls}

    return {
        "output": "Agent stopped due to iteration limit or time limit.",
        "function_calls": function_calls,
    }


# ---------------------------------------------------------------------------
# Public entry point mirroring original signature
# ---------------------------------------------------------------------------

def generate_results(
    queries_path: str | os.PathLike,
    model_name: str,
    tool_selection: str = "all",
    temperature: float = 0,
) -> pd.DataFrame:
    """Run inference over queries CSV and save results."""

    client = OpenAI()

    df_queries = pd.read_csv(queries_path)
    queries = df_queries["query"].tolist()

    # Pre‑build tools / look‑up once (unless overridden per‑row)
    toolkits = [
        "email",
        "calendar",
        "analytics",
        "project_management",
        "customer_relationship_manager",
    ]

    base_tools, base_lookup = build_tools(toolkits)

    results_rows = []
    for i, q in tqdm(enumerate(queries), total=len(queries), desc=f"{model_name} Processing {queries_path} queries... [temperature {temperature}]"):
        if tool_selection == "domains":
            row_tks = (
                df_queries["domains"].iloc[i].strip("[]").replace("'", "").split(", ")
            )
            tools, lookup = build_tools(row_tks)
        else:
            tools, lookup = base_tools, base_lookup

        # main attempt
        error_msg = ""
        try:
            agent_result = run_agent(client, model, q, tools, lookup, temperature=temperature)
            final = agent_result["output"]
            calls = agent_result["function_calls"]
        except Exception as exc:
            final = exc
            calls = []
            error_msg = str(exc)
        results_rows.append({
            "query": q,
            "function_calls": calls,
            "full_response": final,
            "error": error_msg,
        })

        # reset stateful domains if necessary
        for dom in DOMAINS:
            if hasattr(dom, "reset_state"):
                dom.reset_state()

    results_df = pd.DataFrame(results_rows)

    domain_name = Path(queries_path).stem.replace("_queries_and_answers", "")
    save_dir = Path("data/results") / domain_name
    save_dir.mkdir(parents=True, exist_ok=True)
    timestamp = datetime.now().strftime("%Y-%m-%d_%H-%M-%S")
    save_path = save_dir / f"{model_name}_{tool_selection}_{timestamp}.csv"
    results_df.to_csv(save_path, index=False, quoting=csv.QUOTE_ALL)
    print(f"Results saved to {save_path}")
    return results_df

# ---------------------------------------------------------------------------
# CLI usage
# ---------------------------------------------------------------------------
if __name__ == "__main__":

    AVAILABLE_LLMS = ["qwen2.5-32b-instruct"]
    os.environ["OPENAI_BASE_URL"] = "http://localhost:6000/v1"
    os.environ["OPENAI_API_KEY"] = ""

    import argparse, time

    parser = argparse.ArgumentParser(description="OpenAI ReAct Runner")
    # parser.add_argument("model_name", choices=list(MODELS.keys()), help="Model shortcut")
    # parser.add_argument("--tool_selection", default="all", choices=["all", "domains"])
    args = parser.parse_args()

    query_paths = [
        "data/processed/queries_and_answers/multi_domain_queries_and_answers.csv",
        "data/processed/queries_and_answers/email_queries_and_answers.csv",
        "data/processed/queries_and_answers/calendar_queries_and_answers.csv",
        "data/processed/queries_and_answers/analytics_queries_and_answers.csv",
        "data/processed/queries_and_answers/project_management_queries_and_answers.csv",
        "data/processed/queries_and_answers/customer_relationship_manager_queries_and_answers.csv",
    ]

    for i in range(20):
        for tool_selection in ["domains"]:
            for model in AVAILABLE_LLMS:
                for query_path in query_paths:
                    generate_results(
                        queries_path=query_path,
                        model_name=model,
                        tool_selection=tool_selection,
                        temperature=0.05 * i,
                    )