import pandas as pd
from langchain.tools import tool

class CustomerRelationshipManager:
    def __init__(self):
        self.CRM_DATA = pd.read_csv("data/processed/customer_relationship_manager_data.csv", dtype=str)

    def reset_state(self):
        """Resets the CRM data to the original state."""
        self.CRM_DATA = pd.read_csv("data/processed/customer_relationship_manager_data.csv", dtype=str)

    def search_customers(
        self,
        customer_name=None,
        customer_email=None,
        product_interest=None,
        status=None,
        assigned_to_email=None,
        last_contact_date_min=None,
        last_contact_date_max=None,
        follow_up_by_min=None,
        follow_up_by_max=None,
    ):
        """
        Searches for customers based on the given parameters.
        """
        customers = self.CRM_DATA.copy()
        if not any(
            [
                customer_name,
                customer_email,
                product_interest,
                status,
                assigned_to_email,
                last_contact_date_min,
                last_contact_date_max,
                follow_up_by_min,
                follow_up_by_max,
            ]
        ):
            return "No search parameters provided. Please provide at least one parameter."

        if customer_name:
            customers = customers[customers["customer_name"].str.contains(customer_name, case=False)]
        if customer_email:
            customers = customers[customers["customer_email"].str.contains(customer_email, case=False)]
        if product_interest:
            customers = customers[customers["product_interest"].str.contains(product_interest, case=False)]
        if status:
            customers = customers[customers["status"].str.contains(status, case=False)]
        if assigned_to_email:
            customers = customers[customers["assigned_to_email"].str.contains(assigned_to_email, case=False)]
        if last_contact_date_min:
            customers = customers[customers["last_contact_date"] >= last_contact_date_min]
        if last_contact_date_max:
            customers = customers[customers["last_contact_date"] <= last_contact_date_max]
        if follow_up_by_min:
            customers = customers[customers["follow_up_by"] >= follow_up_by_min]
        if follow_up_by_max:
            customers = customers[customers["follow_up_by"] <= follow_up_by_max]
        return customers.to_dict(orient="records")[:5]

    def update_customer(self, customer_id=None, field=None, new_value=None):
        """
        Updates a customer record by ID.
        """
        if not customer_id or not field or not new_value:
            return "Customer ID, field, or new value not provided."

        if field == "status" and new_value not in ["Qualified", "Won", "Lost", "Lead", "Proposal"]:
            return "Status not valid. Please choose from: 'Qualified', 'Won', 'Lost', 'Lead', 'Proposal'"

        if field == "product_interest" and new_value not in ["Software", "Hardware", "Services", "Consulting", "Training"]:
            return "Product interest not valid. Please choose from: 'Software', 'Hardware', 'Services', 'Consulting', 'Training'"

        if field == "customer_email" or field == "assigned_to_email":
            new_value = new_value.lower()

        if customer_id in self.CRM_DATA["customer_id"].values:
            if field in self.CRM_DATA.columns:
                self.CRM_DATA.loc[self.CRM_DATA["customer_id"] == customer_id, field] = new_value
                return "Customer updated successfully."
            else:
                return "Field not valid. Please choose from: 'customer_name', 'assigned_to_email', 'customer_email', 'customer_phone', 'last_contact_date', 'product_interest', 'status', 'notes', 'follow_up_by'"
        else:
            return "Customer not found."

    def add_customer(
        self,
        customer_name=None,
        assigned_to_email=None,
        status=None,
        customer_email=None,
        customer_phone=None,
        last_contact_date=None,
        product_interest=None,
        notes="",
        follow_up_by=None,
    ):
        """
        Adds a new customer record.
        """
        if not all([customer_name, assigned_to_email, status]):
            return "Please provide all required fields: customer_name, assigned_to_email, status."

        assigned_to_email = assigned_to_email.lower()
        if customer_email:
            customer_email = customer_email.lower()

        new_id = str(int(self.CRM_DATA["customer_id"].max()) + 1).zfill(8)
        new_customer = pd.DataFrame(
            {
                "customer_id": [new_id],
                "customer_name": [customer_name],
                "customer_email": [customer_email],
                "customer_phone": [customer_phone],
                "last_contact_date": [last_contact_date],
                "product_interest": [product_interest],
                "status": [status],
                "assigned_to_email": [assigned_to_email],
                "notes": [notes],
                "follow_up_by": [follow_up_by],
            }
        )
        self.CRM_DATA = pd.concat([self.CRM_DATA, new_customer], ignore_index=True)
        return new_id

    def delete_customer(self, customer_id=None):
        """
        Deletes a customer record by ID.
        """
        if not customer_id:
            return "Customer ID not provided."
        if customer_id not in self.CRM_DATA["customer_id"].values:
            return "Customer not found."
        self.CRM_DATA = self.CRM_DATA[self.CRM_DATA["customer_id"] != customer_id]
        return "Customer deleted successfully."
