import pandas as pd
from langchain.tools import tool

class ProjectManagement:
    def __init__(self):
        # 初始化时加载任务数据
        self.PROJECT_TASKS = pd.read_csv("data/processed/project_tasks.csv", dtype=str)

    def reset_state(self):
        """Resets the project tasks to the original state."""
        self.PROJECT_TASKS = pd.read_csv("data/processed/project_tasks.csv", dtype=str)

    def get_task_information_by_id(self, task_id=None, field=None):
        """
        Returns the task information for a given ID.

        Parameters
        ----------
        task_id : str, optional
            8-digit ID of the task.
        field : str, optional
            Field to return. Available fields are: "task_id", "task_name", "assigned_to_email", "list_name", "due_date", "board"

        Returns
        -------
        task : dict
            Task information for the given ID and field.

        Examples
        --------
        >>> project_management.get_task_information_by_id("00000000", "task_name")
        {{"task_name": "Refactor code"}}
        """
        if not task_id:
            return "Task ID not provided."
        if not field:
            return "Field not provided."
        task = self.PROJECT_TASKS[self.PROJECT_TASKS["task_id"] == task_id].to_dict(orient="records")
        if task:
            if field in task[0]:
                return {field: task[0][field]}
            else:
                return "Field not found."
        else:
            return "Task not found."

    def search_tasks(self, task_name=None, assigned_to_email=None, list_name=None, due_date=None, board=None):
        """
        Searches for tasks based on the given parameters.

        Parameters
        ----------
        task_name : str, optional
            Name of the task.
        assigned_to_email : str, optional
            Email address of the person assigned to the task.
        list_name : str, optional
            Name of the list the task belongs to.
        due_date : str, optional
            Due date of the task in "YYYY-MM-DD" format.
        board : str, optional
            Name of the board the task belongs to.

        Returns
        -------
        tasks : dict
            Task information for the given parameters.

        Examples
        --------
        >>> project_management.search_tasks("Refactor code", "tishtrya@example.com" "In progress", "2023-06-01", "Front end")
        {{"task_id": "00000000", "task_name": "Refactor code", "assigned_to_email": "tishtrya@example.com", "list_name": "In Progress", "due_date": "2023-06-01", "board": "Front End"}}
        """
        if not any([task_name, assigned_to_email, list_name, due_date, board]):
            return "No search parameters provided."
        tasks = self.PROJECT_TASKS.copy()
        if task_name:
            tasks = tasks[tasks["task_name"].str.contains(task_name, case=False)]
        if assigned_to_email:
            tasks = tasks[tasks["assigned_to_email"].str.contains(assigned_to_email, case=False)]
        if list_name:
            tasks = tasks[tasks["list_name"].str.contains(list_name, case=False)]
        if due_date:
            tasks = tasks[tasks["due_date"].str.contains(due_date, case=False)]
        if board:
            tasks = tasks[tasks["board"].str.contains(board, case=False)]
        return tasks.to_dict(orient="records")

    def create_task(self, task_name=None, assigned_to_email=None, list_name=None, due_date=None, board=None):
        """
        Creates a new task.

        Parameters
        ----------
        task_name : str
            Name of the task.
        assigned_to_email : str
            Email address of the person assigned to the task.
        list_name : str
            Name of the list the task belongs to.
        due_date : str
            Due date of the task in "YYYY-MM-DD" format.
        board : str
            Name of the board the task belongs to.

        Returns
        -------
        task_id : str
            8-digit ID of the new task.

        Examples
        --------
        >>> project_management.create_task("Integrate API service with frontend", "sam@example.com", "In progress", "2023-06-01", "Front end")
        "00000001"
        """
        if not all([task_name, assigned_to_email, list_name, due_date, board]):
            return "Missing task details."

        assigned_to_email = assigned_to_email.lower()
        if assigned_to_email not in self.PROJECT_TASKS["assigned_to_email"].str.lower().values:
            return "Assignee email not valid. Please choose from the list of team members."
        if list_name not in ["Backlog", "In Progress", "In Review", "Completed"]:
            return "List not valid. Please choose from: 'Backlog', 'In Progress', 'In Review', 'Completed'."
        if board not in ["Back end", "Front end", "Design"]:
            return "Board not valid. Please choose from: 'Back end', 'Front end', 'Design'."

        task_id = str(int(self.PROJECT_TASKS["task_id"].max()) + 1).zfill(8)
        new_task = pd.DataFrame(
            {
                "task_id": [task_id],
                "task_name": [task_name],
                "assigned_to_email": [assigned_to_email],
                "list_name": [list_name],
                "due_date": [due_date],
                "board": [board],
            }
        )
        self.PROJECT_TASKS = pd.concat([self.PROJECT_TASKS, new_task], ignore_index=True)
        return task_id

    def delete_task(self, task_id=None):
        """
        Deletes a task by ID.

        Parameters
        ----------
        task_id : str
            8-digit ID of the task.

        Returns
        -------
        message : str
            Message indicating the status of the deletion.

        Examples
        --------
        >>> project_management.delete_task("00000000")
        "Task deleted successfully."
        """
        if not task_id:
            return "Task ID not provided."

        if task_id in self.PROJECT_TASKS["task_id"].values:
            self.PROJECT_TASKS = self.PROJECT_TASKS[self.PROJECT_TASKS["task_id"] != task_id]
            return "Task deleted successfully."
        else:
            return "Task not found."

    def update_task(self, task_id=None, field=None, new_value=None):
        """
        Updates a task by ID.

        Parameters
        ----------
        task_id : str
            8-digit ID of the task.
        field : str
            Field to update. Available fields are: "task_name", "assigned_to_email", "list_name", "due_date", "board"
        new_value : str
            New value for the field.

        Returns
        -------
        message : str
            Message indicating the status of the update.

        Examples
        --------
        >>> project_management.update_task("00000000", "task_name", "New Task Name")
        "Task updated successfully."
        """
        if not task_id or not field or not new_value:
            return "Task ID, field, or new value not provided."

        if field == "assigned_to_email":
            new_value = new_value.lower()

        if field == "board" and new_value not in ["Back end", "Front end", "Design"]:
            return "Board not valid. Please choose from: 'Back end', 'Front end', 'Design'."
        if field == "list_name" and new_value not in ["Backlog", "In Progress", "In Review", "Completed"]:
            return "List not valid. Please choose from: 'Backlog', 'In Progress', 'In Review', 'Completed'."
        if field == "assigned_to_email" and new_value not in self.PROJECT_TASKS["assigned_to_email"].str.lower().values:
            return "Assignee email not valid. Please choose from the list of team members."

        if task_id in self.PROJECT_TASKS["task_id"].values:
            if field in self.PROJECT_TASKS.columns:
                self.PROJECT_TASKS.loc[self.PROJECT_TASKS["task_id"] == task_id, field] = new_value
                return "Task updated successfully."
            else:
                return "Field not valid."
        else:
            return "Task not found."
