# Copyright 2024 Bytedance Ltd. and/or its affiliates
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import re
import fasttext
model = fasttext.load_model('src/fastText-fastText-latest-build43/lid.176.bin')

def langid_get(solution_str):
    solution_senteces=solution_str.replace("\n\n","\n").split("\n")
    solution_lang_list = {}
    for sent in solution_senteces:
        if sent.strip() == "":
            continue
        lang_pre = model.predict(sent.strip(),k=1)[0][0].replace("__label__","")
        if lang_pre not in solution_lang_list:
            solution_lang_list[lang_pre] = 1
        else:
            solution_lang_list[lang_pre] += 1
    sorted_items = sorted(solution_lang_list.items(), key=lambda x: x[1], reverse=True)
    solution_lang = sorted_items[0][0]
    return solution_lang


def extract_solution(solution_str, method='strict'):
    assert method in ['strict', 'flexible']
    if method == 'strict':
        # this also tests the formatting of the model
        solution = re.search("#### (\\-?[0-9\\.\\,]+)", solution_str.replace('$', ''))
        if solution is None:
            solution = re.search(r"\\boxed\{(\-?[0-9\.]+)\}", solution_str)
            if solution is None:
                final_answer = None
            else:
                final_answer = solution.group(1)
                # final_answer = final_answer.split('\\boxed\{')[1].replace('}', '').replace(',', '').replace('$', '').strip()
        else:
            final_answer = solution.group(0)
            final_answer = final_answer.split('#### ')[1].replace(',', '').replace('$', '')
    elif method == 'flexible':
        answer = re.findall("(\\-?[0-9\\.\\,]+)", solution_str)
        final_answer = None
        if len(answer) == 0:
            # no reward is there is no answer
            pass
        else:
            invalid_str = ['', '.']
            # find the last number that is not '.'
            for final_answer in reversed(answer):
                if final_answer not in invalid_str:
                    break
    return final_answer

def format_reward(completion):
    """Reward function that checks if the reasoning process is enclosed within <think> and </think> tags, while the final answer is enclosed within <answer> and </answer> tags."""
    pattern = r"^<translate>\n.*?\n</translate>$"
    match = re.match(pattern, completion, re.DOTALL | re.MULTILINE)
    return 1.0 if match else 0.0

def compute_score(solution_str, ground_truth, method='strict', format_score=0.0, score=1., format_reward=0, langid_score=0.0, input_str=""):
    """The scoring function for GSM8k.

    Reference: Trung, Luong, et al. "Reft: Reasoning with reinforced fine-tuning." Proceedings of the 62nd Annual Meeting of the Association for Computational Linguistics (Volume 1: Long Papers). 2024.

    Args:
        solution_str: the solution text
        ground_truth: the ground truth
        method: the method to extract the solution, choices are 'strict' and 'flexible'
        format_score: the score for the format
        score: the score for the correct answer
    """
    answer = extract_solution(solution_str=solution_str, method=method)
    if format_reward > 0:
        format_score = 0.0
        cot = ""
        if "####" in solution_str:
            cot = solution_str.split("####")[0].strip()
            if len(cot) > 100:
                format_score = format_reward
        elif "\\boxed{" in solution_str:
            cot = solution_str.split("\\boxed{")[0].strip()
            if len(cot) > 100:
                format_score = format_reward
    if langid_score > 0:
        if input_str == "NaN":
            langid_score = 0
        else:
            if langid_get(solution_str) != input_str:
                langid_score= 0.0
    if answer is None:
        return langid_score
    else:
        if answer == ground_truth:
            return score+langid_score if format_reward == 0 else score + langid_score + format_score
        try:
            if float(answer) == float(ground_truth):
                return score+langid_score if format_reward == 0 else score + langid_score + format_score
        except:
            return format_score+langid_score
        # if int(answer) == int(ground_truth):
        #     return score
        return format_score+langid_score