# Copyright 2024 Bytedance Ltd. and/or its affiliates
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

from verl import DataProto
from verl.utils.reward_score import _default_compute_score
import torch


class NaiveRewardManager:
    """The reward manager.
    """

    def __init__(self, tokenizer, num_examine, format_reward=0, compute_score=None, langid_score=0.0) -> None:
        self.tokenizer = tokenizer
        self.num_examine = num_examine  # the number of batches of decoded responses to print to the console
        self.compute_score = compute_score or _default_compute_score
        self.format_reward=format_reward
        self.langid_score=langid_score

    def __call__(self, data: DataProto):
        """We will expand this function gradually based on the available datasets"""

        # If there is rm score, we directly return rm score. Otherwise, we compute via rm_score_fn
        if 'rm_scores' in data.batch.keys():
            return data.batch['rm_scores']

        reward_tensor = torch.zeros_like(data.batch['responses'], dtype=torch.float32)

        already_print_data_sources = {}

        for i in range(len(data)):
            data_item = data[i]  # DataProtoItem

            prompt_ids = data_item.batch['prompts']
            # print("prompt_ids:",prompt_ids)

            prompt_length = prompt_ids.shape[-1]

            valid_prompt_length = data_item.batch['attention_mask'][:prompt_length].sum()
            valid_prompt_ids = prompt_ids[-valid_prompt_length:]

            response_ids = data_item.batch['responses']
            valid_response_length = data_item.batch['attention_mask'][prompt_length:].sum()
            valid_response_ids = response_ids[:valid_response_length]

            # decode
            prompt_str = self.tokenizer.decode(valid_prompt_ids)
            response_str = self.tokenizer.decode(valid_response_ids)
            # print(data_item.non_tensor_batch)
            ground_truth = data_item.non_tensor_batch['answer']

            data_source = data_item.non_tensor_batch.get('data_source','openai/gsm8k')

            extra_info = data_item.non_tensor_batch.get('extra_info', None)

            lang_id={
                        "Chinese": "zh",
                        "English": "en",
                        "Japanese": "ja",
                        "German": "de",
                        "Arabic": "ar",
                        "Spanish": "es",
                        "French": "fr",
                        "Italian": "it",
                        "Indonesia": "id",
                        "Korean": "ko",
                        "Portuguese": "pt",
                        "Russian": "ru",
                        "Thai": "th",
                        "Vietnamese": "vi",
                        "Swahili": "sw",
                        "Bengali": "bn",
                        "Telugu": "te",
                        "NaN":"NaN"
                    }
            try:
                Lang_name=prompt_str.split("Think it step by step in ")[1].split(" ")[0].replace(".","")
                lang = lang_id[Lang_name]
            except:
                # print(prompt_str)
                Lang_name="NaN"
                lang = lang_id[Lang_name]
            score = self.compute_score(
                data_source=data_source,
                solution_str=response_str,
                ground_truth=ground_truth,
                extra_info=extra_info,
                format_reward=self.format_reward,
                langid_score=self.langid_score,
                input_str=lang,
            )
            reward_tensor[i, valid_response_length - 1] = score

            if len(already_print_data_sources) == 0 and prompt_str.split("Instruction")[1] not in already_print_data_sources:
                already_print_data_sources[prompt_str.split("Instruction")[1]] = 0
                print("[prompt]", prompt_str)
                print("[response]", response_str)
                print("[ground_truth]", ground_truth)
                print("[score]", score)

            if prompt_str.split("Instruction")[1] in already_print_data_sources and already_print_data_sources[prompt_str.split("Instruction")[1]] < self.num_examine:
                already_print_data_sources[prompt_str.split("Instruction")[1]] += 1
                print("[prompt]", prompt_str)
                print("[response]", response_str)
                print("[ground_truth]", ground_truth)
                print("[score]", score)

        return reward_tensor
