# =================================================================================================#
# Description: Produces the training times results
# Author: Ryan Thompson
# =================================================================================================#

using Distributed

Distributed.addprocs(2)

Distributed.@sync Distributed.@everywhere begin

cd("/Experiments")
include("pro_dag.jl")
include("metrics.jl")

import CSV, CUDA, DataFrames, Distributions, Graphs, LinearAlgebra, ProgressMeter, Random, 
    Statistics

#==================================================================================================#
# Function to generate data
#==================================================================================================#

function gendata(par)

    # Save scenario parameters
    n, p, s, id = par

    # Generate graph
    g = Graphs.erdos_renyi(p, s)
    g = Graphs.random_orientation_dag(g)

    # Create weighted adjacency matrix from graph
    w_values = rand(Distributions.Uniform(0.3, 0.7), p, p) .* rand([- 1, 1], p, p)
    w = Matrix(Graphs.adjacency_matrix(g)) .* w_values

    # Generate features
    ε = randn(n, p)
    x = ε * LinearAlgebra.inv(LinearAlgebra.I - w)

    # Return generated data
    x

end

#==================================================================================================#
# Function to evaluate a model
#==================================================================================================#

function evaluate!(result, estimator, time, par)

    # Save scenario parameters
    n, p, s, id = par

    # Update results
    push!(result, [estimator, time, n, p, s, id])

end

#==================================================================================================#
# Function to run a given simulation design
#==================================================================================================#

function runsim(par)

    gpu_id = (Distributed.myid() - 1) % 2
    CUDA.device!(gpu_id)
    CUDA.seed!(hash(par))
    Random.seed!(hash(par))

    # Set aside space for results
    result = DataFrames.DataFrame(
        estimator = [], time = [], n = [], p = [], s = [], id = []
    )

    # Generate data
    x = gendata(par)

    # Evaluate ProDAG
    time = @elapsed ProDAG.fit_linear(x, prior_α = par.p, verbose = false)
    evaluate!(result, "ProDAG", time, par)

    CUDA.reclaim()

    result

end

end

#==================================================================================================#
# Run simulations
#==================================================================================================#

# Specify simulation parameters
simulations = DataFrames.DataFrame(
        (n = n, p = p, s = s, id = id) for
        n = 100, # Number of samples
        (p, s) = [(20, 40), (40, 80), (60, 120), (80, 160), (100, 200)], # Number of variables 
        # and nonzeros
        id = 1:10 # Simulation run ID
        )

# Run simulations in parallel
result = ProgressMeter.@showprogress pmap(runsim, eachrow(simulations))
result = reduce(vcat, result)
CSV.write("Results/timings_training.csv", result)

Distributed.rmprocs(Distributed.workers())