import contextlib
import platform
import typing

from .osworld_types import Getter


class RedirectStdFdsToDockerLogs:
    def __enter__(self) -> "RedirectStdFdsToDockerLogs":
        self._f1 = open("/proc/1/fd/1", "w")
        self._redirect1 = contextlib.redirect_stdout(self._f1)
        self._redirect1.__enter__()
        self._f2 = open("/proc/1/fd/2", "w")
        self._redirect2 = contextlib.redirect_stderr(self._f2)
        self._redirect2.__enter__()
        return self

    def __exit__(
        self,
        exc_type: typing.Optional[type],
        exc_value: typing.Optional[Exception],
        traceback: typing.Optional[typing.Any],
    ) -> None:
        self._redirect1.__exit__(exc_type, exc_value, traceback)
        self._f1.close()
        self._redirect2.__exit__(exc_type, exc_value, traceback)
        self._f2.close()


def to_getter_list(
    getter: Getter | list[Getter] | dict[str, typing.Any] | None,
) -> list[Getter]:
    """Some OSWorld examples use empty dict (e.g. {}) for their getters rather than None"""
    if isinstance(getter, list):
        return getter
    elif getter is None or isinstance(getter, dict):
        return []
    else:
        return [getter]


def assert_linux() -> None:
    os_name = platform.system()
    if os_name != "Linux":
        # reference code for other platforms is in OSWorld repo
        raise NotImplementedError(f"Operating system {os_name} not supported.")
