import os
import json
import argparse
import numpy as np
import openai
from datasets import load_dataset
from alpaca_farm.auto_annotations import alpaca_leaderboard
from pred import load_model_and_tokenizer, seed_everything
import torch
import datasets
from metrics import (
    qa_f1_score,
    rouge_zh_score,
    qa_f1_zh_score,
    rouge_score,
    classification_score,
    retrieval_score,
    retrieval_zh_score,
    count_score,
    code_sim_score,
)
import pdb
openai.api_key_path = "data/openai_api_key.txt"



def str2bool(v):
    """Util function for user friendly boolean flag args"""
    if isinstance(v, bool):
        return v
    if v.lower() in ('yes', 'true', 't', 'y', '1'):
        return True
    elif v.lower() in ('no', 'false', 'f', 'n', '0'):
        return False
    else:
        raise argparse.ArgumentTypeError('Boolean value expected.')


dataset2metric = {
    "narrativeqa": qa_f1_score,
    "qasper": qa_f1_score,
    "multifieldqa_en": qa_f1_score,
    "multifieldqa_zh": qa_f1_zh_score,
    "hotpotqa": qa_f1_score,
    "2wikimqa": qa_f1_score,
    "musique": qa_f1_score,
    "dureader": rouge_zh_score,
    "gov_report": rouge_score,
    "qmsum": rouge_score,
    "multi_news": rouge_score,
    "vcsum": rouge_zh_score,
    "trec": classification_score,
    "triviaqa": qa_f1_score,
    "samsum": rouge_score,
    "lsht": classification_score,
    "passage_retrieval_en": retrieval_score,
    "passage_count": count_score,
    "passage_retrieval_zh": retrieval_zh_score,
    "lcc": code_sim_score,
    "repobench-p": code_sim_score,
    "konwledge_memorization": qa_f1_score,
    "konwledge_understanding": qa_f1_score,
    "longform_qa": rouge_score,
    "finance_qa": rouge_score,
    "short_finance_qa": rouge_score,
}

def parse_args(args=None):
    parser = argparse.ArgumentParser(description="Evaluate texts generated by every method")

    parser.add_argument(
        "--input_dir",
        type=str,
        default="pred/llama2-7b-chat-4k_cc_g0.5_d5.0")
    
    parser.add_argument(
    "--calc_ce",
    type=str2bool,
    default=0)

    parser.add_argument(
    "--initial_seed_llm",
    type=int,
    default=42)
    
    args = parser.parse_args()
    

    return args

def calculate_log_ppl(tokens):
    log_probs = [token['logprob'] for token in tokens]
    return -sum(log_probs)/len(log_probs)

def scorer(dataset, predictions, answers, all_classes):
    total_score = 0.
    for (prediction, ground_truths) in zip(predictions, answers):
        score = 0.
        if dataset in ["trec", "triviaqa", "samsum", "lsht"]:
            prediction = prediction.lstrip('\n').split('\n')[0]
        for ground_truth in ground_truths:
            score = max(score, dataset2metric[dataset](prediction, ground_truth, all_classes=all_classes))
        total_score += score
    return round(100 * total_score / len(predictions), 2)

def alpacafarm_score(prompts, predictions, model_name):
    # outputs should be a list of json as such:
    # [{'instruction': 'What are the names of some famous actors that started their careers on Broadway?', 'input': '', 'output': 'Some famous actors that started their careers on Broadway are Hugh Jackman, Meryl Streep, Denzel Washington, Audra McDonald, and Lin-Manuel Miranda.', 'generator': 'gpt-3.5-turbo-0301', 'dataset': 'helpful_base', 'datasplit': 'eval'},
    # ...]
    my_outputs = []
    alapaca_eval_data = load_dataset("tatsu-lab/alpaca_farm", "alpaca_farm_evaluation")["eval"]
    for i, json_obj in enumerate(alapaca_eval_data):
        prompt = json_obj["instruction"]
        _input = json_obj["input"]
        prediction = predictions[i]
        my_outputs.append({"instruction": prompt, "input": _input, "generator": model_name, "output": prediction})
    #print("my_outputs[0] is:", my_outputs[0])
    df_results = alpaca_leaderboard(
        path_or_all_outputs=my_outputs,
        name=model_name,
        is_add_reference_methods=False,
        annotators_config = "greedy_gpt4/configs.yaml"
    )
    score = df_results.to_string(float_format="%.2f")
    return score


def calc_ce(model, tokenizer, prompts, predictions, device):
    # pdb.set_trace()
    """
    calculation of the CE over a dataset of prompts and predictions.
    steps:
    1. tokenize the prompts and predictions
    2. concatenate the prompts and predictions
    3. calculate the base model -logits
    4. calculate the CE = mean(-logits)
    """
    ce_list = []
    for prompt, prediction in zip(prompts, predictions):
        text = prompt + prediction
        # text = prediction
        in_ = tokenizer(text, return_tensors="pt", truncation=False).to(device)
        in_['labels'] = in_['input_ids'].clone()
        in_['labels'][:, :len(prompt)] = -100
        # output = model(**in_)
        # ce_list.append(output.loss.item())
        with torch.no_grad():
            output = model(input_ids=in_['input_ids'], labels=in_['labels'])
            ce_list.append(output.loss.item())
    return np.mean(ce_list)
    # return np.mean(ce_list), np.std(ce_list)


if __name__ == '__main__':
    # pdb.set_trace()
    args = parse_args()
    scores = dict()
    log_ppl_scores = dict()  # Dictionary to store log-ppl scores
    # get all files from input_dir
    files = os.listdir(args.input_dir)
    #print(f'files dir {files}')
    model_name = args.input_dir.split("/")[-1]
    if args.calc_ce:
        # Load the base model for base calculation
        if 'llama2-7b-chat-4k' in args.input_dir:
            seed_everything(args.initial_seed_llm)
            #print('loading model and tokenizer..')
            device = torch.device("cuda" if torch.cuda.is_available() else "cpu")
            model, tokenizer = load_model_and_tokenizer("meta-llama/Llama-2-7b-chat-hf", 'llama2-7b-chat-4k', device)
            #print('finished loading model and tokenzier')
            ce_dict = dict()
    # get all json files
    json_files = [f for f in files if f.endswith(".jsonl")]
    save_dir =  os.path.join(args.input_dir, "eval")
    os.makedirs(save_dir, exist_ok=True)
    #print("Evaluating on:", files)
    for json_file in json_files:
        if not json_file.endswith("jsonl"):
            continue
        print(f"{json_file} has began.........")
        # read jsons
        dataset = json_file.split(".")[0]
        predictions, answers, lengths, all_classes, log_probabilities_list = [], [], [], [], []
        with open(os.path.join(args.input_dir, json_file), "r") as f:
            # lines
            lines = f.readlines()
            # texts
            prompts = [json.loads(line)["prompt"] for line in lines]
            predictions = [json.loads(line)["pred"] for line in lines]
            if 'poem' not in args.input_dir:
                answers = [json.loads(line)["answers"] for line in lines]
                all_classes = json.loads(lines[0])["all_classes"]
            log_probabilities_list = [calculate_log_ppl(json.loads(line)["completions_tokens"]) for line in lines]
            #print(f"predictions[0] is: {predictions[0]}")
            # pdb.set_trace()
            if 'poem' in args.input_dir and args.calc_ce:
                score = -1
                ce = calc_ce(model, tokenizer, prompts, predictions,device)
                ce_dict[dataset] = ce
            elif dataset == "alpacafarm":
                # score = alpacafarm_score(prompts, predictions, model_name)
                score = -1
            else:
                score = scorer(dataset, predictions, answers, all_classes)
            if args.calc_ce and dataset in ('finance_qa','short_finance_qa'):
                # calculate cross entropy between watermarked tokens and predictions
                ce = calc_ce(model, tokenizer, prompts, predictions,device)
                ce_dict[dataset] = ce
            scores[dataset] = score
            # Calculate log-ppl for the dataset
            log_ppl_scores[dataset] = sum(log_probabilities_list) / len(log_probabilities_list)
    # save
    out_path = os.path.join(save_dir, "result.json")
    with open(out_path, "w") as f:
        json.dump(scores, f, ensure_ascii=False, indent=4)
    # save log-ppl
    log_ppl_path = os.path.join(save_dir, "log_ppl.json")
    with open(log_ppl_path, "w") as f:
        json.dump(log_ppl_scores, f, ensure_ascii=False, indent=4)
    if args.calc_ce:
        ce_path = os.path.join(save_dir, "ce.json")
        with open(ce_path, "w") as f:
            json.dump(ce_dict, f, ensure_ascii=False, indent=4)
    
